; This file is part of the Resolution ][ library
;
; Copyright (c) 2000 Arthur Jerijian
; 
; Permission is hereby granted, free of charge, to any person obtaining
; a copy of this software and associated documentation files
; (the "Software"), to deal in the Software without restriction, including
; without limitation the rights to use, copy, modify, merge, publish,
; distribute, sublicense, and/or sell copies of the Software, and to permit
; persons to whom the Software is furnished to do so, subject to the
; following conditions:
; 
; The above copyright notice and this permission notice shall be included
; in all copies or substantial portions of the Software.
; 
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
; OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
; THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
; DEALINGS IN THE SOFTWARE.

; ======================================================================
;
; Divide a single-byte number by 7. This function performs a fast
; one-byte division specifically optimised for computing the memory
; offset of a pixel X coordinate.
;
; Input parameters
;	divisor - one byte -- single-byte divisor
; Output parameters
;	quotient - one byte -- quotient of X divided by 7
;	remainder - one byte -- remainder of X divided by 7

divide_byte_by_7:
	lda	divisor		; Set the quotient to the divisor
	sta	quotient
	lda	#$00		; Clear the remainder
	sta	remainder

	ldx	bits_per_byte	; Start counting down bits

db7_loop:
	lda	quotient	; Determine the most significant bit
	asl			; of the quotient
	rol	remainder	; Double the remainder while adding
				; the MSB of the quotient
	asl	quotient	; Double the quotient
	sec			; Compute (remainder - 7)
	lda	remainder
	sbc	#$07
	bcc	db7_next	; If (remainder - 7) < 0, then skip

	sta	remainder	; Store remainder = (remainder - 7)
	inc	quotient	; Increment the quotient
				; (A single inc instruction is enough because
				; the LSB of the quotient is always zero)

db7_next:
	dex			; Repeat until all bits used
	bne	db7_loop

	rts

; ======================================================================
;
; Increment the memory offset of a monochrome pixel X coordinate
; NOTE -- Applications should call resii_inc_x_offset instead.
;
; Input parameters (all are modified)
;	x_coord - two bytes -- pixel X coordinate
;	x_offset - one byte -- memory offset from the first column
;	x_page - one byte -- page offset (0 = MAIN, 1 = AUX)
;	x_bit - one byte -- pixel bit offset within a frame buffer byte

inc_mono_x_offset:
	inc	x_coord		; First, increment the X coordinate
	bne	imx_bit_offset
	inc	x_coord + 1

imx_bit_offset:
	inc	x_bit		; Increment the bit offset
	lda	x_bit
	cmp	#$07		; If the bit offset overflows, continue
	beq	imx_check_page	; ahead
	rts			; Otherwise, we're done!

imx_check_page:
	lda	#$00		; Reset the bit offset to zero
	sta	x_bit
	lda	x_page		; Check the page
	beq	imx_set_offset	; If it's main, continue ahead
	lda	#$00		; If it's aux, switch to main and
	sta	x_page		; we're done
	rts

imx_set_offset:
	lda	#$01		; Set the page to aux and increment
	sta	x_page		; the memory offset
	inc	x_offset
	rts

; ======================================================================
;
; Decrement the memory offset of a monochrome pixel X coordinate
; NOTE -- Applications should call resii_dec_x_offset instead.
;
; Input parameters (all are modified)
;	x_coord - two bytes -- pixel X coordinate
;	x_offset - one byte -- memory offset from the first column
;	x_page - one byte -- page offset (0 = MAIN, 1 = AUX)
;	x_bit - one byte -- pixel bit offset within a frame buffer byte

dec_mono_x_offset:
	dec	x_coord		; First, decrement the X coordinate
	lda	x_coord
	cmp	#$FF
	bne	dmx_bit_offset
	dec	x_coord + 1

dmx_bit_offset:
	dec	x_bit		; Decrement the bit offset
	lda	x_bit
	cmp	#$FF		; If the bit offset overflows, continue
	beq	dmx_check_page	; ahead
	rts			; Otherwise, we're done!

dmx_check_page:
	lda	#$06		; Reset the bit offset to 6
	sta	x_bit
	lda	x_page		; Check the page
	bne	dmx_set_offset	; If it's aux, continue ahead
	lda	#$01		; If it's main, switch to aux and
	sta	x_page		; we're done
	rts

dmx_set_offset:
	lda	#$00		; Set the page to main and decrement
	sta	x_page		; the memory offset
	dec	x_offset
	rts

; ======================================================================
;
; Increment the memory offset of a color pixel X coordinate
; NOTE -- Applications should call resii_inc_x_offset instead.
;
; Input parameters (all are modified)
;	x_coord - two bytes -- pixel X coordinate
;       x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- page offset (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- pixel bit offset within a frame buffer byte

inc_color_x_offset:
	inc	x_coord		; First, increment the X coordinate
	bne	icx_bit_offset
	inc	x_coord + 1

icx_bit_offset:
	clc			; Add 4 to the bit offset
	lda	x_bit
	adc	#$04
	sta	x_bit
	cmp	#$07		; If the bit offset overflows, continue
	bpl	icx_check_page	; ahead
	rts			; Otherwise, we're done!

icx_check_page:
	sec			; Subtract 7 from the bit offset
	sbc	#$07
	sta	x_bit
	lda	x_page		; Check the page
	beq	icx_set_offset	; If it's main, continue ahead
	lda	#$00		; If it's aux, switch to main and
	sta	x_page		; we're done
	rts

icx_set_offset:
	lda	#$01		; Set the page to aux and increment
	sta	x_page		; the memory offset
	inc	x_offset
	rts

; ======================================================================
;
; Decrement the memory offset of a color pixel X coordinate
; NOTE -- Applications should call resii_dec_x_offset instead.
;
; Input parameters (all are modified)
;	x_coord - two bytes -- pixel X coordinate
;       x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- page offset (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- pixel bit offset within a frame buffer byte

dec_color_x_offset:
	dec	x_coord		; First, decrement the X coordinate
	lda	x_coord
	cmp	#$FF
	bne	dcx_bit_offset
	dec	x_coord + 1

dcx_bit_offset:
	sec			; Subtract 4 from the bit offset
	lda	x_bit
	sbc	#$04
	sta	x_bit		; If the bit offset overflows, continue
	bcc	dcx_check_page	; ahead
	rts			; Otherwise, we're done!

dcx_check_page:
	clc			; Add 7 to the bit offset
	adc	#$07
	sta	x_bit
	lda	x_page		; Check the page
	bne	dcx_set_offset	; If it's aux, continue ahead
	lda	#$01		; If it's main, switch to aux and
	sta	x_page		; we're done
	rts

dcx_set_offset:
	lda	#$00		; Set the page to main and decrement
	sta	x_page		; the memory offset
	dec	x_offset
	rts

; ======================================================================
;
; Plot a monochrome pixel with precomputed memory offset values
; NOTE -- Applications should call resii_draw_pixel_offset instead.
;
; Input parameters
;	pen_color - one byte -- value to write (0 = OFF, 1 = ON)
;	x_offset - one byte -- memory offset from the first column
;	x_page - one byte -- X coordinate page (0 = MAIN, 1 = AUX)
;	x_bit - one byte -- X coordinate bit offset
;	y_offset - two bytes -- Y coordinate memory offset
; Output parameters
;	None

draw_mono_offset:
	lda	image_buffer_lo	; Start with buffer address
	sta	buffer_ptr
	lda	image_buffer_hi
	sta	buffer_ptr + 1

	clc			; Add the Y memory offset
	lda	y_offset
	adc	buffer_ptr
	sta	buffer_ptr
	lda	y_offset + 1
	adc	buffer_ptr + 1
	sta	buffer_ptr + 1

	ldy	x_page		; Set the coordinate page
	sta	ss_main_page, Y	; Note that ss_aux_page is ss_main_page
				; plus one.

	lda	pen_color	; Determine the pixel value
	beq	dmo_pixel_off

	lda	#$01		; Start with a single-bit "on" pixel
	ldx	x_bit
	beq	dmo_mask

dmo_shift:
	asl			; Shift the bit as many times as needed
	dex			; to mask with the existing buffer byte
	bne	dmo_shift

dmo_mask:
	ldy	x_offset	; Use the X offset as a Y-register index
	ora	(buffer_ptr), Y	; Mask the byte with the pixel
	sta	(buffer_ptr), Y
	
	rts			; Pixel has been drawn -- quit

dmo_pixel_off:
	lda	#$FE		; Start with a single-bit "off" pixel
	ldx	x_bit
	beq	dmo_mask_off
	sec

dmo_shift_off:
	rol			; Shift the bit as many times as needed
	dex 			; to mask with the existing buffer byte 
	bne	dmo_shift_off

dmo_mask_off:
	ldy	x_offset	; Use the X offset as a Y-register index
	and	(buffer_ptr), Y	; Mask the byte with the pixel
	sta	(buffer_ptr), Y
	
	rts			; Pixel has been turned off -- quit

; ======================================================================
;
; Plot a color pixel with precomputed memory offset values
; NOTE -- Applications should call resii_draw_pixel_offset instead.
;
; Input parameters
;       pen_color - one byte -- color to write
;	x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- X coordinate page (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- X coordinate bit offset
;       y_offset - two bytes -- Y coordinate memory offset
; Output parameters
;       None

draw_color_offset:
	lda	image_buffer_lo	; Start with buffer address
	sta	buffer_ptr
	lda	image_buffer_hi
	sta	buffer_ptr + 1

	clc			; Add the Y memory offset
	lda	y_offset
	adc	buffer_ptr
	sta	buffer_ptr
	lda	y_offset + 1
	adc	buffer_ptr + 1
	sta	buffer_ptr + 1

	lda	pen_color	; Translate the pen color to the
	lsr			; buffer color
	bcc	dco_set_masks
	ora	#$08

dco_set_masks:
	sta	color_draw_mask	; Set the drawing mask to the buffer color
	lda	#$00
	sta	color_draw_mask + 1
	lda	#$F0		; Set the erase mask to #$FFF0
	sta	color_erase_mask
	lda	#$FF
	sta	color_erase_mask + 1

	ldx	x_bit
	beq	dco_mask

dco_shift:
	asl	color_draw_mask		; Shift the masks with the current
	rol	color_draw_mask + 1	; two bytes of the frame buffer
	sec
	rol	color_erase_mask
	rol	color_erase_mask + 1
	dex
	bne	dco_shift

	lda	color_erase_mask	; If the MSB of the first byte is
	and	#$80		; being used, shift the upper byte one
	bne	dco_mask	; more time
	asl	color_erase_mask + 1
	lda	color_draw_mask	; Move the MSB of the draw mask up one bit
	asl
	rol	color_draw_mask + 1

dco_mask:
	lda	x_page		; Determine the page of the first byte
	bne	dco_mask_aux

	sta	ss_main_page	; Start with the main page
	ldy	x_offset	; Use the X offset as a Y-register index
	lda	(buffer_ptr), Y	; Mask the next two bytes with the pixel
	and	color_erase_mask ; masks
	ora	color_draw_mask
	sta	(buffer_ptr), Y
	sta	ss_aux_page
	iny			; Write the second byte
	lda	(buffer_ptr), Y
	and	color_erase_mask + 1
	ora	color_draw_mask + 1
	sta	(buffer_ptr), Y
	rts			; Pixel has been drawn -- quit

dco_mask_aux:
	sta	ss_aux_page	; Start with the aux page
	ldy	x_offset	; Use the X offset as a Y-register index
	lda	(buffer_ptr), Y	; Mask the next two bytes with the pixel
	and	color_erase_mask ; masks
	ora	color_draw_mask
	sta	(buffer_ptr), Y
	sta	ss_main_page
	lda	(buffer_ptr), Y	; Write the second byte
	and	color_erase_mask + 1
	ora	color_draw_mask + 1
	sta	(buffer_ptr), Y
	rts			; Pixel has been drawn -- quit

; ======================================================================
;
; Read a monochrome pixel with precomputed memory offset values
; NOTE -- Applications should call resii_read_pixel_offset instead.
;
; Input parameters
;	x_offset - one byte -- memory offset from the first column
;	x_page - one byte -- X coordinate page (0 = MAIN, 1 = AUX)
;	x_bit - one byte -- X coordinate bit offset
;	y_offset - two bytes -- Y coordinate memory offset
; Output parameters
;	read_color - one byte -- value of pixel (0 = OFF, 1 = ON)

read_mono_offset:
	lda	image_buffer_lo	; Start with buffer address
	sta	buffer_ptr
	lda	image_buffer_hi
	sta	buffer_ptr + 1

	clc			; Add the Y memory offset
	lda	y_offset
	adc	buffer_ptr
	sta	buffer_ptr
	lda	y_offset + 1
	adc	buffer_ptr + 1
	sta	buffer_ptr + 1

	ldy	x_page		; Set the coordinate page
	sta	ss_main_page, Y	; Note that ss_aux_page is ss_main_page
				; plus one.

	lda	#$01		; Start with a single-bit
	ldx	x_bit
	beq	rmo_mask

rmo_shift:
	asl			; Shift the bit as many times as needed
	dex			; to mask with the existing buffer byte
	bne	rmo_shift

rmo_mask:
	ldy	x_offset	; Use the X offset as a Y-register index
	and	(buffer_ptr), Y	; Mask the byte with the pixel
	beq	rmo_continue	; Skip ahead if pixel value is zero

	lda	#$01		; Otherwise, pixel value is one

rmo_continue:
	sta	read_color	; Store the pixel value
	rts

; ======================================================================
;
; Read a color pixel with precomputed memory offset values
; NOTE -- Applications should call resii_read_pixel_offset instead.
;
; Input parameters
;	x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- X coordinate page (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- X coordinate bit offset
;       y_offset - two bytes -- Y coordinate memory offset
; Output parameters
;	read_color - one byte -- color of pixel

read_color_offset:
	lda	image_buffer_lo	; Start with buffer address
	sta	buffer_ptr
	lda	image_buffer_hi
	sta	buffer_ptr + 1

	clc			; Add the Y memory offset
	lda	y_offset
	adc	buffer_ptr
	sta	buffer_ptr
	lda	y_offset + 1
	adc	buffer_ptr + 1
	sta	buffer_ptr + 1

	lda	#$0F		; Set the erase mask to #$000F
	sta	color_erase_mask
	lda	#$00
	sta	color_erase_mask + 1

	ldx	x_bit
	beq	rco_mask

rco_shift:
	asl	color_erase_mask	; Shift the mask with the current
	rol	color_erase_mask + 1	; two bytes of the frame buffer
	dex
	bne	rco_shift

	lda	color_erase_mask	; If the MSB of the first byte is
	and	#$80			; being used, shift the upper byte
	beq	rco_mask		; one more time
	sec
	rol	color_erase_mask + 1

rco_mask:
	lda	x_page		; Determine the page of the first byte
	bne	rco_mask_aux

	sta	ss_main_page	; Start with the main page
	ldy	x_offset	; Use the X offset as a Y-register index
	lda	(buffer_ptr), Y	; Mask the next two bytes with the pixel
	and	color_erase_mask ; mask
	sta	color_draw_mask
	sta	ss_aux_page
	iny			; Read the second byte
	lda	(buffer_ptr), Y
	and	color_erase_mask + 1
	sta	color_draw_mask + 1
	clc
	bcc	rco_restore_mask

rco_mask_aux:
	sta	ss_aux_page	; Start with the aux page
	ldy	x_offset	; Use the X offset as a Y-register index
	lda	(buffer_ptr), Y	; Mask the next two bytes with the pixel
	and	color_erase_mask ; mask
	sta	color_draw_mask
	sta	ss_main_page
	lda	(buffer_ptr), Y	; Read the second byte
	and	color_erase_mask + 1
	sta	color_draw_mask + 1

rco_restore_mask:
	lda	color_erase_mask ; Restore the erase mask
	and	#$80
	beq	rco_get_color
	lsr	color_erase_mask + 1

rco_get_color:
	lsr	color_draw_mask + 1 ; Shift the color value mask
	bcc	rco_shift_color	; and preserve the LSB of the upper byte
	lda	color_draw_mask
	ora	#$80
	sta	color_draw_mask

rco_shift_color:
	lda	color_erase_mask
	and	#$01
	bne	rco_translate_color

	lsr	color_draw_mask + 1 ; Shift the masks until we get a
	ror	color_draw_mask	; color value that can be stored
	lsr	color_erase_mask + 1
	ror	color_erase_mask
	lda	color_erase_mask
	clc
	bcc	rco_shift_color

rco_translate_color:
	lda	color_draw_mask	; Translate the buffer color to a
	asl			; pen color, and return
	tax
	and	#$10
	beq	rco_low_color
	txa
	and	#$0F
	ora	#$01
	sta	read_color

rco_low_color:
	txa
	sta	read_color
	rts

; ======================================================================
;
; Compute the memory offset of a pixel X coordinate
;
; Input parameters
;	use_color - one byte -- whether to use color (0 = no, 1 = yes)
;	x_coord - two bytes -- pixel X coordinate
; Output parameters
;	x_offset - one byte -- memory offset from the first column
;	x_page - one byte -- page offset (0 = MAIN, 1 = AUX)
;	x_bit - one byte -- pixel bit offset within a frame buffer byte

resii_x_offset:
	lda	x_coord		; Copy the X coordinate to a working area
	sta	x_coord_work
	lda	x_coord + 1
	sta	x_coord_work + 1

	lda	use_color	; If it's monochrome, skip ahead
	beq	xo_skip_color
	asl	x_coord_work	; Multiply the X coordinate by 4
	rol	x_coord_work + 1 ; if the pixel will be in color
	asl	x_coord_work
	rol	x_coord_work + 1

xo_skip_color:
	ldx	x_coord_work + 1 ; Determine the range of the X coordinate
	beq	xo_first_band	; so that it can be quickly divided by 7
	dex
	beq	xo_second_third_band

	lda	x_coord_work	; X >= 512. Add 1 to the lower byte,
	sta	divisor		; divide the lower byte by 7, and add
	inc	divisor		; 73 to the quotient.
	jsr	divide_byte_by_7
	clc
	lda	quotient
	adc	#$49
	sta	quotient
	clc
	bcc	xo_continue

xo_second_third_band:
	sec			; 256 <= X <= 511. Now, if X < 260,
	lda	x_coord_work	; we have to treat this case specially.
	sbc	#$03
	bcs	xo_third_band

	lda	#$24		; 256 <= X <= 259. The quotient is 36
	sta	quotient	; and the remainder is the lower byte
	clc			; plus 4.
	lda	x_coord_work
	adc	#$04
	sta	remainder
	clc
	bcc	xo_continue

xo_third_band:
	sta	divisor		; 260 <= X <= 511. Accumulator contains
	jsr	divide_byte_by_7 ; the lower byte minus 3. Divide it
	clc			; by 7 and add 37 to the quotient.
	lda	quotient
	adc	#$25
	sta	quotient
	clc
	bcc	xo_continue
	
xo_first_band:
	lda	x_coord_work	; X <= 255. Just divide the lower byte
	sta	divisor		; by 7.
	jsr	divide_byte_by_7

xo_continue:
	lda	remainder	; The remainder is the bit offset
	sta	x_bit

	lda	quotient	; The toggled LSB of the quotient
	and	#$01		; is the page offset
	eor	#$01
	sta	x_page

	lda	quotient	; Half the quotient is the memory offset
	lsr
	sta	x_offset

	rts

; ======================================================================
;
; Increment the memory offset of a pixel X coordinate
;
; Input parameters (all are modified except for use_color)
;	use_color - one byte -- whether to use color (0 = no, 1 = yes)
;	x_coord - two bytes -- pixel X coordinate
;       x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- page offset (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- pixel bit offset within a frame buffer byte

resii_inc_x_offset:
	lda	use_color
	beq	ixo_mono
	jmp	inc_color_x_offset
ixo_mono:
	jmp	inc_mono_x_offset

; ======================================================================
;
; Decrement the memory offset of a pixel X coordinate
;
; Input parameters (all are modified except for use_color)
;	use_color - one byte -- whether to use color (0 = no, 1 = yes)
;	x_coord - two bytes -- pixel X coordinate
;       x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- page offset (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- pixel bit offset within a frame buffer byte

resii_dec_x_offset:
	lda	use_color
	beq	ixo_mono
	jmp	dec_color_x_offset
dxo_mono:
	jmp	dec_mono_x_offset

; ======================================================================
;
; Compute the memory offset of a pixel Y coordinate
;
; Input parameters
;	y_coord - one byte -- pixel Y coordinate
; Output parameters
;	y_offset - two bytes -- memory offset from the first row

resii_y_offset:
	lda	y_coord		; Divide the Y coordinate by 8,
	lsr			; double it, and transfer it to the Y
	lsr			; register. This is the memory index
	lsr			; of the Y offset octet lookup table.
	asl
	tay

	lda	y_offset_table, Y	; Look up the Y offset octet value.
	sta	y_offset
	lda	y_offset_table + 1, Y
	sta	y_offset + 1

	lda	y_coord		; The last 3 bits of the Y coordinate,
	and	#$07		; multiplied by 1024, are then OR'ed into
	asl			; the offset.
	asl
	ora	y_offset + 1
	sta	y_offset + 1

	rts

; ======================================================================
;
; Increment the memory offset of a pixel Y coordinate
;
; Input parameters (all are modified)
;	y_coord - one byte -- pixel Y coordinate
;	y_offset - two bytes -- memory offset from the first row

resii_inc_y_offset:
	inc	y_coord
	jmp	resii_y_offset

; ======================================================================
;
; Decrement the memory offset of a pixel X coordinate
;
; Input parameters (all are modified)
;	y_coord - one byte -- pixel Y coordinate
;	y_offset - two bytes -- memory offset from the first row

resii_dec_y_offset:
	dec	y_coord
	jmp	resii_y_offset

; ======================================================================
;
; Plot a pixel with precomputed memory offset values
;
; Input parameters
;	use_color - one byte -- whether to use color (0 = no, 1 = yes)
;       pen_color - one byte -- color or value to write
;	x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- X coordinate page (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- X coordinate bit offset
;       y_offset - two bytes -- Y coordinate memory offset
; Output parameters
;       None

resii_draw_pixel_offset:
	lda	use_color
	beq	dpo_mono
	jmp	draw_color_offset
dpo_mono:
	jmp	draw_mono_offset

; ======================================================================
;
; Plot a pixel
;
; Input parameters
;	use_color - one byte -- whether to use color (0 = no, 1 = yes)
;       pen_color - one byte -- color or value to write
;	x_coord - two bytes -- pixel X coordinate
;	y_coord - one byte -- pixel Y coordinate
; Output parameters
;	None

resii_draw_pixel:
	jsr	resii_x_offset
	jsr	resii_y_offset
	jsr	resii_draw_pixel_offset
	rts

; ======================================================================
;
; Read a pixel with precomputed memory offset values
;
; Input parameters
;	use_color - one byte -- whether to use color (0 = no, 1 = yes)
;       pen_color - one byte -- color or value to write
;	x_offset - one byte -- memory offset from the first column
;       x_page - one byte -- X coordinate page (0 = MAIN, 1 = AUX)
;       x_bit - one byte -- X coordinate bit offset
;       y_offset - two bytes -- Y coordinate memory offset
; Output parameters
;	read_color - one byte -- value of pixel (0 = OFF, 1 = ON)

resii_read_pixel_offset:
	lda	use_color
	beq	rpo_mono
	jmp	read_color_offset
rpo_mono:
	jmp	read_mono_offset

; ======================================================================
;
; Read a pixel
;
; Input parameters
;	x_coord - two bytes -- pixel X coordinate
;	y_coord - one byte -- pixel Y coordinate
; Output parameters
;	read_color - one byte -- value of pixel (0 = OFF, 1 = ON)

resii_read_pixel:
	jsr	resii_x_offset
	jsr	resii_y_offset
	jsr	resii_read_pixel_offset
	rts

