/* 
 * Apple // emulator for Linux: Video-driver interface
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999, 2000, 2001 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#ifndef A2_VIDEO_H
#define A2_VIDEO_H

#ifndef __ASSEMBLER__

/* Prepare the video system, converting console to graphics mode, or 
 * opening X window, or whatever.  This is called only once when the 
 * emulator is run
 */
void video_init(void);

/* Undo anything done by video_init. This is called before exiting the 
 * emulator.
 */
void video_shutdown(void);

/* Setup the display. This may be called multiple times in a run, and is 
 * used when graphics parameters (II+ vs //e, hires color representation) may
 * have changed.
 *
 * In future, all relevant information will be communicated through 
 * FLAGS. For now, however, it is ignored and global variables are used 
 * instead.
 *
 * This function is responsible for inserting any needed video-specific hooks 
 * into the 6502 memory indirection table.  It should *not* hook the 
 * soft-switches.  
 *
 */
void video_set(int flags);

/* Set the font used by the display.  QTY characters are loaded starting 
 * with FIRST, from DATA.  DATA contains 8 bytes for each character, each 
 * byte representing a row (top-to-bottom).  The row byte contains 7 
 * pixels in little-endian format.
 *
 * MODE selects the colors to use
 *
 *   0 - Normal text
 *   1 - MouseText (usually treat as Normal)
 *   2 - Inverse
 *   3 - Flash
 *
 * The extra MouseText mode is in case we want to emulate certain RGB 
 * adaptors which color normal text and MouseText differently.  I had one 
 * once for a //c.
 */
void video_loadfont(int first, 
                    int qty, 
                    const unsigned char *data,
                    int mode);

/* Redraw the display. This is called after exiting an interface display, 
 * when changes have been made to the Apple's emulated framebuffer that
 * bypass the driver's hooks, or when the video mode has changed.
 */
void video_redraw(void);

/* Change the displayed video page to PAGE
 *   0 - Page 1: $400-$7ff/$2000-$3fff
 *   1 - Page 2: $800-$bff/$4000-$5fff
 */
void video_setpage(int page);

/* Like loadfont, but writes to a seperate 256 character table used by
 * video_plotchar and not the apple text-mode display.
 */
void video_loadfont_int(int first, int qty, const unsigned char *data);

/* Plot a character to the text mode screen, *not* writing to apple 
 * memory. This is used by the interface screens.
 *
 * ROW, COL, and CODE are self-expanatory. COLOR gives the color scheme 
 * to use:
 *
 *  0 - Green text on Black background
 *  1 - Green text on Blue background
 *  2 - Red text on Black background
 */
void video_plotchar(int row, int col, int color, unsigned char code);

/* Called at about 30Hz (this may change in future), and when waiting in 
 * the interface screens. 
 * 
 * Should flush any video data to the real screen (if any kind of caching 
 * is in use), check for keyboard input (presently reported via 
 * c_read_raw_key), and handle flashing text characters.
 */
void video_sync(int block);

#define MAX_KEYCODES 256

/* Called to interrogate driver about which code it will sumbit to 
 * c_read_raw_char for the given key.  The names uses are as in 
 * X Windows.
 *
 * The keycodes must be in the range 0 - (MAX_KEYCODES - 1) 
 */
int video_keycode(const char *name);

#endif /* !__ASSEMBLER__ */
#endif /* !A2_VIDEO_H */
