/*
 * DESMO target kernel for Blackfin
 *
 * Kernel commands:
 *	All kernel commands begin with '$' ... Kernel sends '.' in
 *	response and waits for remainder of command.
 *	Responses to commands are shown in [square braces]
 *	'a'=address byte (binary), 'd'=data byte (binary),
 *	all other characters and sent/received as the ASCII character.
 *
 * $[.]baaaa[d]			=  8-bit memory read
 * $[.]haaaa[dd]		= 16-bit memory read
 * $[.]waaaa[dddd]		= 32=bit memory read
 * $[.]Baaaad			=  8-bit memory write
 * $[,]Haaaadd			= 16-bit memory write
 * $[.]Waaaadddd		= 32-bit memory write
 * $[.]Laaaab			=  8-bit loop memory read *
 * $[.]Laaaah			= 16-bit loop memory read * 
 * $[.]Laaaaw			= 32-bit loop memory read *
 * $[.]LaaaaBd			=  8-bit loop memory write *
 * $[.]LaaaaHdd			= 16-bit loop memory write *
 * $[.]LaaaaWdddd		= 32-bit loop memory write *
 *
 * *: LOOP commands are terminated by sending ESCAPE (0x1B)
 *
 * Copyright 2003-2005 Dave Dunfield - All rights reserved.
 */
#define	DDRF	*(volatile unsigned short*)0xFFC00730
#define PORTF	*(volatile unsigned short*)0xFFC00700
#define	PORTF_FER *(volatile unsigned short*)0xFFC03200

#define ULCR	*(volatile unsigned short*)0xFFC0040C
#define	UMCR	*(volatile unsigned short*)0xFFC00410
#define	ULSR	*(volatile unsigned short*)0xFFC00414
#define	UTHR	*(volatile unsigned short*)0xFFC00400
#define	URBR	*(volatile unsigned short*)0xFFC00400
#define	UIER	*(volatile unsigned short*)0xFFC00404
#define	UIIR	*(volatile unsigned short*)0xFFC00408
#define	UDLL	*(volatile unsigned short*)0xFFC00400
#define	UDLH	*(volatile unsigned short*)0xFFC00404
#define	USCR	*(volatile unsigned short*)0xFFC0041C
#define	UGCTL	*(volatile unsigned short*)0xFFC00424


/*
 * Write a character to the UART
 */
void putc(unsigned char c)
{
    while(!(ULSR & 0x40))
    	;
    UTHR = c;
}

/*
 * Get character from UART
 */
unsigned testc(void)
{
    if(ULSR & 0x01)	// Data received
    	return URBR & 0xFF;
    return 0xFFFF;
}

// Startup entry point - used to restart kernel
extern void BOOT(void);

/*
 * Send 16-bit data to host
 */
void put16(unsigned n)
{
	putc(n >> 8);
	putc(n);
}

/*
 * Send 32-bit data to host
 */
void put32(unsigned long n)
{
	putc(n >> 24);
	putc(n >> 16);
	putc(n >> 8);
	putc(n);
}

/*
 * Wait for character/byte from host
 */
unsigned getc(void)
{
	unsigned v;

	while((v = testc()) == 0xFFFF);
	return v;
}

/*
 * Wait for 8-bit data from host
 * Timeout and restart if no character received.
 */
unsigned get8(void)
{
	unsigned v;
	unsigned long t;

	t = 0x100000;
	while((v = testc()) == 0xFFFF) {
		if(!--t) {
			putc('?');
			t = 0x100000;
			while(!--t);
			BOOT(); } }
	return v;
}

/*
 * Get 16 bit data from host (with  timeout)
 */
unsigned get16(void)
{
	unsigned v;
	v = get8() << 8;
	v |= get8();
	return v;
}

/*
 * Get 32 bit data from host (with timeout)
 */
unsigned long get32(void)
{
	unsigned long v;

	v = get8() << 24;
	v |= (get8() << 16);
	v |= (get8() << 8);
	v |= get8();
	return v;
}


/*
 * DESMON main program - wait for host commands and process
 */
void main(void)
{
	unsigned i;
	unsigned long a, l;

    PORTF_FER = 0x0F;	// Enable uarts
    DDRF = 0xFFF0;		//Eable GPIO's

    UGCTL	= 0x01;		// Enable UART clocks
    ULCR	= 0x83;		// Access divisor latch
    UDLL	= 70;		// 326 = 9600 bps?
    UDLH	= 1;
    ULCR	= 0x03;		// 8 bits, no parity, 1 stop
    UMCR	= 0x00;		// No disconnect
    UIER	= 0x00;		// No interrupt
 
	i = 0;
	while(--i);
	putc('!');		// Indicate startup   

	// Main loop - wait for commands and process
	for(;;) {
		while(getc() != '$');		// Wait for lead-in
		putc('.');					// Send response
		switch(get8()) {			// Get command-code
		case '-' :					// Verify kernel
			putc('+');
			continue;
		case 'b' :	// Byte read
			a = get32();
			putc(*(volatile unsigned char*)a);
			continue;
		case 'h' :	// Halfword read
			a = get32();
			put32(*(volatile unsigned short*)a);
			continue;
		case 'w' :	// word read
			a = get32();
			put32(*(volatile unsigned long*)a);
			continue;
		case 'B' :	// byte write
			a = get32();
			*(volatile unsigned char*)a = get8();
			continue;
		case 'H' :	// Halfword write
			a = get32();
			*(volatile unsigned short*)a = get16();
			continue;
		case 'W' :	// word write
			a = get32();
			*(volatile unsigned long*)a = get32();
			continue;
		case 'L' :	// Loop commands
			a = get32();
			switch(get8()) {
			case 'b' :	// byte read
				do {
					i = *(volatile unsigned char*)a; }
				while(testc() != 0x1B);
			endloop:
				putc('!');
				continue;
			case 'h' :	// halfword read
				do {
					i = *(volatile unsigned short*)a; }
				while(testc() != 0x1B);
				goto endloop;
			case 'w' :	// word read
				do {
					l = *(volatile unsigned long*)a; }
				while(testc() != 0x1B);
				goto endloop;
			case 'B' :	// byte write
				i = get8();
				do {
					*(volatile unsigned char*)a = i; }
				while(testc() != 0x1B);
				goto endloop;
			case 'H' :	// Halfword write
				i = get16();
				do {
					*(volatile unsigned short*)a = i; }
				while(testc() != 0x1B);
				goto endloop;
			case 'W' :	// word write
				l = get32();
				do {
					*(volatile unsigned long*)a = l; }
				while(testc() != 0x1B);
				goto endloop;
		} default: putc('?');
	} }
}
