;
; DESMO kernel for Precedia CYPHER Eval Board (80320 core)
;
; Notes on CYPHER DESMO Implementation:
; -------------------------------------
; DESMO does not know about the various address spaces in the 8051
; architecture. I have partitioned the 32 bit DESMO address within
; my kernel into the following blocks:
;	xx0000aa  <= Internal   (indirect) memory
;	xx0001aa  <= Internal/SFR (direct) memory
;	xx01aaaa  <= XDATA memory
;	xx02aaaa  <= CODE memory (external pseudo-flash on Eval Board)
; First (highest) byte is "don't care" (xx)
; 'aa' indicate actual address portion
;
; DESMO normally tests the target system to see if the kernel is installed
; before it loads it, however it initializes the serial port with RTS
; asserted which resets the Eval board, deactivating the kernel. Therefore
; the kernel always loads when you start DESMO.
;
; I have provided a simple example script which provides functions to
; turn LED's on and OFF, and a TEST script which does a sweeping "bar
; graph" type display. To run the demo, use: DESMO cypher
;
;
; Assemble with DDS ASM320: asm320 cypher
;
; Copyright 2003-2005 Dave Dunfield
; All rights reserved.
;
; Cypher internal peripheral registers
MPAGE	EQU	$92			; Memory page register
; Fixed memory addresses
RXBUF	EQU	$3FFE			; Receive  buffer (1 byte)
TXBUF	EQU	$3FFF			; Transmit buffer (1 byte)
;
FLASH	EQU	$C000			; Flash access in PGM mode
	ORG	$0000
	AJMP	entx			; Restart on SoftReset
;
; Write character[A] to serial port
;
WRCHR:	MOV	MPAGE,#=TXBUF		; Point to output buffer
	MOV	R1,#TXBUF		; Point to output buffer
	MOVX	[R1],A			; Write it
	MOV	MPAGE,#$B0		; Reset MPAGE
	MOV	R1,#$EB			; Serial 1 transmit length
	CLR	A			; Zero high
	MOVX	[R1],A			; Write high
	INC	R1			; Skip to low
	INC	A			; Packet length = 1
	MOVX	[R1],A			; Write low
; Start transfer
	MOV	R1,#$E8			; Serial DMA control
	MOV	A,#$01			; Start transfer
	MOVX	[R1],A			; Write it
; Wait for transfer to finish
	MOV	R1,#$15			; Interrupt source register
wrchr1:	MOVX	A,[R1]			; Read the data
	ANL	A,#%00100000		; Wait for port-1 complete
	JZ	wrchr1			; Wait for it
	MOVX	[R1],A			; Reset interrupt
	RET
;
; Test for character from serial port, on exit:
;   C=0, A=00 - No data available
;   C=1, A=xx - Data xx received
;
TSTCHR:	MOV	R1,#$13			; Serial interrupt source
	MOVX	A,[R1]			; Read register
	CLR	C			; No carry = No data
	ANL	A,#%00100000		; Packet available?
	JZ	tstch2			; No, report no data
	MOV	R1,#$C8			; Serial DMA 1 control
	MOV	A,#$01			; Start transfer
	MOVX	[R1],A			; Write it
	MOV	R1,#$14			; Serial interrupt source
tstch1:	MOVX	A,[R1]			; Read data
	ANL	A,#%00100000		; Wait for port-1 complete
	JZ	tstch1			; Wait for it
	MOV	R1,#$13			; Serial interrupt clear
	MOVX	[R1],A			; Clear FIFO data
	INC	R1			; Next
	MOVX	[R1],A			; Clear Packet ready
	MOV	MPAGE,#=RXBUF		; Set new memory page
	MOV	R1,#RXBUF		; Point to RX buffer
	MOVX	A,[R1]			; Read the data
	MOV	MPAGE,#$B0		; Reset MPAGE
	SETB	C			; Carry = Data Available
tstch2:	RET
;
; Get address into R2:R3:R4
;
GETADR	ACALL	RTCHR			; Get mem type
	MOV	R4,A			; Save high
	ACALL	RTCHR			; Get high address
	MOV	R3,A			; Save address
	ACALL	RTCHR			; Get low address
	MOV	R2,A			; Save address
	RET
;
; Wait for character with timeout
; R0:R2 = timeout counter
;
RTCHR:	CLR	A			; Zero ACC
	MOV	R0,A			; Zero timeout (low)
	MOV	R7,A			; Zero timeout (high)
rtchr1:	ACALL	TSTCHR			; Character ready?
	JC	tstch2			; Yes, we have it
	DJNZ	R0,rtchr1		; Timeout
	DJNZ	R7,rtchr1		; Timeout
ERROR:	MOV	A,#'?'			; Report error
	ACALL	WRCHR			; Write it
	AJMP	ENTX			; Restart
;
	ORG	$0066			; Transfer entry point
;
; Entry point - Boot code calls here to execute stub
; !!!DO NOT CHANGE first two instruction!!!
;
ENTER:	MOVX	[R1],A			; Write register
RAMTFR:	NOP				; Common for prefetch
	AJMP	entx			; Launch debugger
	MOV	A,R7			; Get return
	MOVX	[R1],A			; Set value
	RET
entx:	MOV	SP,#8			; Reset stack
	MOV	DPS,#0			; Select DPTR1
	MOV	MPAGE,#$B0		; Reset MPAGE
	MOV	A,#'$'			; Prompt
	ACALL	WRCHR			; Write it
ent1:	ACALL	TSTCHR			; Check for character
	CJNE	A,#'$',ent1		; Wait for key code
	MOV	A,#'.'			; Indicator
	ACALL	WRCHR			; Send it
	ACALL	RTCHR			; Read command
; Byte read
	CJNE	A,#'b',norb		; Not read byte
	ACALL	getadr			; Get address
	ACALL	XREAD			; Read the byte
xr1:	ACALL	WRCHR			; Write it
	AJMP	ent1			; Next command
; Byte write
norb:	CJNE	A,#'B',nowb		; Not write byte
	ACALL	GETADR			; Get address
	ACALL	RTCHR			; Get data
	ACALL	XWRITE			; Write it
	AJMP	ent1			; Next command
; Halfword read
nowb:	CJNE	A,#'h',norh		; Not read halfword
	ACALL	GETADR			; Get address
	ACALL	XREAD			; Read the data
xr2:	ACALL	WRCHR			; Send it
	ACALL	XREADI			; Read next
	AJMP	xr1			; Write & exit
; Halfword write
norh:	CJNE	A,#'H',nowh		; Not write halfword
	ACALL	GETADR			; Get address
	ACALL	RTCHR			; Get Data
	ACALL	XWRITE			; Write it
xw1:	ACALL	RTCHR			; Get Data
	ACALL	XWRITEI			; Write next
	AJMP	ent1			; And proceed
; Word read
nowh:	CJNE	A,#'w',norw		; Not read word
	ACALL	GETADR			; Get address
	ACALL	XREAD			; Read 1st
	ACALL	WRCHR			; Write it
	ACALL	XREADI			; Read 2nd
	ACALL	WRCHR			; Write it
	ACALL	XREADI			; Read 3rd
	AJMP	xr2			; Write, do 4th
; Word write
norw:	CJNE	A,#'W',noww		; Not write word
	ACALL	GETADR			; Get address
	ACALL	RTCHR			; Read 1st
	ACALL	XWRITE			; Write it
	ACALL	RTCHR			; Read 2nd
	ACALL	XWRITEI			; Write it
	ACALL	RTCHR			; Read 3rd
	ACALL	XWRITEI			; Write it
	AJMP	xw1			; Do 4th
; Verify kernal loader
verif:	CJNE	A,#'-',error1		; Not verify
	MOV	A,#'+'			; Get response
	ACALL	WRCHR			; Write it
	AJMP	ent1			; And continue
; Loop operations
noww:	CJNE	A,#'L',verif		; Not a loop command
	ACALL	GETADR			; Get address
	ACALL	RTCHR			; Read subcommand
; Loop read byte
	CJNE	A,#'b',nolrb		; Not loop read byte
lrb:	ACALL	XREAD			; Read the byte
	ACALL	TSTCHR			; test for end
	CJNE	A,#$1B,lrb		; Keep going
lrx:	MOV	A,#'!'			; End indicator
	ACALL	WRCHR			; Write it
	AJMP	ent1			; And continune
; Loop write byte
nolrb:	CJNE	A,#'B',nolwb		; Not loop write byte
	ACALL	RTCHR			; Get next byte
	MOV	R6,A			; Save for later
lwb:	MOV	A,R6			; Restore
	ACALL	XWRITE			; Write the byte
	ACALL	TSTCHR			; Test for exit
	CJNE	A,#$1B,lwb		; Keep going
	AJMP	lrx			; And continue
; Loop reading word
nolwb:	CJNE	A,#'w',nolrw		; Not loop read word
lrw:	ACALL	XREAD			; Read it
	ACALL	XREADI			; Read next
	ACALL	DEC32			; Backup
	ACALL	TSTCHR			; Test for exit
	CJNE	A,#$1B,lrw		; Keep going
	AJMP	lrx			; And continue
; Loop writing word
nolrw:	CJNE	A,#'W',error1		; report error
	ACALL	RTCHR			; Read byte
	MOV	R5,A			; Save for later
	ACALL	RTCHR			; read byte
	MOV	R6,A			; Save for later
lww:	MOV	A,R5			; Get 1st
	ACALL	XWRITE			; Write it
	MOV	A,R6			; Get 2nd
	ACALL	XWRITEI			; Write it
	ACALL	DEC32			; Backup
	ACALL	TSTCHR			; Test for exit
	CJNE	A,#$1B,lww		; Keep going
	AJMP	lrx			; And continue
ERROR1:	AJMP	error			; Indicate error
;
; Advance R3:2 to next location
;
INC32:	INC	R2			; Advance R2
	CJNE	R2,#0,incx		; No carry
	INC	R3			; Advance low
incx:	RET
;
; Decrement R3:R2
;
DEC32:	DEC	R2			; Reduce low
	CJNE	R2,#$FF,dec32a		; No borrow
	DEC	R3			; Reduce high
dec32a:	RET
;
; Read from memory
;
XREADI:	ACALL	INC32		; Advance to next
XREAD:	CJNE	R4,#0,xread2	; Not SFR/internal
; xx0000aa - internal (indirect) memory
	CJNE	R3,#0,xread1	; SFR access
	MOV	R1,2
	MOV	A,[R1]
	RET
; xx0001aa - SFR (direct) internal memory
xread1:	ACALL	CPRD		; Copy in read function
	MOV	A,R2		; Get address
	MOVX	[DPTR],A	; Patch with address
	MOV	A,#$03		; Programming mode
	ACALL	ENTER		; Call subroutine
	MOV	A,R0		; Read the value
	RET
; xx01aaaa - (XDATA) memory
xread2:	CJNE	R4,#1,xread3
	MOV	DPL,R2
	MOV	DPH,R3
	MOVX	A,[DPTR]
	RET
; xx02aaaa - (CODE) memory
xread3:	MOV	DPL,R2
	MOV	DPH,R3
	CLR	A
	MOVC	A,[A+DPTR];
	RET
;
; Write to memory
;
XWRITEI: ACALL	INC32		; Increment register
XWRITE:	CJNE	R4,#0,xwrite2	; Not internal
; xx0000aa - Internal (indirect) memory
	CJNE	R3,#0,xwrite1	; Not direct
	MOV	R1,2		; Get address
	MOV	[R1],A		; Write it
	RET
; xx0001aa - SFR (direct) memory
xwrite1: MOV R0,A		; Save write value
	MOV	DPTR,#WDIRECT	; Point to subroutine
	ACALL	CPSS		; Copy short
	MOV	A,R2		; Get address
	MOVX	[DPTR],A	; Patch it
	MOV	A,#3		; Programming mode
	AJMP	ENTER		; Call function
; xx01aaaa - (XDATA) memory
xwrite2: CJNE	R4,#1,xwrite3	; Not xdata
	MOV	DPL,R2		; Get low
	MOV	DPH,R3
	MOVX	[DPTR],A
	RET
; xx02aaaa - (CODE) memory
xwrite3: MOV R0,A		; Save write value
	MOV	DPTR,#WCODE	; Write to CODE memory
	ACALL	CPSS		; Copy it in
	MOV	A,R3		; Get high address value
	ANL	A,#%11000000	; Get block
	RR	A		; Block into bits 6:5
	ORL	A,#%00000011	; PGM/External
	MOV	R5,A		; Save mode for later
	MOV	A,R3		; Get high again
	ORL	A,#%11000000	; Offset to 16k block
	MOV	DPH,A		; Resave
	MOV	DPL,R2		; Set low offset
	MOV	A,R5		; Get mode
	AJMP	ENTER		; Perform write
;
; Install RAM subroutine to READ
CPRD:	MOV	DPTR,#RDIRECT		; Point to read function
; Copy short subroutine (length=6)
CPSS:	MOV	R7,#SSIZE		; Get size
; Install RAM subroutine
CPSUB:	MOV	DPH1,#=RAMTFR		; Set dest high
	MOV	DPL1,#RAMTFR		; Set dest low
cpsub1:	CLR	A			; Get zero
	MOVC	A,[A+DPTR]		; get data from flash
	INC	DPTR			; Advance
	INC	DPS			; DPTR1
	MOVX	[DPTR],A		; Write to external
	INC	DPTR			; Advance
	INC	DPS			; DPTR0
	DJNZ	R7,cpsub1		; Copy them all
	MOV	DPTR,#RAMTFR+2		; Set patch address
	MOV	R1,#$09			; Address control register
	MOV	R7,#$01			; Return mode
	RET

;
; RAM subroutine to READ direct memory
;
RDIRECT	NOP				; Required for SYNC
	MOV	R0,0			; Read data
	MOV	A,R7			; Get return
	MOVX	[R1],A			; Write it
	RET
SSIZE	EQU	*-RDIRECT		; Size of short function
; RAM subroutine to WRITE direct memory
WDIRECT	NOP				; Required for SYNC
	MOV	0,R0			; Write value
	MOV	A,R7			; Get return
	MOVX	[R1],A			; Write it
	RET
; RAM subroutine to write CODE memory
WCODE:	NOP			; Common code for prefetch
	MOV	A,R0		; Read result
	MOVX	[DPTR],A	; Write to xflash
	MOV	A,R7		; Get return value
	MOVX	[R1],A		; Select mode
	RET
