/*
 * DESMO target kernel for Cirrus EP7311/ARM
 *
 * Kernel commands:
 *	All kernel commands begin with '$' ... Kernel sends '.' in
 *	response and waits for remainder of command.
 *	Responses to commands are shown in [square braces]
 *	'a'=address byte (binary), 'd'=data byte (binary),
 *	all other characters and sent/received as the ASCII character.
 *
 * $[.]baaaa[d]			=  8-bit memory read
 * $[.]waaaa[dddd]		= 32=bit memory read
 * $[.]Baaaad			=  8-bit memory write
 * $[.]Waaaadddd		= 32-bit memory write
 * $[.]Laaaab			=  8-bit loop memory read *
 * $[.]Laaaaw			= 32-bit loop memory read *
 * $[.]LaaaaBd			=  8-bit loop memory write *
 * $[.]LaaaaWdddd		= 32-bit loop memory write *
 *
 * *: LOOP commands are terminated by sending ESCAPE (0x1B)
 *
 * Copyright 2003-2005 Dave Dunfield - All rights reserved.
 */

// EP7311 System registers
#define SYSCON1	*(volatile unsigned      *)0x80000100
#define SYSCON2	*(volatile unsigned      *)0x80001100
#define SYSCON3	*(volatile unsigned      *)0x80002200
#define	SYSFLG1	*(volatile unsigned      *)0x80000140
#define SYNCIO	*(volatile unsigned      *)0x80000500
#define MEMCFG1	*(volatile unsigned      *)0x80000180
#define MEMCFG2	*(volatile unsigned      *)0x800001C0
#define SDCONF	*(volatile unsigned      *)0x80002300
#define SDRFPR	*(volatile unsigned      *)0x80002340
#define UBRLCR1	*(volatile unsigned      *)0x800004C0
#define	UARTD	*(volatile unsigned char *)0x80000480

// Startup entry point - used to restart kernel
extern void BOOT(void);

/*
 * Send 8-bit character to host
 */
void putc(unsigned char c)
{
	while(SYSFLG1 & 0x00800000)
		;
	UARTD = c;
}

/*
 * Send 32-bit data to host
 */
void put32(unsigned n)
{
	putc(n >> 24);
	putc(n >> 16);
	putc(n >> 8);
	putc(n);
}

/*
 * Test for character/byte from host
 */
unsigned testc(void)
{
	if(SYSFLG1 & 0x00400000)
		return 0xFFFFFFFF;
	return (unsigned)UARTD;
}

/*
 * Wait for character/byte from host
 */
unsigned getc(void)
{
	unsigned v;
	while((v = testc()) == 0xFFFFFFFF);
	return v;
}

/*
 * Wait for 8-bit data from host
 * Timeout and restart if no character received.
 */
unsigned get8(void)
{
	unsigned v, t;
	t = 0x100000;
	while((v = testc()) == 0xFFFFFFFF) {
		if(!--t) {
			putc('?');
			t = 0x100000;
			while(!--t);
			BOOT(); } }
	return v;
}

/*
 * Get 32 bit data from host (with timeout)
 */
unsigned get32(void)
{
	unsigned v;
	v = get8() << 24;
	v |= (get8() << 16);
	v |= (get8() << 8);
	v |= get8();
	return v;
}

// Bitmap for register setup values:
//			 3322 2222 2222 1111 1111 1100 0000 0000
//			 1098 7654 3210 9876 5432 1098 7654 3210
//	SYSCON1	%0000,0000,0000,0000,0000,0001,0000,0010
//	SYSCON2	%0000,0000,0000,0000,0000,0000,0000,0000
//	SYSCON3	%0000,0000,0000,0000,0000,0000,0000,0111
//	SDCONF	%0000,0000,0000,0000,0000,0111,0010,0010
//	MEMCFG1	%1010,1100,0000,1100,0000,1111,0000,1101
//	MEMCFG2	%1111,1111,1111,1101,0001,1101,0001,1101

/*
 * DESMON main program - wait for host commands and process
 */
void xmain(void)
{
	unsigned a, i;

	// Setup EP7311 configuration
	// you may need to change this for other hardware configurations
	SYSCON1 = 0x00030102;
	SYSCON2 = 0x00000000;
	SYSCON3 = 0x00000007;
	SDCONF  = 0x00000722;
	MEMCFG1 = 0xAC0D0F0C;
	MEMCFG2 = 0xFFFD1D1D;

	// Bump speed to 19200 for faster monitor access
	while(SYSFLG1 & 0x00000800);	// Wait for TX complete
	UBRLCR1 = 0x0007000B;			// 19200

	// Main loop - wait for commands and process
	for(;;) {
		while(getc() != '$');		// Wait for lead-in
		putc('.');					// Send response
		switch(get8()) {			// Get command-code
		case '-' :					// Verify kernel
			putc('+');
			continue;
		case 'b' :	// Byte read
			a = get32();
			putc(*(volatile unsigned char*)a);
			continue;
		case 'w' :	// word read
			a = get32();
			put32(*(volatile unsigned*)a);
			continue;
		case 'B' :	// byte write
			a = get32();
			*(volatile unsigned char*)a = get8();
			continue;
		case 'W' :	// word write
			a = get32();
			*(volatile unsigned*)a = get32();
			continue;
		case 'L' :	// Loop commands
			a = get32();
			switch(get8()) {
			case 'b' :	// byte read
				do {
					i = *(volatile unsigned char *)a; }
				while(testc() != 0x1B);
			endloop:
				putc('!');
				continue;
			case 'w' :	// word read
				do {
					i = *(volatile unsigned*)a; }
				while(testc() != 0x1B);
				goto endloop;
			case 'B' :	// byte write
				i = get8();
				do {
					*(volatile unsigned char*)a = i; }
				while(testc() != 0x1B);
				goto endloop;
			case 'W' :	// word write
				i = get32();
				do {
					*(volatile unsigned*)a = i; }
				while(testc() != 0x1B);
				goto endloop;
		} default: putc('?');
	} }
}
