/*
 * "Edit Big"
 *
 * This utility auto-splits a large file into chunks of <=45k so that it
 * can be edited with the DOS version of the EDT editor.
 *
 * Compile with DDS Micro-C/PC, available from: www.dunfield.com
 *
 * compile command: cc eb -fop
 */
#include <stdio.h>
#include <file.h>
#include <window.h>

#define	EDITOR		"EDT.EXE"		// Name of editor
#define	MAX_FILES	100				// Maxumum number of sub-files
#define	WSIZE		23				// Display WINDOW size
#define	TAGS		10				// Number of TAG entries

// Video attributes
#define	CO_TITLE	0x67			// TITLE screens
#define	CO_FOOT		0x63			// Bottom/status line
#define	CO_MAIN		0x17			// Main window
#define	CO_ERROR	0x4E			// Error messages
#define	CO_PROMPT	0x24			// Prompts
#define	CO_HILITE	0x71			// Hilited entries

unsigned
	select,						// Main menu selection
	tag,						// View tag selection
	line,						// Current view line
	maxh,						// Maximum display width
	current_file,				// Open file
	current_line,				// Open file line
	current_base,				// Open file base line
	total_lines,				// Total lines in file
	tab_size = 4,				// Tab width selection
	hpos = 0,					// View horizontal position
	fmax,						// Number of block files
	bline,						// Blank line counter
	tag_line[TAGS],				// Tagged line numbers
	bsize = 45000,				// Blocking chunk size
	lines[MAX_FILES],			// # lines in each block
	ladjust[MAX_FILES],			// Adjustment for edits
	bytes[MAX_FILES],			// # bytes in each block
	badjust[MAX_FILES];			// Adjustment for edits

FILE
	*fpr, *fpw;

unsigned char
	fix,						// input file fixups
	eflag,						// Editing has occured flag
	tsave,						// Tag save flag
	*filename,					// Filename
	*message,					// Error message
	highword[4] = { 0x00, 0x00, 0xFF, 0xFF },	// Add to make negative
	tag_file[65],				// Tag filename
	tag_text[TAGS][31],			// Tagged line descriptions
	temp[65],					// Temporary directory
	editor[65],					// Editor to invoke
	search[61],					// Search buffer
	Filename[256],				// Short filename
	buffer[4096];				// General buffer

unsigned char Help[] = { "\n\
Use:	EB	filename [-options] [line#]\n\n\
opts:	B	- remove duplicate Blank lines\n\
	L	- convert Leading whitespace into tabs\n\
	T	- remove Trailing whitespace from lines\n\
	F	- apply all Fixes (B, L & T)\n\
\nCopyright 2005-2009 Dave Dunfield\nAll rights reserved.\n " };

/*
 * Flush input line to output file, applying corrections enabled
 */
unsigned flush_line(void)
{
	unsigned s, p, t;

	if(fix & 0x01) {		// Remove trailing whitespace
		while(maxh && isspace(buffer[maxh-1]))
			--maxh; }

	buffer[maxh] = p = s = t = 0;

	if(fix & 0x02) {		// Convert leading spaces to tabs
t1:		switch(buffer[p]) {
		case ' ' :
			++p;
			++s;
			goto t1;
		case '\t':
			++p;
			do {
				++s; }
			while(s & 3);
			goto t1; }
		if(s < 4)			// No Tab fix
			p = 0;
		else {
			while(s >= 4) {
				++t;
				s -= 4; }
			switch(s) {
			case 1 :
				if(buffer[p] == '*')
					++p;
				break;
			case 2 :
			case 3 :
				++t; } }
		maxh -= p;
		s = 0; }

	if(fix & 0x04) {		// Suppress multiple blank lines
		if(!(maxh|t)) {			// Blank line
			++bline;
			return 0; }
		if(bline) {				// Not blank, output pending
			putc('\n', fpw);
			++s; } }

	while(t) {				// Pending tabs
		putc('\t', fpw);
		--t; }

	if(maxh)
		fput(buffer+p, maxh, fpw);
	putc('\n', fpw);
	maxh = bline = 0;
	return s+1;
}

/*
 * Split input file into 'bsize' sized chunks
 */
void split_file(char *f)
{
	int c;
	unsigned b, l;

	fpr = fopen(f, "rvq");
open_next:
	sprintf(buffer, "%s\\EB%u.EB", temp, ++fmax);
	fpw = fopen(buffer, "wvq");
	l = b = maxh = bline = 0;
	for(;;) {
		if((c = getc(fpr)) == EOF) {
			if(maxh)
				l += flush_line();
			lines[fmax-1] = l;
			bytes[fmax-1] = b;
			fclose(fpw);
			fclose(fpr);
			return; }
		++b;
		if(c == '\n') {
			l += flush_line();
			if(b >= bsize) {
				lines[fmax-1] = l;
				bytes[fmax-1] = b;
				fclose(fpw);
				goto open_next; }
			continue; }
		buffer[maxh++] = c; }
}

/*
 * Concatinate sub-files back into single large output file
 */
void save_file(char *f)
{
	int c;
	unsigned i;

	fpw = fopen(f, "wvq");
	for(i=0; i < fmax; ++i) {
		sprintf(buffer, "%s\\EB%u.EB", temp, i+1);
		fpr = fopen(buffer, "rvq");
		while((c = getc(fpr)) != EOF)
			putc(c, fpw);
		fclose(fpr); }
	fclose(fpw);
}

/*
 * Read sub-file just edited and calculate
 * adjustments to byte and line counts.
 */
void calc_adjust(void)
{
	int c;
	unsigned x, l;

	sprintf(buffer, "%s\\EB%u.EB", temp, select+1);
	if(fpr = fopen(buffer, "rv")) {
		l = x = 0;
		while((c = getc(fpr)) != EOF) {
			++x;
			if(c == '\n')
				++l; }
		fclose(fpr);
		badjust[select] = x - bytes[select];
		ladjust[select] = l - lines[select]; }
}

/*
 * Read character "continuously" from sub-files (crossing file boundaries)
 */
int readc(void)
{
	int c;

	if(!fpr)
		return -1;
	while((c = getc(fpr)) == EOF) {
		fclose(fpr);
		if(++current_file > fmax) {
			current_file = current_line = fpr = 0;
			return -1; }
		sprintf(buffer, "%s\\EB%u.EB", temp, current_file);
		fpr = fopen(buffer, "rvq");
		current_base = current_line; }
	if(c == '\n')
		++current_line;
	return c;
}

/*
 * Display one line from file
 */
void display_line(void)
{
	int c;
	unsigned p, ht;

	p = 0;
	ht = hpos + 80;
	while((c = readc()) != '\n') {
		if(c == EOF)
			break;
		if(c == '\t') {
			do {
				if((p >= hpos) && (p < ht))
					wputc(' ');
				++p; }
			while(p % tab_size);
			continue; }
		if((p >= hpos) && (p < ht))
			wputc(c);
		++p; }
	if(p > maxh)
		maxh = p;
}

/*
 * Reposition to line number and correct sub-file
 */
void goto_line(unsigned l)
{
	unsigned i, n, ln;

	for(i=n=0; i < fmax; ++i) {
		ln = n;
		n += (lines[i] + ladjust[i]);
		if(l < n) {		// Line is in this file
			if(((i+1) != current_file) || (l < current_line)) {
				if(fpr)
					fclose(fpr);
				sprintf(buffer, "%s\\EB%u.EB", temp, current_file = i+1);
				fpr = fopen(buffer, "rvq");
				current_line = current_base = ln; }
			while(l > current_line) {
				if(readc() == EOF)
					return -1; }
				return 0; } }
}

/*
 * Prompt for an input string
 */
int prompt(char *s, char *d, unsigned l)
{
	unsigned x;
	wgotoxy(0, 0);
	*W_OPEN = CO_PROMPT;
	wputs(s);
	wcleol();
	x = W_OPEN->WINcurx+1;
	for(;;) switch(wgets(x, 0, d, l)) {
		case 0x1B:	
			*W_OPEN = CO_MAIN;
			return -1;
		case '\n':
			*W_OPEN = CO_MAIN;
			return 0; }
}

/*
 * Select or set a tag
 */
int select_tag(char x)
{
	unsigned i;
	char d;

retitle:
	if(!x) {		// Setting tag
		wgotoxy(0, 0);
		*W_OPEN = CO_TITLE;
		wprintf(" Tag saving is %s - Press F10 to toggle.", tsave ? "ON" : "OFF");
		wcleol(); }

	wopen(x?10:30, 5, 40, TAGS+2, WSAVE|WBOX1|WCOPEN|CO_PROMPT);
	wcursor_off();
	d = 0;
redraw:
	i =  tag;
	while(x && !tag_line[tag]) {
		if(d)	// Going UP
			tag = (tag ? tag : TAGS)-1;
		else	// Going DOWN
			tag = (tag < (TAGS-1)) ? tag+1 : 0;
		if(tag == i) {
			wclose();
			message = "No Tags set";
			return -1; } }
	for(i=0; i < TAGS; ++i) {
		wgotoxy(0, i);
		*W_OPEN = (i == tag) ? CO_HILITE : CO_PROMPT;
		if(!tag_line[i])
			wputs("-----");
		else
			wprintf("%-6u%s", tag_line[i], tag_text[i]); }
	for(;;) switch(wgetc()) {
	case _KUA :	d=-1;	tag = (tag ? tag : TAGS)-1;			goto redraw;
	case _KDA : d=0;	tag = (tag < (TAGS-1)) ? tag+1 : 0;	goto redraw;
	case _KHO : d=tag=0;									goto redraw;
	case _KEN : d=tag=TAGS-1;								goto redraw;
	case '\n' :	wclose();	return 0;
	case 0x1B :	wclose();	return -1;
	case _K10 :
		if(!x) {
			tsave = tsave ? 0 : -1;
			wclose();
			goto retitle; } }
}

/*
 * View file in one complete display
 */
void browse(void)
{
	int c;
	unsigned i, f, b, l;
	unsigned char *p, cc;
	static char cf;

reopen:
	fpr = 0;
redraw:
	goto_line(line);
	b = current_base;
	wgotoxy(0, 0);
	if(message) {
		*W_OPEN = CO_ERROR;
		wputs(message);
		message = 0; }
	else {
		*W_OPEN = CO_TITLE;
		wprintf("%s: %-5u of %-5u  [Sec:%-4u %-5u]%5u-%-5u",
			filename, l = current_line+1, total_lines,
			f = current_file, l-current_base, hpos+1, hpos+80); }
	wcleol();
	*W_OPEN = CO_MAIN;
	for(i=maxh=0; i < WSIZE; ++i) {
		wgotoxy(0, i+1);
		wcleol();
		display_line();
		if(!fpr) {
			*W_OPEN = CO_HILITE;
			wputs("*EOF*");
			*W_OPEN = CO_MAIN;
			wcleow();
			break; } }
	*W_OPEN = CO_FOOT;
	wgotoxy(0, 24);
	wputs("F1:goto F2/F3/F4:find/cs/again F5:gotag F6:setag F7:tabsize ENTER:Edit ESC:exit");
	wcleol();
	*W_OPEN = CO_MAIN;
recmd:
	switch(c = wgetc()) {
	case _KRA :	++hpos;								goto redraw;
	case _KLA : if(hpos) --hpos;					goto redraw;
	case _KHO :	hpos = 0; 							goto redraw;
	case _KEN : hpos = (maxh > 80) ? maxh - 80 : 0;	goto redraw;
	case _KUA :	if(line) --line;					goto redraw;
	case _KDA : if(++line >= total_lines) --line;	goto redraw;
	case _KPU : line = (line < WSIZE)?0:line-WSIZE;	goto redraw;
	case _CPU :	line = 0;							goto redraw;
	case _KPD : if((line += WSIZE) < total_lines)	goto redraw;
	case _CPD :	line = (total_lines > WSIZE) ? total_lines-WSIZE:0;	goto redraw;
	case _K1 :	// Goto line
		*buffer = 0;
		if(prompt("Goto?", buffer, 0x85))
			goto redraw;
		if(i = atoi(buffer))
			line = ((i < total_lines) ? i : total_lines) - 1;
		goto redraw;
	case _K2 :	// Search (not case sensitive)
		if(prompt("Find?", search, sizeof(search)-1))
			goto redraw;
		cf = -1;
		strupr(search);
		goto dosearch1;
	case _K3 :	// Search (case sensitive)
		if(prompt("Find?", search, sizeof(search)-1))
			goto redraw;
		cf = 0;
	dosearch1:
		goto_line(line);
		goto dosearch;
	case _K4 :	// Search again
		goto_line(line+1);
	dosearch:
		for(;;) {
			p = buffer;
			while((c = readc()) != '\n') {
				if(c == EOF) {
					message = "Not found";
					goto redraw; }
				*p++ = c; }
			*p = 0;
			if(cf)
				strupr(buffer);
			for(p=buffer; *p; ++p) {
				if(strbeg(p, search)) {
					line = current_line-1;
					goto redraw; } }
			if(!++cc) {
				if(wtstc() == 0x1B) {
					message = "Aborted!";
					goto redraw; } } }
	case _K5 :	// Goto Tag
		if(select_tag(-1))
			goto redraw;
		line = tag_line[tag]-1;
		goto redraw;
	case _K6 :	// Set Tag
		if(select_tag(0))
			goto redraw;
		if(!prompt("Description?", tag_text[tag], sizeof(tag_text[0])-1))
			tag_line[tag] = line+1;
		goto redraw;
	case _K7 :	// Tab size
		sprintf(buffer, "%u", tab_size);
		if(prompt("Tab width?", buffer, 0x83))
			goto redraw;
		if(i = atoi(buffer))
			tab_size = i;
		goto redraw;
	case '\n' :
		if(fpr)
			fclose(fpr);
		current_line = l;
		if(f)
			select = f-1;
		wclose();
		sprintf(buffer, "%s\\EB%u.EB C=%u", temp, select+1, l-b);
		eflag = -1;
		exec(editor, buffer);
		calc_adjust();
		wopen(0, 0, 80, 25, WSAVE|WCOPEN|CO_MAIN);
		wcursor_off();
		for(i=total_lines = 0; i < fmax; ++i)
			total_lines += (lines[i] + ladjust[i]);
		goto reopen;
	case 0x1B :
		if(fpr)
			fclose(fpr);
		current_file = fpr = 0;
		return c; }
	goto recmd;
}

/*
 * Load previously saved poosition tags
 */
void load_tags(void)
{
	unsigned t;
	unsigned char c, *p, *p1, *d;

	p1 = Filename;
restart:
	p = tag_file;
	d = 0;
	while(*p++ = c = toupper(*p1++)) switch(c) {
		case ':' :
		case '\\' :
			goto restart;
		case '.' :
			d = p; }
	if(!d)
		strcpy(p-1, ".$");
	else {
		if(*d) {
			++d;
			if(*d)
				++d; }
		strcpy(d, "$"); }
	if(fpr = fopen(tag_file, "rv")) {
		for(t=0; t < TAGS; ++t) {
			fgets(p=buffer, 50, fpr);
			c = 0;
			while(isdigit(*p)) {
				c = -1;
				++p; }
			if(!c) {
				message = "Bad tag file format";
				fclose(fpr);
				return; }
			while(isspace(*p))
				++p;
			tag_line[t] = atoi(buffer);
			p1 = tag_text[t];
			while(*p1++ = *p++); }
		fclose(fpr); }
}

/*
 * Get short filename from long filename
 */
int get_short(short, long) asm
{
		PUSH	DS				; Get DS
		POP		ES				; Set ES
		MOV		SI,6[BP]		; Get source (short)
		MOV		DI,4[BP]		; Get dest   (long)
		MOV		CX,8001h		; Allow SUBST
		MOV		AX,7160h		; LFN service
		INT		21h				; Ask DOS
}

/*
 * Test for long filename & resolve
 */
void test_long(void)
{
	int c;
	unsigned n, e, s;
	unsigned char ef, lf, *f, *p;

	f = filename;
	p = Filename;
	lf = 0;
	if(f[1] == ':') {
		*p++ = *f++;
		*p++ = *f++; }
	while(*f == '\\')
		*p++ = *f++;
new:
	n = e = s = ef = 0;
	for(;;) switch(*p++ = c = *f++) {
	case ' ' : ++s;
	default:
		if(ef)
			++e;
		else
			++n;
		continue;
	case '.' : ef = 255;	continue;
	case '/' : *(p-1) = *(f-1) = '\\';
	case '\\':
	case 0 :
		if((n > 8) || (e > 3) || s)
			lf = 255;
		if(c) goto new;
		if(lf) {
			if(n = get_short(filename, Filename)) {
				printf("LFN resolve failed (%u)", n);
				exit(-1); } }
		filename = f = Filename;
		for(;;) switch(*f++) {
		case 0 :
			return;
		case ':' :
		case '\\': filename = f; } }
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i, l, ll, s1, s2, x, y, bt[2], l1[2], l2[2], sl;
	static unsigned Xline;

	for(i=1; i < argc; ++i) {
		message = argv[i];
		if((*message == '-') || (*message == '/')) {
			for(;;) {
				switch(toupper(*++message)) {
				default: abort(Help);
				case 'T' : fix |= 0x01; continue;
				case 'L' : fix |= 0x02;	continue;
				case 'B' : fix |= 0x04;	continue;
				case 'F' : fix = 255;	continue;
				case 0 : ; }
				break; }
			continue; }
		if(filename) {
			if(isdigit(*message)) {
				Xline = atoi(message);
				continue; }
			abort(Help); }
		filename = message; }

	if(!filename)
		abort(Help);

	test_long();		// Resolve any long filenames

	i=l=0;
	while(x = (*argv)[i]) {
		editor[i++] = x;
		if((x == '\\') || (x == ':'))
			l = i; }
	strcpy(editor+l, EDITOR);

	if(getenv("TEMP", temp)) {
		for(i = x = 0; l = temp[i]; ++i)
			x = l;
		if((x == '\\') || (x == ':'))
			temp[i-1] = 0; }
	else
		strcpy(temp, ".");

	IOB_size = 4096;
	split_file(Filename);
	load_tags();
	message = s1 = 0;

reopen:
	wopen(0, 0, 80, 25, WSAVE|WCOPEN|CO_MAIN);
	wcursor_off();
redraw:
	l = 1;
	while(select < s1)
		--s1;
	while((s2 = s1+WSIZE) <= select)
		++s1;
	longset(bt, 0);
	for(i=total_lines=0; i < fmax; ++i) {
		if(i == select)
			sl = total_lines;
		total_lines += (ll = lines[i]) + (x = ladjust[i]);
		longset(l1, bytes[i]);
		longset(l2, y = badjust[i]);
		if(y & 0x8000)
			longadd(l2, highword);	// Make negative
		longadd(l1, l2);
		longadd(bt, l1);
		if((i >= s1) && (i < s2)) {
			wgotoxy(0, (i-s1)+1);
			*W_OPEN = (i == select) ? CO_HILITE : CO_MAIN;
			*buffer = buffer[50] = 0;
			if(x) {
				if(x & 0x8000)
					sprintf(buffer,"-%u", -x);
				else
					sprintf(buffer, "+%u", x); }
			if(y) {
				if(y & 0x8000)
					sprintf(buffer+50, "-%u", -y);
				else
					sprintf(buffer+50, "+%u", y); }
			wprintf("%4u   %5u-%-5u   L:%5u%-6s=%-5u   B:%-5u%-6s=%u\n",
			i+1, l, (l+ll+x)-1,
			ll, buffer, ll+x, 
			bytes[i], buffer+50, bytes[i]+y); }
		l += ll+x; }

	wgotoxy(0, 0);
	if(message) {
		*W_OPEN = CO_ERROR;
		wputs(message);
		message = 0; }
	else {
		*W_OPEN = CO_TITLE;
		ltoa(bt, buffer, 10);
		wprintf(" %s: %u sections, %u lines %s bytes", filename, fmax, total_lines, buffer); }
	wcleol();
	*W_OPEN = CO_FOOT;
	wgotoxy(0, 24);
	wputs(" ENTER:view  SPACE:view-last  F1:edit  ESC:exit");
	wcleol();
	*W_OPEN = CO_MAIN;

	if(Xline) {
		sl = Xline - 1;
		Xline = 0;
		goto doview; }
	switch(wgetc()) {
	case _KHO :	select = 0;							break;
	case _KEN : select = fmax-1;					break;
	case _KUA : if(select) --select;				break;
	case _KDA : if(++select >= fmax) --select;		break;
	case '\n' :	// View
	doview:
		line = sl;
	case ' ' :	// View last
		browse();
		wclwin();
		goto redraw;
	case _K1 :	// Edit
		wclose();
		sprintf(buffer, "%s\\EB%u.EB", temp, select+1);
		eflag = -1;
		exec(editor, buffer);
		calc_adjust();
		goto reopen;
	case 0x1B :
		if(!eflag)
			goto quit;
		wopen(20, 10, 40, 3, WSAVE|WCOPEN|WBOX1|CO_PROMPT);
		wprintf("Save file '%s' (Y/N)?", filename);
		for(;;) switch(wgetc()) {
		case 'y' :
		case 'Y' :
			wclose();
			wclose();
			save_file(Filename);
			goto delfile;
		case 'n' :
		case 'N' :
			wclose();
		quit:
			wclose();
		delfile:
			for(i=0; i < fmax; ++i) {
				sprintf(buffer, "%s\\EB%u.EB", temp, i+1);
				if(x = delete(buffer))
					printf("Error %d erasing: %s\n", x, buffer); }
			if(tsave) {
				fpw = fopen(tag_file, "wvq");
				for(i=0; i < TAGS; ++i) {
					if(x = tag_line[i])
						fprintf(fpw, "%-6u%s\n", x, tag_text[i]);
					else
						fputs("0\n", fpw); }
				fclose(fpw); }
			return;
		case 0x1B :
			wclose();
			goto redraw; } }
	goto redraw;
}
