/*
 * Edit BIOS drive table - write EVEN.BIN & ODD.BIN files
 *
 * Run EDTABLE on the machine containing the BIOS to be modified.
 * If necessary, change the BIOS segment, size, drive table offset
 * and number of drive entries to match your BIOS. Then select
 * Proceed. You will be presented with the drive table currently
 * existing in the BIOS. Press SPACE to change an entry, enter the
 * entry number, then select the parameters to change, and enter
 * new values. Press ESC to return to the entry selection.
 *
 * When you have finished changing all the entries that you want to
 * change, press ENTER. EDTABLE will write two binary files called
 * EVEN.BIN and ODD.BIN, which contain the even and odd bytes of the
 * bios code respectively.
 *
 * Copyright 1993-2003 Dave Dunfield
 * All rights reserved.
 *
 * Permission granted to use for personal (non-commercial) use only.
 *
 * Compile command: cc EDTABLE -fop
 */
#include <stdio.h>
#include <window.h>

unsigned
	bios_seg = 0xF000,		/* Bios segment */
	bios_size = 0x0000, 	/* Bios size (in bytes - 0=64K) */
	dt_offset = 0xE401,		/* Offset to drive table */
	dt_size = 47;			/* Number on entries */

unsigned work_seg;			/* RAM buffer for BIOS copy */

main()
{
	unsigned i, j, k, start;
	FILE *fpe, *fpo;

	if(!(work_seg = alloc_seg(4096)))
		abort("Cannot allocate work buffer");

	wopen(0, 0, 80, 25, WSAVE|WCOPEN|WBOX1|NORMAL);
	wopen(15, 2, 50, 3, WCOPEN|WBOX2|REVERSE);
	wputs("   EDTABLE - Edit BIOS hard drive parameters");
	wclose();
	wcursor_off();
	wgotoxy(11, 21);
	wputs("Copyright 1993-2003 Dave Dunfield - All rights reserved.");

	/* Display starting parameters & prompt for changes */
top:
	wgotoxy(17, 7); wprintf("1) BIOS segment         = %04x", bios_seg);
	wgotoxy(17, 9); wprintf("2) BIOS size (in bytes) = %-5u (0=64K)", bios_size);
	wgotoxy(17,11); wprintf("3) Drive table offset   = %04x", dt_offset);
	wgotoxy(17,13); wprintf("4) Number of entries    = %-5u", dt_size);
	wgotoxy(17,15); wprintf("5) Proceed");
	wgotoxy(17,17); wprintf("6) Quit to DOS");
	switch(wgetc()) {
		case '1' :
			bios_seg = get_number("BIOS segment", bios_seg, 16);
			goto top;
		case '2' :
			bios_size = get_number("BIOS size", bios_size, 10);
			goto top;
		case '3' :
			dt_offset = get_number("Drive table offset", dt_offset, 16);
			goto top;
		case '4' :
			dt_size = get_number("Drive table size", dt_size, 10);
		default:
			goto top;
		case 0x1B:
		case '6' : wclose(); return;
		case '5' : }

	/* Read BIOS into work buffer */
	i = start = 0;
	do {
		poke(work_seg, i, peek(bios_seg, i)); }
	while(++i != bios_size);

	/* Display drive table & allow editing */
	wclwin();
top1:
	for(i = start; (k = i - start) < 22; ++i) {
		wgotoxy(0, k);
		if(i >= dt_size) {
			wcleol();
			continue; }
		j = dt_offset + (i * 16);
		wprintf("#%-2u Cyl:%-5u Head:%-3u Precomp:%-5u Ctl:%02x Lzone:%-5u Sectors: %-3u",
			i+1,								/* Entry number */
			(unsigned)peekw(work_seg, j),		/* Cylinders */
			(unsigned)peek(work_seg, j+2),		/* Heads */
			(unsigned)peekw(work_seg, j+5),		/* Precomp */
			(unsigned)peek(work_seg, j+8),		/* Control */
			(unsigned)peekw(work_seg, j+12),	/* Lzone */
			(unsigned)peek(work_seg, j+14));	/* Sectors */ }
	wgotoxy(0, 22); wputs("PgUp/PgDn=Scroll, SPACE=Change, ESC=Quit, ENTER=Save");
	wcleol();
	switch(wgetc()) {
		case _KUA : if(start) --start; goto top1;
		case _KPU : start = (start < 22) ? 0 : start - 22; goto top1;
		case _KPD : start += 21;
		case _KDA : if(++start >= dt_size) start = dt_size - 1;
		default:
			goto top1;
		case 0x1B : wclose(); return;
		case '\n' : goto exit;
		case ' ' : }
	i = get_number("Entry number", 0, 10) -1;
	if(i >= dt_size)
		goto top1;
	j = dt_offset + (i * 16);

	/* Edit an individual entry */
top2:
	wgotoxy(0, 22); wprintf("#%-2u C)ylinders H)eads P)recomp K)ontrol L)zone S)ectors ESC=Quit", i+1);
	wcleol();
	switch(toupper(wgetc())) {
		case 'C' :
			pokew(work_seg, j, get_number("Cylinders", peekw(work_seg, j), 10));
			break;
		case 'H' :
			poke(work_seg, j+2, get_number("Heads", peek(work_seg, j+2), 10));
			break;
		case 'P' :
			pokew(work_seg, j+5, get_number("Precomp", peekw(work_seg, j+5), 10));
			break;
		case 'K' :
			poke(work_seg, j+8, get_number(peek("Control byte", work_seg, j+8), 16));
			break;
		case 'L' :
			pokew(work_seg, j+12, get_number("Landing zone", peekw(work_seg, j+12), 10));
			break;
		case 'S' :
			poke(work_seg, j+14, get_number("Sectors/track", peek(work_seg, j+14), 10));
			break;
		case '\n' :
		case 0x1B :
			goto top1; }
	goto top2;

	/* Write EVEN and ODD files, then terminate */
	/* If you need only a single output file, modify this section */
exit:
	wclose();
	fpe = fopen("EVEN.BIN", "wvqb");
	fpo = fopen("ODD.BIN", "wvqb");
	i = 0;
	do {
		putc(peek(work_seg, i), (i & 1) ? fpo : fpe); }
	while(++i != bios_size);
	fclose(fpe);
	fclose(fpo);
	free_seg(work_seg);
}

/*
 * Get a number of the specified base
 */
get_number(char *prompt, unsigned value, unsigned base)
{
	unsigned v, c;
	char *ptr, flag;

	switch(base) {
		case 10: ptr = "\r%s: %05u"; break;
		case 16: ptr = "\r%s: %04x"; }

	wopen(1, 23, 78, 1, WSAVE|WCOPEN|REVERSE);
	v = value;
	flag = -1;
	for(;;) {
		wprintf(ptr, prompt, v);
		if(isdigit(c = toupper(wgetc())))
			c -= '0';
		else if((c >= 'A') && (c <= 'F') && (base == 16))
			c -= 'A'-10;
		else {
			switch(c) {
				case 0x1B :	wclose(); return value;
				case '\n' : wclose(); return v;
				case _KBS : flag = 0; v /= base; }
			continue; }
		if(flag)
			v = flag = 0;
		v = (v * base) + c; }
}		
