/*
 * Program to scan hard drive to specific strings & attempt recovery
 * (See DISASTER.TXT for detailed information on this program)
 *
 * Copyright 1995-2003 Dave Dunfield
 * All rights reserved.
 *
 * Permission granted to use for personal (non-commercial) use only.
 *
 * Compile command: cc needle -fop
 */
#include <stdio.h>

#define	SECSIZE	512			/* Size of a disk sector */
#define	MAXLIST	5000		/* Maximum number of entries in match list */

/*
 * Structure for int 24/25 disk block
 */
struct {
	unsigned sec_low;
	unsigned sec_high;
	unsigned length;
	unsigned offset;
	unsigned segment; } disk;

/*
 * Sector buffer
 */
unsigned char buffer[SECSIZE];

/*
 * Pattern matching list
 */
unsigned lhigh[MAXLIST], llow[MAXLIST], loffset[MAXLIST], lcount = 0;

/*
 * Misc variables
 */
char ascii = 0;					/* Display mode */
char drive = 2;					/* Disk drive */
char *fname = "A:NEEDLE.DAT";	/* Output file */
char *ptr;						/* General pointer */
char search[81];				/* Search string */
unsigned slen;					/* Search string lenght */
unsigned High, Low;				/* Actual size of disk */

/*
 * Command line help text
 */
char help0[] = { "\n\
Use: needle [D=drive F=file M=file S=hi:lo T=text]\n\
" };

/*
 * Interactive command help text
 */
char help1[] = { "\n\
K	- Keep this sector (write to file) and advance to next\n\
N	- advance to the Next sector\n\
B	- Backup to the previous sector\n\
V	- View sector match list\n\
Paddr	- Patch sector beginning at address\n\
Ghi:lo	- Goto sector\n\
Lhi:lo	- set drive scanning Limit\n\
S	- Scan disk (prompts for text)\n\
+	- goto next sector in match list\n\
-	- goto previous sector in match list\n\
1	- display format 1 - 0000-01FF ASCII only\n\
2	- display format 2 - 0000-00FF HEX/ASCII\n\
3	- display format 3 - 0100-01FF HEX/ASCII\n\
Q	- Quit\n\nPress ENTER..." };

/*
 * Patch command help text
 */
char help2[] = { "\n\
Gaddr	- Goto byte address\n\
Q	- Quit and throw away changes\n\
S	- Save sector and exit\n\
V[1-3]	- View sector dump\n\
+	- advance to next byte\n\
-	- backup to previous byte\n\
'string	- replace byte(s) with character string\n\
hexval	- replace byte with hex value\n\n" };

/*
 * Read disk sector into memory
 */
read_disk() asm
{
		MOV		BX,OFFSET DGRP:_disk; Get parameter block
		MOV		CX,-1			; Indicate extended
		MOV		AL,DGRP:_drive	; Get disk drive
		INT		25h				; Read disk
		JC		error1			; It failed
		POPF					; Clean flags
		XOR		AX,AX			; Zero return
		POP		BP
		RET
error1:	POPF
}

/*
 * Write disk sector from memory
 */
write_disk() asm
{
		MOV		BX,OFFSET DGRP:_disk; Get parameter block
		MOV		CX,-1			; Indicate extended
		MOV		AL,DGRP:_drive	; Get disk drive
		INT		26h				; Write to disk !!!
		JC		error1			; It failed
		POPF					; Clean flags
		XOR		AX,AX			; Zero return
}

get_drive_data() asm
{
		MOV		DL,DGRP:_drive			; Get the drive ID
		INC		DL						; Convert to absolute
		MOV		AH,1Ch					; Get drive data
		PUSH	DS						; Save DS
		INT		21h						; Ask DOS
		POP		DS						; Restore DS
		XOR		AH,AH					; Zero HIGH
		MUL		DX						; DX:AX = Sectors/drive
		MOV		DGRP:_High,DX			; Set high size
		MOV		DGRP:_Low,AX			; Set low size
}

/*
 * Skip blanks & return next non-blank
 */
skip_blanks()
{
	while(isspace(*ptr))
		++ptr;
	return *ptr;
}

/*
 * Main packrat
 */
main(int argc, char *argv[])
{
	unsigned h, l, e, i, lpos, end;
	int j;
	char inline[81], c;
	FILE *fp, *fp1;

	fputs("NEEDLE v1.0 - Copyright 1995-2003 Dave Dunfield - All rights reserved.\n", stdout);

	fp = lpos = h = l = slen = 0;
	disk.length = 1;
	disk.offset = buffer;
	disk.segment = get_ds();

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case 'D=' : drive = atoi(ptr);	continue;
			case 'F=' : fname=ptr;			continue;
			case 'M=' :
				fp1 = fopen(ptr, "rvq");
				while(fgets(ptr = inline, sizeof(inline)-1, fp1)) {
					get_long(&lhigh[lcount], &llow[lcount]);
					skip_blanks();
					loffset[lcount++] = atoi(ptr); }
				fclose(fp1);
				continue;
			case 'S=' :
				if(get_long(&h, &l))
					goto badopt;
				continue;
			case 'T=' : set_search(ptr);	continue;
			case '-?' :
			case '/?' :
			case '?'<<8 : abort(help0); }
	badopt:
		printf("Invalid option: %s\n", argv[i]);
		exit(-1); }

	get_drive_data();
	printf("Drive size=%x:%04x\n", High, Low);

	if(!slen)
		goto grabfile;

/*
 * Scan the disk, matching the pattern
 */
scandisk:
	if(!slen) {
		printf("\7***No search string   ");
		goto grabfile; }
	end = SECSIZE - slen;
	printf("Scanning '");
	show_search();
	printf("' (ESC to cancel):\n");

	for(;;) {
		disk.sec_high = h;
		do {
			if((h >= High) && (l >= Low)) {
				printf("\7***End of drive   ");
				goto grabfile; }
			if(!(l & 0x0F)) {
				printf("\r%x:%04x", h, l);
				if(kbtst() == 0x1B) {
					putc('\n', stdout);
					goto grabfile; } }
			disk.sec_low = l;
			if(e = read_disk())
				printf("Error (%04x) %x:%04x\n", e, h, l);
			if(i = scansec(end, search, slen)) {
				printf("\r%x:%04x \7***FOUND***", h, l);
				if(lcount < MAXLIST) {
					lhigh[lcount] = h;
					llow[lcount] = l;
					loffset[lcount++] = i-1; }
				else
					printf(" LIST FULL!");
				putc('\n', stdout); }
		} while ++l;
		++h; }

/*
 * Interactive disk commands
 */
grabfile:
	for(i=0; i < lcount; ++i)
		if((lhigh[i] == h) && (llow[i] == l)) {
			printf("List(%u) ", i);
			break; }
	printf("Sector %x:%04x", h, l);
	disk.sec_high = h;
	disk.sec_low = l;
	if(e = read_disk())
		printf(" \7Error: %u", e);
	putc('\n', stdout);
	dump_block(0);
	printf("K)eep N)ext B)ack V)iew P[addr] G[hi:lo] L[hi:lo] S)can +- 123 Q)uit?");
	fgets(ptr = inline, sizeof(inline)-1, stdin);
	switch(toupper(skip_blanks())) {
		case 'K' :	/* Keep this sector */
			if(!fp)
				fp = fopen(fname, "wvqba");
			fput(buffer, SECSIZE, fp);
		case 'N' :	/* Next block */
			if(!++l)
				++h;
			goto grabfile;
		case 'B' :	/* Backup block */
			if(!l)
				--h;
			--l;
			goto grabfile;
		case 'V' :	/* View list */
			for(i = 0; i < lcount; ++i) {
				printf("%-3u %3x:%04x ", i, disk.sec_high = lhigh[i], disk.sec_low = llow[i]);
				read_disk();
				e = (64 - slen)/2;
				j = loffset[i] - e;
				if(j < 0) j = 0;
				if(j > (SECSIZE-64)) j = SECSIZE-64;
				for(e=0; e < 64; ++e)
					putc(isprint(c = buffer[j+e]) ? c : ' ', stdout);
				putc('\n', stdout);
				if(i && !(i % 20)) {
					printf("MORE: # Q)uit W[file] ?");
					fgets(ptr = inline, 10, stdin);
					if(isdigit(c = toupper(skip_blanks()))) goto movelist;
					if(c == 'Q') goto grabfile;
					if(c == 'W') goto writelist; } }
			printf("END: # W[file] ?");
			fgets(ptr = inline, 10, stdin);
			if(isdigit(c = toupper(skip_blanks()))) goto movelist;
			if(c != 'W') goto grabfile;
		writelist:
			++ptr; skip_blanks();
			if(fp1 = fopen(ptr, "wv")) {
				for(i=0; i < lcount; ++i)
					fprintf(fp1, "%03x:%04x %u\n", lhigh[i], llow[i], loffset[i]);
				fclose(fp1); }
			goto grabfile;
		movelist:
			if((i = atoi(ptr)) >= lcount) {
				printf("\7***Not on list   ");
				goto grabfile; }
			h = lhigh[i];
			l = llow[i];
			goto grabfile;
		case 'P' :
			++ptr; skip_blanks();
			patch(atox(ptr));
			goto grabfile;
		case 'G' :	/* GOTO */
			++ptr;
			get_long(&h, &l);
			goto grabfile;
		case 'L' :	/* Limits */
			++ptr;
			if(skip_blanks())
				get_long(&High, &Low);
			printf("Scanning limit is currently: %x:%04x\n", High, Low);
			goto grabfile;
		case 'S' :	/* Scan disk */
			if(fp) fclose(fp);
			printf("Scan '");
			show_search();
			printf("' (ENTER or new text): ");
			fgets(inline, sizeof(inline)-1, stdin);
			if(*inline)
				set_search(inline);
			goto scandisk;
		case '+' :	/* Forward in list */
			e = h; i = l;
			do {
				if(lpos >= lcount) {
					printf("\7***At end of list   ");
					goto grabfile; }
				h = lhigh[lpos];
				l = llow[lpos++]; }
			while((h == e) && (l == i));
			goto grabfile;
		case '-' :	/* Backup in list */
			e = h; i = l;
			do {
				if(!lpos) {
					printf("\7***At beginning of list   ");
					goto grabfile; }
				h = lhigh[--lpos];
				l = llow[lpos]; }
			while((h == e) && (l == i));
			goto grabfile;
		case '1' : ascii = 0; goto grabfile;
		case '2' : ascii = 1; goto grabfile;
		case '3' : ascii = 2; goto grabfile;
		case '?' :
			fputs(help1, stdout);
			fgets(inline, sizeof(inline)-1, stdin);
			goto grabfile;
		default:
			printf("\7***Unknown command   ");
			goto grabfile;
		case 'Q' :
			if(fp)
				fclose(fp); }
}

/*
 * Search a sector for an occurance of a search string
 */
scansec(end, search, slen) asm
{
		XOR		BX,BX				; Offset zero
		MOV		DX,8[BP]			; Get END
; Locate possible start of block
scan1:	MOV		DI,6[BP]			; Point to start of search
		MOV		AL,[DI]				; Get char
; Scan for character
scan2:	CMP		AL,DGRP:_buffer[BX]	; Is this it?
		JZ		scan4				; Yes, it is!
scan3:	INC		BX					; Advance pointer
		CMP		BX,DX				; Are we over?
		JB		scan2				; Do them all
		XOR		AX,AX				; No result
		POP		BP					; Restore caller
		RET
; We found the first character... See if this is it!
scan4:	INC		BX					; Skip to next
		LEA		SI,DGRP:_buffer[BX]	; Point to starting position
		MOV		CX,4[BP]			; Get length
		DEC		CX					; Reduce count
		JZ		scan6				; Special case
scan5:	INC		DI					; Advance in dest
		MOV		AL,[SI]				; Get source
		CMP		AL,[DI]				; Test with dest
		JNZ		scan1				; Match failed
		INC		SI					; Advance source
		LOOP	scan5				; Check them all
scan6:	MOV		AX,BX				; Return value
}

/*
 * Dump a block in the selected format
 */
dump_block()
{
	unsigned h, l;
	char c;

	switch(ascii) {
		case 0 :	/* ASCII display */
			for(h=0; h < 8; ++h) {
				printf("%04x ", h*64);
				for(l=0; l < 64; ++l)
					putc(isprint(c = buffer[(h*64)+l]) ? c : ' ', stdout);
				putc('\n', stdout); }
			break;
		case 1 :	/* HEX low */
			for(h=0; h < 16; ++h) {
				printf("%04x", h*16);
				for(l=0; l < 16; ++l) {
					if(!(l & 3)) putc(' ', stdout);
					printf(" %02x", buffer[(h*16)+l]); }
				printf("  ");
				for(l=0; l < 16; ++l)
					putc(isprint(c = buffer[(h*16)+l]) ? c : '.', stdout);
				putc('\n', stdout); }
			break;
		case 2 :	/* HEX high */
			for(h=16; h < 32; ++h) {
				printf("%04x", h*16);
				for(l=0; l < 16; ++l) {
					if(!(l & 3)) putc(' ', stdout);
					printf(" %02x", buffer[(h*16)+l]); }
				printf("  ");
				for(l=0; l < 16; ++l)
					putc(isprint(c = buffer[(h*16)+l]) ? c : '.', stdout);
				putc('\n', stdout); } }
}

/*
 * Patch the sector
 */
patch(unsigned addr)
{
	unsigned char inline[50], c;

	for(;;) {
		if(addr >= SECSIZE)
			addr = 0;
		printf("%04x(", addr);
		if(isprint(c = buffer[addr]))
			printf("%02x %c): ", c, c);
		else
			printf(" %02x ): ", c);
		fgets(ptr = inline, sizeof(inline)-1, stdin);
		switch(toupper(skip_blanks())) {
			case '\'' :
				while((c = *++ptr) && (addr < SECSIZE))
					buffer[addr++] = c;
				continue;
			default:
				if(!isxdigit(*ptr)) {
					printf("\7***Invalid entry\n");
					continue; }
				buffer[addr] = atox(ptr);
			case '+' :
			case 0 :
				++addr;
				continue;
			case '-' :
				--addr;
				continue;
			case 'V' :
				++ptr;
				switch(skip_blanks()) {
					case '1' : ascii = 0; break;
					case '2' : ascii = 1; break;
					case '3' : ascii = 2; }
				dump_block();
				continue;
			case 'Q' :
				if(yesno("Quit and THROW AWAY changes"))
					return;
				continue;
			case 'S' :
				if(!yesno("Exit and WRITE SECTOR (Are you SURE!!!)"))
					continue;
				if(write_disk())
					printf("Write error!\n");
				return;
			case 'G' :
				++ptr; skip_blanks();
				addr = atox(ptr);
				continue;
			case '?' :
				fputs(help2, stdout); } }
}

/*
 * Force a Yes/No answer
 */
yesno(char *prompt)
{
	char inline[50];
	for(;;) {
		printf("%s? ", prompt);
		fgets(ptr = inline, sizeof(inline)-1, stdin);
		switch(skip_blanks()) {
			case 'Y' :
			case 'y' :
				return -1;
			case 'N' :
			case 'n' :
				return 0; }
		printf("\7Please answer Y or N. "); }
}

/*
 * Get a High:Low number
 */
get_long(int *high, int *low)
{
	unsigned t, t1;

	t1 = *high;
	if(!isxdigit(skip_blanks())) goto bad;
	t = atox(ptr);
	while(isxdigit(*ptr)) ++ptr;
	if(skip_blanks() == ':') {
		++ptr;
		if(!isxdigit(skip_blanks())) goto bad;
		t1 = t;
		t = atox(ptr);
		while(isxdigit(*ptr)) ++ptr; }
	if(*ptr && !isspace(*ptr)) {
	bad:
		printf("\7***Invalid HEX number   ");
		return -1; }
	*high = t1;
	*low = t;
	return 0;
}

/*
 * Set search string from input string
 */
set_search(char *ptr)
{
	char c, d, *ptr1;

	slen = 0;
	while(c = *ptr++) {
		if(c == '^') {
			if(isalpha(c = *ptr++))
				c &= 0x1F;
			else if(isdigit(c)) {
				d = *(ptr1 = ptr+2);
				*(ptr1) = 0;
				c = atox(ptr-1);
				while(isxdigit(*ptr))
					++ptr;
				*(ptr1) = d; } }
		search[slen++] = c; }
}

/*
 * Display the search string
 */
show_search()
{
	int i;
	unsigned char c;

	for(i=0; i < slen; ++i) {
		if(isprint(c = search[i])) {
			putc(c, stdout);
			continue; }
		putc('^', stdout);
		if((c > ('A'-'@')) && (c <= ('Z'-'@'))) {
			putc(c+'@', stdout);
			continue; }
		printf("%03x", c); }
}
