/*
 * Network communication using the Crynwr packet drivers.
 *
 * This program implements a basic DIX ethernet interface using the Crynwr
 * packet driver collection, which is available from the Simtel archive
 * (oak.oakland.edu pub/msdos/pktdrvr). This collection contains drivers
 * for practically all network cards in existance... At the time of this
 * writing, the current packet driver collection is named: "PKTD11*.ZIP"
 *
 * The specification for this packet driver interface, including details
 * on the parameters used by these routines is documented in the Crynwr
 * collection, in a file called: "PACKET_D.109" (The three digit extension
 * signifies the specification version, and may change).
 *
 * This program is divided into two parts, the "PACKET DRIVER INTERFACE",
 * which is a set of generalized functions to communicate with the packet
 * driver, and a "DEMONSTRATION PROGRAM", which shows the use of the packet
 * driver interface.
 *
 * When you run the program, it will show the return codes from the various
 * setup functions, as well as the information returned, and then enter a
 * loop, displaying any packet of the defined type which is received by this
 * interface. As new network addresses are encountered in the "From address"
 * fields, they are added to an internal list. Pressing '0' will send a multi-
 * cast packet, which should be received by all nodes running this program.
 * Pressing '1-9' will send to the curresponding addresses from the internal
 * list, and should be received by that node only.
 * Pressing ESC will terminate the program.
 *
 * NOTE: in order for the program to be able to send to any node, at least
 * one packet has to have been received from that node (otherwise the nodes
 * address is not known). The easist way to accomplish this is to press '0'
 * on each node to send a multicast packet.
 *
 * If you define the symbol NEOS, the program will also monitor for packets
 * from the NEOS dos network (I used this in debugging).
 *
 * Copyright 1999-2003 Dave Dunfield.
 * This program may be used under the terms and conditions of the GNU General
 * Public License. Complete information on the GNU license is included in the
 * Crynwr packet driver collection, or contact: the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Compile command: cc packet -fop
 */

#include <stdio.h>

/*
 * -------------------- PACKET DRIVER INTERFACE --------------------
 */

/* PKT_INT must be in DECIMAL, as it is used by C and inline assembly */
#define	PKT_INT		96				/* Packet driver interrupt (0x60) */
#define	PKT_TYPE	0x1313			/* Type for our packets */
#define	PKT_DATA	100				/* Size of data area in packet */
#define	PKT_SIZE	PKT_DATA+14		/* Size of packet with overhead */
#define	PKT_EASIZE	6				/* Size of ethernet address */
/* PKT_WINDOW must be a power of 2, so we can mask with PKT_WINDOW-1 */
#define	PKT_WINDOW	16				/* Size of packet receive queue */

/*
 * Structure of information returned by pkt_driver_info()
 */
struct PD_driver_info {
	unsigned version;				/* Driver version */
	unsigned char class;			/* Interface class */
	unsigned type;					/* Interface type */
	unsigned char number;			/* Interface number */
	unsigned char functionality;	/* Driver functionality */
	unsigned char name[9]; };		/* Driver name (?) */

/*
 * Structure of packet as sent/received by driver
 */
struct PD_packet {
	unsigned char to_addr[PKT_EASIZE];
	unsigned char from_addr[PKT_EASIZE];
	unsigned ptype;
	unsigned char data[PKT_DATA]; };

/*
 * Structure of a packet receive queue entry
 */
struct PD_queue {
	unsigned handle;
	unsigned length;
	struct PD_packet packet; };

/*
 * Packet driver interface global variables
 */
struct PD_queue
	pkt_queue[PKT_WINDOW];			/* Queue for received packets */

unsigned
	pkt_head = 0,					/* Head of RX packet queue */
	pkt_tail = 0;					/* Tail of RX packet queue */

/*
 * Obtain packet driver information
 *
 * This function *MUST* be called before pkt_access_type() is called, in
 * order to insure that the SAVEDS location is initialized with the Micro-C
 * data segment. Another reason is that you need to obtain the interface
 * class/type/number so that you can pass them to pkt_access_type();
 *
 * Arguments:
 *	info_struct	= "PD_driver_info" structure to fill in.
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_driver_info(info_struct) asm
{
		MOV		CS:SAVEDS,DS		; Save out data seg
		MOV		AX,01ffh			; Request info
		INT		PKT_INT				; Ask packet driver
		JC		di2					; Error occured
		PUSH	DS					; Save driver DS
		MOV		DS,CS:SAVEDS		; Restore DATA segment
		POP		ES					; Point to driver data
		MOV		DI,4[BP]			; Get struct
		MOV		0[DI],BX			; Save version
		MOV		2[DI],CH			; Save Class
		MOV		3[DI],DX			; Save type
		MOV		5[DI],CL			; Save number
		MOV		6[DI],AL			; Save functionality
		LEA		BX,7[DI]			; Point to name
di1:	MOV		AL,ES:[SI]			; Get data
		MOV		[BX],AL				; Copy it
		INC		SI					; Advance source
		INC		DI					; Advance dest
		AND		AL,AL				; End of string?
		JNZ		di1					; Copy it all
		XOR		AX,AX				; Return 0
		JMP		di3
; Report driver error
di2:	MOV		AL,DH				; Get error code
		MOV		AH,0FFh				; Insure non-zero
di3:
}

/*
 * Obtain access to a packet type
 *
 * Arguments:
 *	if_class	= Interface class  \
 *	if_type		= Interface type    > - From pkt_driver_info()
 *	if_number	= Interface number /
 *	type		= Pointer to packet type to access
 *	typelen		= Length of packet type (0=Access ALL packets)
 *	handle		= Pointer to 'int' variable to receive handle
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_access_type(if_class, if_type, if_number, type, typelen, handle) asm
{
		MOV		AL,14[BP]			; Get if_class
		MOV		BX,12[BP]			; Get if_type
		MOV		DL,10[BP]			; Get if_number
		MOV		SI,8[BP]			; Get type pointer
		MOV		CX,6[BP]			; Get typelen
		MOV		DI,OFFSET pktrdy	; Point to handler
		PUSH	CS					; Save code segment
		POP		ES					; ES = handler segment
		MOV		AH,02h				; Indicate "access type" request
		INT		PKT_INT				; Call packet driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get handle
		MOV		[BX],AX				; Save handle
		XOR		AX,AX				; Indicate success
}

/*
 * Release access to a packet type
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_release_type(handle) asm
{
		MOV		BX,4[BP]			; Get handle
		MOV		AH,03h				; Indicate "release type" request
		INT		PKT_INT				; Ask for it
		JC		di2					; Indicate failed
		XOR		AX,AX				; Indicate success
}

/*
 * Send a packet
 *
 * Arguments:
 *	buffer		= Pointer to buffer to send
 *	length		= Length of buffer to send
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_send(buffer, length) asm
{
		MOV		SI,6[BP]			; Get buffer address
		MOV		CX,4[BP]			; Get length
		MOV		AH,04h				; Indicate "send packet" request
		INT		PKT_INT				; Ask for it
		JC		di2					; Report error
		XOR		AX,AX				; Indicate success
}

/*
 * Get our ethernet address
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	buffer		= Pointer to buffer to receive address
 *	length		= Size of buffer (maximum)
 *	alen		= Pointer to 'int' variable to receive size of address
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_get_address(handle, buffer, length, alen) asm
{
		MOV		BX,10[BP]			; Get handle
		MOV		DI,8[BP]			; Get buffer
		MOV		CX,6[BP]			; Get length
		PUSH	DS					; Save out segment
		POP		ES					; Set ES
		MOV		AH,06h				; Indicate "get address" request
		INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get received length
		MOV		[BX],CX				; Save length
		XOR		AX,AX				; Indicate success
}

/*
 * Set the receive mode for a packet
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	mode		=	1: Turn off receiver
 *					2: Receive only packets sent to our address
 *					3: mode 2 + broadcast packets
 *					4: mode 3 + limited multicast packets
 *					5: mode 3 + all multicast packets
 *					6: all packets
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_set_receive_mode(handle, mode) asm
{
		MOV		BX,6[BP]			; Get handle
		MOV		CX,4[BP]			; Get mode
		MOV		AH,20				; Set function
		INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		XOR		AX,AX				; Report success
}

/*
 * Receive packet handler. Called by packet driver when a packet
 * has been received for us.
 *
 * Check to see if we have enough space, and buffer the packet in our
 * receive packet queue, notify the mainline when received.
 */
asm {
; Entry:
;  AX = flag: 0=Allocate buffer, 1=Transfer complete
;  BX = handle
;  CX = Length
; Exit:
;  ES:DI = Buffer to receive packet (0:0=Not available)
pktrdy:	PUSH	DS					; Save our data segment
		MOV		DS,CS:SAVEDS		; Restore Micro-C DS
		AND		AX,AX				; Is this allocate?
		JNZ		pktr2				; No, try next
; Allocate space for a packet
		CMP		CX,PKT_SIZE			; Is packet too large?
		JA		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head of queue
		INC		AX					; Advance to next
		AND		AX,PKT_WINDOW-1		; Mask for wrap
		CMP		AX,DGRP:_pkt_tail	; Is queue full?
		JZ		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head back
		MOV		DI,PKT_SIZE+4		; Get size
		MUL		DI					; Compute offset
		ADD		AX,OFFSET DGRP:_pkt_queue ; Offset to location
		MOV		DI,AX				; Point to location
		MOV		[DI],BX				; Save handle
		MOV		2[DI],CX			; Save length
		ADD		DI,4				; Skip to data
		PUSH	DS					; Get DS
		POP		ES					; Copy to ES
		JMP SHORT pktrx				; And exit
; No space for buffer, or packet too large - reject packet
pktr1:	MOV		DI,0				; Indicate no buffer
		MOV		ES,DI				; ""
		JMP SHORT pktrx				; And exit
; Packet has been copied into our buffer, notify application
pktr2:	MOV		AX,DGRP:_pkt_head ; Get packet queue head
		INC		AX					; Skip to next
		AND		AX,PKT_WINDOW-1		; Wrap circular buffer
		MOV		DGRP:_pkt_head,AX ; Save new queue head
pktrx:	POP		DS					; Restore data seg
		DB		0CBh				; Far return
SAVEDS	DW		?
}



/*
 * -------------------- DEMONSTRATION PROGRAM --------------------
 */

#define	ADDRS		9				/* Number of addresses in our list */

unsigned char
	addr_list[ADDRS][PKT_EASIZE];	/* List of received addresses */

unsigned
	addr_top = 0;					/* Top of address list */

/*
 * Display an ethernet address address
 */
void show_address(unsigned char *address, unsigned alen)
{
	unsigned i;

	for(i=0; i < alen; ++i) {
		if(i) putc(':', stdout);
		printf("%02x", *address++); }
}

/*
 * Test for a new address, and if so, collect into address list
 */
void collect_address(unsigned char *address, unsigned alen)
{
	unsigned i;

	for(i=0; i < addr_top; ++i)
		if(!memcmp(addr_list[i], address, alen))
			return;
	printf("New address: ");
	show_address(address, alen);
	if(i >= ADDRS) {
		printf(" - List full!\n");
		return; }
	memcpy(addr_list[addr_top], address, alen);
	printf(" - Added at position: %u\n", ++addr_top);
}

/*
 * Dump a buffer of memory with the specified address & size
 */
void dump(unsigned char *buffer, unsigned size, unsigned addr)
{
	unsigned i, j, c;

	for(i=0; i < size; i += 16) {
		printf("%04x ", addr+i);
		for(j=0; j < 16; ++j) {						/* Display HEX */
			if(!(j & 0x03))
				putc(' ', stdout);
			if((i+j) < size) {
				printf("%02x ", buffer[j]); }
			else
				fputs("   ", stdout); }
		putc(' ', stdout);
		for(j=0; (j < 16) && ((i+j) < size); ++j) {	/* Display ASCII */
			c = *buffer++;
			putc(((c >= ' ') && (c < 0x7f)) ? c : '.', stdout); }
		putc('\n', stdout); }
}

/*
 * Main demonstration program
 */
main()
{
	unsigned ps, po;							/* Driver segment,offset */
	unsigned h1, h2, h3, h4;					/* Driver handles */
	unsigned char my_address[PKT_EASIZE];		/* Our ethernet address */
	unsigned alen;								/* Length of address */
	struct PD_driver_info info;					/* Info from driver */
	struct PD_packet send_buffer;				/* Packet to send */
	static char pkt_driver_id[] = {"PKT DRVR"};	/* Driver ID string */
	static unsigned packet_count = 0;			/* Count of packets sent */
	unsigned i, t;								/* Misc variables */

	/* Check that a packet driver is present */
	po = peekw(0, PKT_INT*4) + 3;	/* Offset to packet ID text */
	ps = peekw(0, (PKT_INT*4)+2);	/* Segment of packet driver */
	t = 0;
	do {
		if(pkt_driver_id[t] != peek(ps, po+t)) {
			printf("No packet driver at %04x:%04x\n", ps, po-3);
			exit(-1); } }
	while(pkt_driver_id[++t]);

	/* Obtain information from driver and display */
	printf("driver_info=%u ", pkt_driver_info(info));
	printf("ver=%u ", info.version);
	printf("Class/Type/Num=%02x/%04x/%02x ", info.class, info.type, info.number);
	printf("Func=%02x ", info.functionality);
	printf("Name='%s'\n", info.name);

	/* Obtain access to our packet type */
	t = PKT_TYPE;
	printf("access_type=%u ", pkt_access_type(info.class, info.type, 0, &t, sizeof(t), &h1));
	printf("Handle=%04x\n", h1);
#ifdef NEOS
	t = 0x1010;
	printf("access_type1=%u ", pkt_access_type(info.class, info.type, 0, &t, sizeof(t), &h2));
	printf("Handle=%04x\n", h2);
	t = 0x1111;
	printf("access_type2=%u ", pkt_access_type(info.class, info.type, 0, &t, sizeof(t), &h3));
	printf("Handle=%04x\n", h3);
	t = 0x1212;
	printf("access_type3=%u ", pkt_access_type(info.class, info.type, 0, &t, sizeof(t), &h4));
	printf("Handle=%04x\n", h4);
#endif

	/* Get our ethernet address & display */
	printf("get_address=%u ", pkt_get_address(h1, my_address, sizeof(my_address), &alen));
	printf("alen=%u Address=", alen);
	show_address(my_address, alen);

	printf("\nUse: '0' to send multicast, '1-9' to send to address list, ESC to exit.\n");

	/*
	 * Enter into a loop, monitoring the receive packet queue.
	 * If a packet is received, display it and remove from the queue.
	 * Also monitor keyboard, and exit if the ESC key is pressed.
	 * '0' causes transmission of a multicast packet.
	 * '1-9' transmits to collected addresses.
	 */
	memset(send_buffer, 0, sizeof(send_buffer));
	do {
		/* Check for received packets, and display */
		if(pkt_head != pkt_tail) {		/* Packet has been received */
			printf("Q[%u] Handle:%04x Len:%u, ", pkt_tail,
				pkt_queue[pkt_tail].handle, pkt_queue[pkt_tail].length);
			printf("To:");
			show_address(pkt_queue[pkt_tail].packet.to_addr, alen);
			printf(" Fr:");
			show_address(pkt_queue[pkt_tail].packet.from_addr, alen);
			printf(" Type:%04x\n", pkt_queue[pkt_tail].packet.ptype);
			dump(pkt_queue[pkt_tail].packet.data,
				pkt_queue[pkt_tail].length-(PKT_SIZE-PKT_DATA), 0);
			collect_address(pkt_queue[pkt_tail].packet.from_addr, alen);
			pkt_tail = (pkt_tail+1) & (PKT_WINDOW-1); }

		/* Check for transmit packets */
		if(isdigit(t = kbtst())) {		/* Transmit requested */
			if(!(t -= '0')) {			/* Multicast packet */
				for(i=0; i < alen; ++i)
					send_buffer.to_addr[i] = 0xFF; }
			else if(--t < addr_top)		/* From address list */
				memcpy(send_buffer.to_addr, addr_list[t], alen);
			else {
				printf("Address not defined!\n");
				continue; }
			printf("Sending %u bytes to ", sizeof(send_buffer));
			show_address(send_buffer.to_addr, alen);
			memcpy(send_buffer.from_addr, my_address, alen);
			send_buffer.ptype = PKT_TYPE;
			sprintf(send_buffer.data, "This is packet number %u ...", ++packet_count);
			pkt_send(send_buffer, sizeof(send_buffer));
			printf(" as packet #%u\n", packet_count); } }
	while(t != 0x1B);

	/*
	 * Release access types before terminating, to prevent the packet
	 * driver from calling into our code after it has been unloaded!!
	 */
#ifdef NEOS
	printf("release_type3=%u\n", pkt_release_type(h4));
	printf("release_type2=%u\n", pkt_release_type(h3));
	printf("release_type1=%u\n", pkt_release_type(h2));
#endif
	printf("release_type=%u\n", pkt_release_type(h1));
}
