/*
 * XCOPY command for MS-DOS
 *
 * This version of XCOPY is similar to the original MS-DOS XCOPY command
 * except that is has a number of additional options and features.
 *
 * Copyright 1998-2003 Dave Dunfield
 * Freely distributable.
 *
 * Compile command: cc xcopy -fop
 */
#include <stdio.h>
#include <file.h>
// #define PATHDEBUG			/* Debug path/pattern detection logic */

/* Fixed parameters */
#define	MAXPATH		80			/* Maximum size of a directory path */
#define ALLFILES	0x3F		/* findfirst mask for all files */
#define	MAXSTACK	1820		/* Maximum # files to stack */
#define	BUFFER_SIZE	16384		/* Size of copy buffer */

/* DOS return codes */
#define	RC_NOFILE	2			/* File not found */
#define	RC_NODIR	3			/* Directory not found */
#define	RC_NOMORE	18			/* No more files */

/*
 * Structure of file/directory information block
 */
struct FI_block {
	char		FI_attrib;			/* file Attributes */
	unsigned	FI_time;			/* file Time stamp */
	unsigned	FI_date;			/* file Date stamp */
	char		FI_name[13]; };		/* Name of file */

/*
 * Source/Dest path/pattern variables
 */
char
	source[MAXPATH+1],	/* Source path */
	*source_ptr,		/* Pointer to current level along source path */
	sourcefile[13],		/* Source file matching pattern */
	dest[MAXPATH+1],	/* Destination path */
	*dest_ptr,			/* Pointer to current level along dest path */
	destfile[13];		/* Dest file replacement pattern */

/*
 * Command line options
 */
char
	archive,			/* Copy files with archive: 1=leave, 2=reset */
	prompt,				/* Prompt before creating destination file */
	subdir,				/* Copy subdirectories: 1=No empty. 2=+empties */
	verify,				/* Verify each file after copy */
	hidden,				/* Copy hidden/system files */
	date,				/* Date specific copy: 1=After, 2=Before */
	tstamp,				/* Update timestamp */
	econt,				/* Continue even if errors */
	wait,				/* Wait for verification */
	message = -1		/* Show informational messages */

/*
 * Misc global variables
 */
unsigned
	rc,					/* Return code from DOS operations */
	file_count,			/* Count of files copied */
	xdate;				/* Date for date operations */

/*
 * Stack to hold processed file/dir names
 * Filenames build up from the bottom, dir names grow down from the top
 */
struct FI_block file_stack[MAXSTACK];
unsigned sfile, sdir = MAXSTACK;

/*
 * Help text
 */
char help[] = { "Copies files and directory trees.\n\n\
XCOPY source [destination] [options]\n\n\
 /A          Copy only files with archive attribute set, leaves set\n\
 /M          Copy only files with archive attribute set, clears archive\n\
 /Ddd/mm/yy  Copy files changed on or after this date (default: today)\n\
 /D-dd/mm/yy Copy files changed before or on this date\n\
 /H          Include Hidden & System files\n\
 /K          Keep going, even if errors occur\n\
 /L          Show long information (both source & dest)\n\
 /P          Prompt before copying each file\n\
 /Q          Quiet, disables unnecessary messages\n\
 /S          Copy directories and subdirectories except empty ones\n\
 /E          Copy all subdirectories, even if empty\n\
 /T          Copied files will have todays time/date\n\
 /V          Verify each file after copy\n\
 /W          Prompt to press a key before copying\n\n\
Copyright 1998-2003 Dave Dunfield - Freely distributable.\n" };

/*
 * Report an error
 */
error(char *string)
{
	char f;
	f = *string++;
	printf("Error : %s", string);
	if(f & 0x04) printf(", DosErrorCode=%u", rc);
	putc('\n', stdout);
	if(f & 0x01) stpcpy(source_ptr, sourcefile);
	if(f & 0x01) printf("\t%s\n", source);
	if(f & 0x02) printf("\t%s\n", dest);
	if((f & 0x08) || !econt) exit(rc);
}

/*
 * Parse an option string, allowing multiple options within one string
 */
parse_option(char *ptr)
{
	unsigned y, m, d;

	while(*ptr) {
		if(*ptr++ != '/')
			abort("Invalid switch");
		switch(toupper(*ptr++)) {
			case 'A' : archive = 1;	continue;
			case 'M' : archive = 2;	continue;
			case 'H' : hidden = -1;	continue;
			case 'K' : econt = -1;	continue;
			case 'L' : message = 1;	continue;
			case 'P' : prompt = -1;	continue;
			case 'S' : subdir |= 1;	continue;
			case 'E' : subdir |= 3;	continue;
			case 'Q' : message = 0;	continue;
			case 'T' : tstamp = -1; continue;
			case 'V' : verify = -1;	continue;
			case 'W' : wait = -1;	continue;
			case 'D' :
				get_date(&d, &m, &y);
				date = 1;
				if(*ptr == '-') { date = 2; ++ptr; }
				if(isdigit(*ptr)) {
					d = atoi(ptr);
					while(isdigit(*ptr)) ++ptr;
					if(*ptr++ != '/') abort("Invalid date");
					m = atoi(ptr);
					while(isdigit(*ptr)) ++ptr;
					if(*ptr++ != '/') abort("Invalid date");
					y = atoi(ptr);
					while(isdigit(*ptr)) ++ptr;
					if(y < 80)
						y += 2000;
					else if(y < 100)
						y += 1900; }
				xdate = ((y-1980) << 9) | (m << 5) | d;
				continue;
			case '?' : abort(help); }

		abort("Invalid switch"); }
}

/*
 * Test for a string containing wildcards
 */
iswild(char *p)
{
	while(*p) switch(*p++) {
		case '*' :
		case '?' :
			return -1; }
	return 0;
}

/*
 * Put an entry on the file information stack
 */
stack(unsigned entry, struct FF_block *f)
{
	struct FI_block *d;
	d = file_stack[entry];
	stpcpy(d->FI_name, f->FF_name);
	d->FI_attrib = f->FF_attrib;
	d->FI_time = f->FF_time;
	d->FI_date = f->FF_date;
}

/*
 * Match filename to source file matching pattern
 */
fmatch(char *f)
{
	char *p, c;
	p = sourcefile;
	for(;;) switch(c = *p++) {
		case '?' :		/* Match (skip) one character from filename */
			if((*f != '.') && *f)
				++f;
			break;
		case '*' :		/* Match (skip) remaining characters to . or end */
			if(!*p)
				return 1;
			while((*f != '.') && *f)
				++f;
			continue;
		case 0 :		/* End, match only if also end of filename */
			return !*f;
		case '.' :		/* Don't count '.' if it is at the end */
			if(!*f) continue;
		default:		/* Character in filename must match */
			if(c != *f++)
				return 0; }
}

/*
 * Build destination filename from a source name and the replacement pattern
 */
build_dest_name(char *d, char *s)
{
	char c, *p;
	p = destfile;
	while(c = *p++) switch(c) {
		case '?' :		/* Copy one character from source */
			if(*s)
				*d++ = *s++;
			continue;
		case '*' :		/* Copy up to end or '.' in source) */
			while(*s && (*s != '.'))
				*d++ = *s++;
			continue;
		default:		/* Replace char in dest with one from pattern */
			*d++ = c;
			if(*s) ++s; }
	*d = 0;
}

/*
 * Touch the file with encoded date
 */
touch(handle, time, date) asm
{
		MOV		DX,4[BP]	; Get date
		MOV		CX,6[BP]	; Get time
		MOV		BX,8[BP]	; Get file handle
		MOV		AX,5701h	; Set date & time function
		INT		21h			; Ask DOS
		JC		touch1		; Error, do not reset AX
		XOR		AX,AX		; Return 0
touch1:
}

/*
 * High speed compare of two block of memory
 */
mcompare(block1, block2, size)
asm {
		PUSH	DS				; Save DATA segment
		POP		ES				; Set EXTRA segment
		MOV		SI,8[BP]		; Get first string
		MOV		DI,6[BP]		; Get second string
		MOV		CX,4[BP]		; Get count
		XOR		AX,AX			; Assume success
	REPE CMPSB					; Do the compare
		JZ		cok				; Its ok
		DEC		AX				; Set -1 (fail)
cok:
}

/*
 * Copy one file, update timestamp and verify if requested
 */
copy_file(unsigned entry)
{
	HANDLE s, d;
	unsigned n;
	struct FI_block *fi;
	char buffer[BUFFER_SIZE], r;

	fi = file_stack[entry];
	stpcpy(source_ptr, fi->FI_name);
	build_dest_name(dest_ptr, fi->FI_name);
	if(prompt || message) {
		fputs(source, stdout);
		if(message == 1)
			printf(" -> %s", dest);
		if(prompt) {	/* If selected, prompt before copying */
			fputs("(Y/N)?", stdout);
	prmpt:
			switch(toupper(kbget())) {
				default: goto prmpt;
				case 0x03:
						fputs("^C", stdout);
				case -1:
					exit(-1);
				case 'N' : fputs("N\n", stdout); return;
				case 'Y' : fputs("Y\n", stdout); } }
		else
			putc('\n', stdout); }

	/* Open the source file */			
	if(!(s = open(source, F_READ))) {
		error("\x01Cannot read source");
		return; }

	/* Open the destination file */
	if(!(d = open(dest, F_WRITE))) {
		close(s);
		error("\x02Cannot write dest");
		return; }

	/* Copy all data from source to destination */
	do {
		if(n = read(buffer, sizeof(buffer), s))
			if(rc = write(buffer, n, d)) {
				close(d);
				close(s);
				error("\x06Write error on dest file");
				return; } }
	while(n == sizeof(buffer));

	/* If requested, update timestamp of new file to match old */
	rc = (tstamp) ? 0 : touch(d, fi->FI_time, fi->FI_date);

	close(s);
	close(d);

	if(rc)	/* Report any error in timestamp update */
		error("\x06Can't update dest time/date");

	/* Set attributes to match original file */
	if(rc = set_attr(dest, fi->FI_attrib))
		error("\x06Can't update dest attributes");

	/* If requested, reset archive bit of source file */
	if(archive == 2) {
		if(rc = set_attr(source, fi->FI_attrib & ~ARCHIVE))
			error("\x05Can't reset source ARCHIVE bit"); }

	++file_count;

	/* If requested, verify source & dest by read and compare */
	if(verify) {
		if(!(s = open(source, F_READ))) {
			error("\x01Cannot read source");
			return; }
		if(!(d = open(dest, F_READ))) {
			close(s);
			error("\x02Cannot read dest");
			return; }
		r = 0;
		do {
			n = read(buffer, BUFFER_SIZE/2, s);
			if(read(buffer+(BUFFER_SIZE/2), BUFFER_SIZE/2, d) != n) {
				r = -1;
				break; }
			if(n && mcompare(buffer, buffer+(BUFFER_SIZE/2), n)) {
				r = -1;
				break; } }
		while(n == (BUFFER_SIZE/2));
		close(d);
		close(s);
		if(r)
			error("\x03Verify failed"); }
}

/*
 * Copy a directory and subdirectories (if requested)
 */
copy_dir()
{
	int i;
	struct FF_block f;
	struct FI_block *fi;
	unsigned sdir_hold;
	char *s, *d;
	static unsigned level = 0;

	sdir_hold = sdir;
	stpcpy(source_ptr, "*.*");
	*dest_ptr = 0;
	sfile = 0;

	/*
	 * See of any files exist
	 */
	if(rc = findfirst(source, f, ALLFILES)) {
		if((rc != RC_NOFILE) && (rc != RC_NOMORE)) {
			error("\x05Unable to access source path");
			return; }
		if(!level)
			error("\x15File not found!"); }
	else do {
		if(f.FF_attrib & VOLUME) continue;
		if(*f.FF_name == '.') continue;
		if(f.FF_attrib & DIRECTORY) {
			if(subdir)
				stack(--sdir, f);
			continue; }
		if(!fmatch(f.FF_name)) continue;
		if(archive && !(f.FF_attrib & ARCHIVE)) continue;
		if((f.FF_attrib & (HIDDEN|SYSTEM)) && !hidden) continue;
		if(date) {
			if(date == 2) {
				if(f.FF_date > xdate) continue; }
			else {
				if(f.FF_date < xdate) continue; } }
		stack(sfile++, f); }
	while(!findnext(f));

	/* Not the first time, check on our directory */
	if(level) {
		/* Lookup our path... see if it exists */
		if(findfirst(dest, f, DIRECTORY)) {
			/* No files, unless enabled, do not make empty directory */
			if((sdir == sdir_hold) && (subdir == 1) && !sfile)
				return;
			/* Create the directory */
			if(rc = mkdir(dest)) {
				error("\x06Unable to make directory");
				return; } }
		*dest_ptr++ = '\\'; }
	/* Report an error if no matching files and not copying subdirs */
	else if((sdir == sdir_hold) && !sfile) {
		error("\x11File not found!");
		return; }

	/* Copy all the files that we found */
	for(i=0; i < sfile; ++i)
		copy_file(i);

	/* Copy subdirectories by recursive call for each directory */
	i = sdir_hold;
	s = source_ptr;
	d = dest_ptr;
	++level;
	while(i > sdir) {
		fi = file_stack[--i];
		source_ptr = stpcpy(stpcpy(s, fi->FI_name), "\\");
		dest_ptr = stpcpy(d, fi->FI_name);
		copy_dir(); }
	--level;
	dest_ptr = d;
	source_ptr = s;
	sdir = sdir_hold;
}

/*
 * Main program, parse arguments, determine paths & patterns, and COPY!
 */
main(int argc, char *argv[])
{
	unsigned i;
	struct FF_block f;

	for(i=1; i < argc; ++i) {
		if(*argv[i] == '/')
			parse_option(argv[i]);
		else if(!source_ptr)
			source_ptr = stpcpy(source, argv[i]);
		else if(!dest_ptr)
			dest_ptr = stpcpy(dest, argv[i]);
		else
			abort("Too many parameters"); }
	
	if(!source_ptr)
		abort(help);
	if(!dest_ptr)
	*(dest_ptr = dest) = 0;

	/* Insure all filenames are uppercase so that patterns etc. match */
	strupr(source);
	strupr(dest);

	/*
	 * Determine source path and file matching pattern
	 */
	switch(*(source_ptr-1)) {
		default:		/* Unknown, lets look for it */
			/* If source path contains wildcards, it cannot be a directory */
			if(iswild(source))
				break;
			/* Lookup file and determine what it is */
			if(findfirst(source, f, ALLFILES))
				break;
			/* Must be a file, as it is not a directry */
			if(!(f.FF_attrib & DIRECTORY))
				break;
			*source_ptr++ = '\\';	/* Append trailing '\' to directory */
		case '\\' :		/* Ends with '\', must be a directory */
		case ':' :		/* Ends with ':', must be a directory */
			stpcpy(source_ptr, "*.*"); }
	/* Pull off last entry as file match pattern */
	while(source_ptr > source) {
		switch(*(source_ptr-1)) {
			case '\\' :		/* We found last entry, copy to pattern */
			case ':' :		/* Then truncate to end at '\' or ':' */
				stpcpy(sourcefile, source_ptr);
				*source_ptr = 0;
				goto source_ok; }
		--source_ptr; }
	/* This was the only entry, assume current dir & argument is pattern */
	stpcpy(sourcefile, source);
	*source = 0;
source_ok:

#ifdef PATHDEBUG
	printf("Source: %s\n", source);
	printf("Sourcefile: %s\n", sourcefile);
#endif

	/*
	 * Determine destination path and file replacement pattern
	 */
	stpcpy(destfile, "*.*");		/* Assume replace with original name */
	if(!*dest) goto dest_ok;		/* No dest, copy to current dir */
	switch(*(dest_ptr-1)) {			/* If ends with '\' or ':', its a dir */
		case '\\' :					/* Strip trailing '\' if not root */
			if(((dest_ptr-1) != dest) && (*(dest_ptr-2) != ':'))
				*--dest_ptr = 0;
		case ':' :
			goto dest_ok; }
	/* If destination contains wildcards, it could be a directory */
	if(!iswild(dest)) {
		if(!findfirst(dest, f, ALLFILES)) {
			if(f.FF_attrib & DIRECTORY)		/* It's a dir, OK */
				goto dest_ok; }
		else {		/* Does not exist, prompt for file/dir */
			printf("Is: %s", dest);
			printf("\na file or directory (F=File, D=Directory)?");
		reprompt:
			switch(kbget()) {
				default: goto reprompt;
				case 0x03 : fputs("^C", stdout);
				case -1 : exit(-1);
				case 'D' : /* If a directory, no further processing */
				case 'd' : fputs("D\n", stdout); goto dest_ok;
				case 'F' : /* If a file, continue */
				case 'f' : fputs("F\n", stdout); } } }
	/* Pull off last entry as destination replacement pattern */
	while(dest_ptr > dest) {
		switch(*(dest_ptr-1)) {
			case '\\' :
				stpcpy(destfile, dest_ptr);
				*--dest_ptr = 0;
				goto dest_ok;
			case ':' :
				stpcpy(destfile, dest_ptr);
				*dest_ptr = 0;
				goto dest_ok; }
		--dest_ptr; }
	/* This was last entry, assume current dir & argument is pattern */
	stpcpy(destfile, dest);
	*dest = 0;
dest_ok:

	/*
	 * A destination path has been specified
	 * Insure it exists, create if necessary.
	 */
	if(*dest) {
		if(!findfirst(dest, f, ALLFILES)) {
			if(!(f.FF_attrib & DIRECTORY))	/* Exists, but not a dir */
				error("\x0ADestination path is not a directory"); }
		else switch(*(dest_ptr-1)) {		/* Does not exist */
			default:
				if(mkdir(dest))				/* Could not create */
					error("\x0AUnable to create directory");
			case '\\' :
			case ':' : }
		/* Append trailing '\' if needed */
		switch(*(dest_ptr-1)) {
			default:
				*dest_ptr++ = '\\';
			case '\\' :
			case ':' : } }

#ifdef PATHDEBUG
	printf("Dest: %s\n", dest);
	printf("Destfile: %s\n", destfile);
	return;
#endif

	if(wait) {
		fputs("Press any key to begin...", stdout);
		switch(kbget()) {
			case 3 :
				fputs("^C", stdout);
			case -1 :
				exit(-1); }
		putc('\n', stdout); }

	copy_dir();

	if(message)
		printf("%10u file(s) copied.\n", file_count);
}
