/*
 * LRG SPRITE DEMO: Moving animated "choppers"
 *
 * Not really a game yet, but this is a good place to start. This demo
 * shows the use of multiple animated sprites.
 *
 * Press LEFT/RIGHT/UP/DOWN to accelerate the active helicopter in the
 * corresponding direction. Press SPACE to let go of active helicopter,
 * and start a new one on the helipad. Press ESC to exit.
 *
 * Copyright 1998-2003 Dave Dunfield
 * All rights reserved.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc CHOPPERS -pof
 */
#include <lrg.h>

#define	COPTERS		10			/* Maximum number of choppers */
#define	XLIMIT		300			/* X limit of chopper movement */
#define	YLIMIT		185			/* Y limit of chipper movement */

#define _KUA		0x4800		/* Up    arrow	*/
#define _KDA		0x5000		/* Down  arrow	*/
#define _KLA		0x4B00		/* Left  arrow	*/
#define _KRA		0x4D00		/* Right arrow	*/

/*
 * Helicopter sprites
 */
char heli1[] = { 20, 11,	/* Facing right, blade to right */
  0,  0,  0,  0,  0,  0,  0,  0, 19, 22, 24,  7,  7,  7,  7,  7,  7,  7,  7,
  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10,  2,  0,
  0,  0,  0, 10,  2,  0,  0,  0,  0,  2, 10, 10,  9,  9,  9,  9,  9, 10, 10,
  2,  0,  0,  0, 10, 10,  2,  0,  0,  2, 10, 10, 10, 10,  1,  1,  1,  1,  1,
 10, 10,  2,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  0,  0,  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,  0,
  0,  0,  0, 12,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,
  0,  0,  0,  0, 12,  0,  0,  4, 12,  0,  0,  0,  0,  0,  0, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12, 12, 12, 12,  4,  0 };

char heli2[] = { 20, 11,	/* Facing right, blade to left */
  0,  0,  0,  7,  7,  7,  7,  7,  7,  7,  7,  7, 24, 22, 19,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10,  2,  0,
  0,  0,  0, 10,  2,  0,  0,  0,  0,  2, 10, 10,  9,  9,  9,  9,  9, 10, 10,
  2,  0,  0,  0, 10, 10,  2,  0,  0,  2, 10, 10, 10, 10,  1,  1,  1,  1,  1,
 10, 10,  2,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  0,  0,  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,  0,
  0,  0,  0, 12,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,
  0,  0,  0,  0, 12,  0,  0,  4, 12,  0,  0,  0,  0,  0,  0, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12, 12, 12, 12,  4,  0 };

char heli3[] = { 20, 11,	/* Facing left, blade right */
  0,  0,  0,  0,  0, 19, 22, 24,  7,  7,  7,  7,  7,  7,  7,  7,  7,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  2, 10, 10,  9,  9,  9,  9,  9, 10, 10,  2,  0,  0,
  0,  0,  2, 10,  0,  0,  2, 10, 10,  1,  1,  1,  1,  1, 10, 10, 10, 10,  2,
  0,  0,  2, 10, 10,  0,  0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  2,  0,  0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  2,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0,
 12,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  4,  0,  0, 12,  0,  0,  0,  0,
  0, 12,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4, 12, 12, 12, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12,  0,  0,  0,  0,  0 };

char heli4[] = { 20, 11,	/* Facing left, blade left */
  7,  7,  7,  7,  7,  7,  7,  7,  7, 24, 22, 19,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  2, 10, 10,  9,  9,  9,  9,  9, 10, 10,  2,  0,  0,
  0,  0,  2, 10,  0,  0,  2, 10, 10,  1,  1,  1,  1,  1, 10, 10, 10, 10,  2,
  0,  0,  2, 10, 10,  0,  0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  2,  0,  0, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10, 10, 10, 10,  2,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10,
 10, 10,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0,
 12,  0,  0,  0,  0,  0,  0,  0,  0,  0, 12,  4,  0,  0, 12,  0,  0,  0,  0,
  0, 12,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4, 12, 12, 12, 12, 12, 12, 12,
 12, 12, 12, 12, 12, 12,  0,  0,  0,  0,  0 };

char heli5[] = { 19, 11,	/* Frontal view, blade right */
  0,  0,  0,  0,  0,  0, 19, 22, 24,  7,  7,  7,  7,  7,  7,  7,  7,  7,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  2, 10, 10,  9,  9,  9,  9,  9, 10, 10,  2,  0,  0,  0,  0,
  0,  0,  0,  2, 10, 10,  1,  1,  1,  1,  1,  1,  1, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0, 12, 12, 12,  0,  0,  0, 12, 12, 12,  0,  0,  0,  0,  0 };

char heli6[] = { 19, 11,	/* Frontal view, blade left */
  0,  7,  7,  7,  7,  7,  7,  7,  7,  7, 24, 22, 19,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  2, 10, 10,  9,  9,  9,  9,  9, 10, 10,  2,  0,  0,  0,  0,
  0,  0,  0,  2, 10, 10,  1,  1,  1,  1,  1,  1,  1, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  2, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10,  2,  0,  0,  0,
  0,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0, 12,  0,  0,  0,  0,  0,  0,
  0,  0,  0,  0,  0, 12, 12, 12,  0,  0,  0, 12, 12, 12,  0,  0,  0,  0,  0 };

char star[] = { 11, 9,
  0,  0,  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  7,  0,  0,  7,  0,  0,
  7,  0,  0,  0,  0,  0,  7,  0,  7,  0,  7,  0,  0,  0,  0,  0,  0,  0,  7,
  7,  7,  0,  0,  0,  0,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  7,  0,  0,
  0,  0,  7,  7,  7,  0,  0,  0,  0,  0,  0,  0,  7,  0,  7,  0,  7,  0,  0,
  0,  0,  0,  7,  0,  0,  7,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  7,  0,
  0,  0,  0,  0 };

/*
 * Structure to represent each individual helicopter
 */
struct helicopter {
	int x, y;						/* Current X and Y position */
	int ox, oy;						/* Last X and Y position (for erase) */
	int hdir, vdir;					/* Current Horiz and Vert directions */
	char *s1, *s2;					/* Current sprites for animation */
	char *os;						/* Last drawn sprite (for erase) */
	char save[sizeof(heli1)-2]; };	/* Save area for sprites */

unsigned tick;						/* Tick's once per iteration */

struct helicopter copters[COPTERS];	/* Storage for the choppers */
unsigned ncopters = 0;				/* Count of active choppers */

extern int RAND_SEED;

/*
 * Draw a background
 */
draw_background()
{
	unsigned x, w, h;
	char array[3];
	static char pad[] = { 4,  14, 0,  5, 17,  -24, 0,  5, -17};

	/* Define some custom colors for night sky shading */
	for(x=0; x < 20; ++x) {
		array[0]=array[1] = x/2;
		array[2] = x;
		lrg_setpal(200+x, array); }

	/* Color the night sky */
	w = 220;
	for(h=0; h < 200; h += 10)
		lrg_fbox(0, h, 320, 10, --w);

	/* Randomize the random number generator */
	get_time(&x, &w, &h);
	RAND_SEED = (w*60)+h;

	/* Draw some buildings */
	x = 0;
	for(;;) {
		w = random(5) + 5;
		h = random(10)+10;
		if((x+w) >= 320)
			break;
		lrg_box(x, 180-h, w, h, RED);
		if(random(10) < 4)
			x += random(4)+2;
		x += w; }

	/* Draw the horizon */
	lrg_hline(0, 180, 320, BROWN);

	/* Draw the moon */
	lrg_fcircle(280, 145, 15, B_YELLOW);
	lrg_fcircle(275, 145, 15, 204);

	/* Draw some stars */
	x = 5;
	while((x += (random(20)+12)) < (320-11))
		lrg_draw(x, 100-random(95), star, copters);

	/* Draw the helipad */
	lrg_polygon(150, 182, B_YELLOW, pad);
	lrg_line(153, 184, 153-3, 184+12, B_WHITE);
	lrg_line(153+8, 184, 153+11, 184+12, B_WHITE);
	lrg_line(153-1, 184+6, 153+9, 184+6, B_WHITE);
}

/*
 * Create a new chopper
 */
struct helicopter *create_copter()
{
	struct helicopter *h;

	h = copters[ncopters++];		/* Allocate a structure area */

	h->x = 148;						/* Chopper on helipad (Horz) */
	h->y = YLIMIT;					/* Chopper on helipad (Vert) */
	h->vdir = h->hdir = 2;			/* Stopped (no motion) */
	h->s1 = heli5;					/* Frontal view */
	h->s2 = heli6;					/* Frontal view */

	return h;
}

main()
{
	unsigned i;

	struct helicopter *h, *new;

	lrg_open();

	draw_background();

	new = 0;					/* Force creation of initial chopper */

	for(;;) {
		if(!new)				/* New chopper requested */
			new = create_copter();

		++tick;					/* update tick count */

		/* Draw all active choppers */
		for(i=0; i < ncopters; ++i) {
			h = copters[i];
			lrg_draw(h->ox = h->x, h->oy = h->y, (h->os = (tick & 1) ?
				h->s1 : h->s2), h->save); }

		/* update X/Y co-ordinates of each chopper */
		for(i=0; i < ncopters; ++i) {
			h = copters[i];
			switch(h->vdir) {
				case 0 :				/* Fast up */
					if(h->y > 1) {
						h->y -= 2;
						break; }
					h->vdir = 4;
					break;
				case 1 :				/* Slow up */
					if(h->y) {
						--h->y;
						break; }
					h->vdir = 3;
					break;
				case 3 :				/* Slow down */
					if(h->y < YLIMIT) {
						++h->y;
						break; }
					h->vdir = 1;
					break;
				case 4 :				/* Fast down */
					if(h->y < (YLIMIT-1)) {
						h->y += 2;
						break; }
					h->vdir = 0; }
			switch(h->hdir) {
				case 0 :				/* Fast left */
					h->s1 = heli3; h->s2=heli4;
					if(h->x > 1) {
						h->x -= 2;
						break; }
					h->hdir = 4;
					break;
				case 1 :				/* Slow left */
					h->s1 = heli3; h->s2=heli4;
					if(h->x) {
						--h->x;
						break; }
					h->hdir = 3;
					break;
				case 2 :				/* Stopped */
					h->s1 = heli5; h->s2=heli6;
					break;
				case 3 :				/* Slow right */
					h->s1 = heli1; h->s2=heli2;
					if(h->x < XLIMIT) {
						++h->x;
						break; }
					h->hdir = 1;
					break;
				case 4 :				/* Fast right */
					h->s1 = heli1; h->s2=heli2;
					if(h->x < (XLIMIT-1)) {
						h->x += 2;
						break; }
					h->hdir = 0; } }

		/* Handle keyboard input */
		switch(kbtst()) {
			case _KUA :					/* Accelerate chopper up */
				if(new->vdir)
					--new->vdir;
				break;
			case _KDA :					/* Accelerate chopper down */
				if(new->vdir < 4)
					++new->vdir;
				break;
			case _KLA :					/* Accelerate chopper left */
				if(new->hdir)
					--new->hdir;
				break;
			case _KRA :					/* Accelerate chopper right */
				if(h->hdir < 4)
					++new->hdir;
				break;
			case ' ' :					/* Request new chopper */
				if(ncopters < COPTERS)
					new = 0;
				break;
			case 0x1B :					/* End program */
				lrg_close();
				return; }

		/* Slow pace to see animations, also align on vertical retrace */
		lrg_delay(3);

		/* Erase choppers in reverse order so as to not corrupt screen */
		i = ncopters;
		do {
			h = copters[--i];
		lrg_erase(h->ox, h->oy, h->os, h->save); }
		while(i); }
}
