/*
 * Demo of 3->4 Binary->ASCII encoding
 *
 * This program encodes a binary file into printable ASCII characters, in
 * such a way that it can be decoded back into the original binary file.
 * In doing so, it increases in size by 25% (3 bytes become four), however
 * it can then be transmitted through text only channels such as email.
 *
 * Compile command: cc 3to4 -fop
 */
#include <stdio.h>

int
	cn,				/* Number of pending characters to write/read */
	c1,				/* First pending character */
	c2,				/* Second pending character */
	c3,				/* Third (and last) pending character */
	ci,				/* High bits prefix byte (used for read only) */
	outcount;		/* Output position counter for newline insertion */

unsigned
	width = 70;		/* Selectable width for output file */

FILE
	*fpi,			/* Input file pointer */
	*fpo;			/* Output file pointer */

/*
 * 6-bit binary->ASCII translation table, index this table with a number
 * in the range of 0-3F, and output the corresponding character.
 */
unsigned char otable[] =		/* Output translate table */
	{ "abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ.," };


/*
 * Check for output width exceeding the selected output file width,
 * and output a newline character if necessary.
 */
void xlimit(void)
{
	if(++outcount >= width) {
		putc('\n', fpo);
		outcount = 0; }
}

/*
 * Write any pending data characters into the ASCII output stream
 * Three bytes of data are written as four ASCII characters.
 * The ASCII characters are one of 64 possible values, representing 6
 * bit of information with each character.
 *
 * The first 6 bits written contain:
 *  - High 2 bits of first  data byte
 *  - High 2 bits of second data byte
 *  - High 2 bits of third  data byte
 * Following this, the lower 6 bits of each of the data bytes are
 * written in order (byte1, byte2, byte3)
 *
 * If less then three bytes are available (as indicated by CN), this
 * must be the last data written to the file, in this case, the prefix
 * data for the values which are not present will be "don't care".
 */
void flush_write(void)
{
	if(cn) {
		/* Write prefix character */
		putc(otable[(c1&0xC0)>>2 | ((c2&0xC0)>>4) | ((c3&0xC0)>>6)], fpo);
		xlimit();
		putc(otable[c1 & 0x3f], fpo);
		xlimit(); }
	if(cn > 1) {
		putc(otable[c2 & 0x3F], fpo);
		xlimit();
		if(cn > 2) {
			putc(otable[c3 & 0x3F], fpo);
			xlimit(); } }

	cn = c1 = c2 = c3 = 0;
}

/*
 * Flush read characters into output file
 * - This is essentially the reverse of the above
 */
void flush_read(void)
{
	if(cn > 1) {
		putc(((ci << 2) & 0xC0) | c1, fpo);
		if(cn > 2) {
			putc(((ci << 4) & 0xC0) | c2, fpo);
			if(cn > 3)
				putc(((ci << 6) & 0xC0) | c3, fpo); } }
	cn = 0;
}

/*
 * Main program, open a file, read it and write the encoded characters.
 */
main(int argc, char *argv[])
{
	int c;
	unsigned char d, itable[256];
	static char help[] = { "\nUse: 3to4 Encode|Decode infile outfile\n" };

	if(argc < 4)
		abort(help);

	/* Perform Encode/Decode based on command argument */
	switch(*argv[1]) {
	case 'e' :		/* Encoding a file Binary->ASCII */
	case 'E' :
		fpi = fopen(argv[2], "rvqb");
		fpo = fopen(argv[3], "wvq");
		/* Read binary data and flush every three characters */
		while((c = getc(fpi)) >= 0) switch(++cn) {
			case 1 : c1 = c;	break;				/* Buffer 1st character */
			case 2 : c2 = c;	break;				/* Buffer 2nd character */
			case 3 : c3 = c;	flush_write(); }	/* Buffer 3rd and write */
		flush_write();
		if(outcount)
			putc('\n', fpo);
		break;
	case 'd' :		/* Decoding a file ASCII->Binary */
	case 'D' :
		fpi = fopen(argv[2], "rvq");
		fpo = fopen(argv[3], "wvqb");
		/* First, create inverse of output table to avoid lookup scan */
		memset(itable, -1, sizeof(itable));		/* Mark all chars bad */
		for(c=0; c < 64; ++c)					/* Then fill in good ones */
			itable[otable[c]] = c;
		/* Then read 4 character pairs and flush to output file */
		while((c = getc(fpi)) >= 0) {
			if(c <= ' ')			/* Ignore space/control */
				continue;
			if((d = itable[c]) > 63) {	/* Check for invalid char */
				fprintf(stderr, "Bad character '%c'[%02x]\n", c, c);
				continue; }
			switch(cn++) {
			case 0 : ci = d;	break;				/* Get index byte */
			case 1 : c1 = d;	break;				/* Get 1st character */
			case 2 : c2 = d;	break;				/* Get 2nd character */
			case 3 : c3 = d;	flush_read(); }	}	/* Get 3rd and write */
		flush_read();
		break;	
	default:
		abort(help); }

	fclose(fpo);
	fclose(fpi);
}
