/*
 * Program to demonstrate the MICRO-C/PC WINDOWING library.
 *
 * Copyright 1989-2003 Dave Dunfield
 * All rights reserved.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc windemo -fop
 */
#include <stdio.h>		/* Standard I/O definitions */
#include <window.h>		/* Window   I/O definitions */

/* Color index's for different window types */
#define	MENU	0
#define	FORM	1
#define	TEXT	2
#define	PAUSE	3

/* Pointer to active color selections, and default values */
unsigned *color,
	m_colors[] = { 0x70, 0x07, 0x70, 0x07 },	/* Monochrome defaults */
	c_colors[] = { 0x21, 0x17, 0x67, 0x42 };	/* Color defaults */

/* Pointer for access to title window */
struct WINDOW *titlewin;

/* Introductory pre-ramble */
char *intro[] = {
	"",
	"   This program demonstrates some of the capabilities of the MICRO-C",
	"PC windowing libary. This package allows you to produce nice looking",
	"DOS applications with little effort. When coupled with MICRO-C's TSR",
	"function, even TSR's become simple to write... The windowing library",
	"automatically saves and restores the interrupted programs video.",
	"",
	"Enjoy MICRO-C...",
	"",
	"   Dave Dunfield",
	"",
	"Press any key to proceed...",
	0 };

/* Main selection menu */
char *main_menu[] = {
	"Library functions",
	"Window creation",
	"Window scrolling",
	"Concurrent windows",
	"Form entry",
	"OPEN attributes",
	"Set window colors",
	"Clear main window",
	0 };

/* Colors selection menu */
char *menu1[] = {
	"Menu window colors",
	"Form window colors",
	"Text window colors",
	"Pause window colors",
	0 };

/* Available colors */
char *menu2[] = {
	"Black",
	"Blue",
	"Green",
	"Cyan",
	"Red",
	"Magenta",
	"Brown",
	"White",
	0 };

/* Sample input form */
char *form[] = {
	50<<8|6,
	"\x01\x00\x20Software  :",
	"\x01\x01\x20Author    :",
	"\x01\x02\x20Directory :",
	"\x01\x03\x20Filename  :",
	0 };

/* Data areas for input form */
char software[0x21]	= "MICRO-C",
	author[0x21]	= "Dave Dunfield",
	directory[0x21]	= "C:\\MC",
	filename[0x21]	= "MC*.*";

/* Help text for form entry window */
char *ftext[] = {
	"Use ARROW keys to move around form",
	"HOME/END moves to START/END of line",
	"PgUp clears entire line, PgDn clears to end",
	"INSERT toggles insert ON/OFF (Observe cursor)",
	"DELETE deletes the character under cursor",
	"BACKSPACE deletes character preceeding cursor",
	"Press ESCAPE to return to the main menu",
	0 }
	
/* Window attribute bits & corresponding defaults */
char *atext[] = {
	"SAVE/RESTORE screen under window",
	"Enable SINGLE LINE border",
	"Enable DOUBLE LINE border",
	"CLEAR window on entry",
	"CLEAR window on exit",
	"NEWLINE(0x0A) = LF only",
	"Enable SCROLLING",
	"Enable LINE-WRAP" };
char wattrs[8] = { 1, 1, 0, 1, 0, 0, 1, 1 };

/* Help text for attribute checkout window */
char *awtext[] = {
	"Use ARROW keys to move around",
	"HOME = Clear entire window",
	"PgUp = Clear to end of window",
	"PgDn = Clear to end of line",
	"ENTER = Newline character",
	"ESCAPE = Return to previous menu",
	0 };

/* Rambling about basic window functions (screen 1) */
char *basic1[] = {		/* Screen 1 */
	"",
	"The basic window functions (written in assembler) are:",
	"",
	"   wopen        - Open a new window",
	"   wclose       - Close active window",
	"   wputc        - Write character to active window",
	"   wclwin       - Clear active window",
	"   wcleow       - Clear to end of active window",
	"   wcleol       - Clear to end of line in active window",
	"   w_close      - Close any window",
	"   w_putc       - Write character to any window",
	"   w_clwin      - Clear any window",
	"   w_cleow      - Clear to end of any window",
	"   w_cleol      - Clear to end of line in any window",
	"",
	"Press any key to proceed...",
	0 };

char *basic2[] = {		/* Screen 2 */
	"",
	"More basic functions:",
	"",
	"   wgotoxy      - Position cursor in active window",
	"   wupdatexy    - Update cursor position in current window",
	"   wgetc        - Get character with cursor in active window",
	"   wtstc        - Test for character with cursor in active window",
	"   w_gotoxy     - Position cursor in any window",
	"   w_updatexy   - Update cursor position in any window",
	"   w_getc       - Get character with cursor in any window",
	"   w_tstc       - Test for character with cursor in any window",
	"   wcursor_off  - Turn cursor off *",
	"   wcursor_line - Turn cursor on as a line *",
	"   wcursor_block- Turn cursor on as a block *",
	"",
	"* The cursor is saved/restored when windows are opened/closed",
	"",
	"Press any key to proceed...",
	0 };

/* Rambling about supplementary video functions */
char *supp1[] = {
	"",
	"The supplementary video functions (written in 'C' are):",
	"",
	"   wputs        - Write a string to active window",
	"   wputf        - Write a string in a field to active window",
	"   wprintf      - Perform 'printf' to active window",
	"   wgets        - Get a string (with editing) in active window",
	"   wmenu        - Perform a multiple selection menu in window",
	"   wform        - Display/update an input form in window",
	"   w_puts       - Write a string to any window",
	"   w_printf     - Perform 'printf' to any window",
	"",
	"Press any key to proceed...",
	0 };

/*
 * Main function, present main menu & execute sub-functions
 */
main()
{
	int m1;

	m1 = 0;

	titlewin = wopen(0, 0, 80, 3, WBOX2|WCOPEN|0x47);
	color = (W_BASE == 0xB000) ? m_colors : c_colors;
	wopen(0,3,80,22,WSAVE|WBOX1|WCOPEN|0x30);
	wcursor_off();
	title("MICRO-C Windowing Demonstration");
	help(intro);
	wgetc();
	wclwin();
	for(;;) {
		title("MICRO-C Window Library Demonstration - ESC to exit");
		if(wmenu(12, 5, WBOX1|WCOPEN|color[MENU], main_menu, &m1))
			break;
		switch(m1) {
			case 0 :		/* Library function info */
				help(basic1);
				wgetc();
				help(basic2);
				wgetc();
				help(supp1);
				wgetc();
				break;
			case 1 :		/* Window creation demo */
				create_demo();
				break;
			case 2 :		/* Scrolling window demo */
				scroll_demo();
				break;
			case 3 :		/* Concurrent window demo */
				simul_demo();
				break;
			case 4 :		/* Form entry demo */
				form_demo();
				break;
			case 5 :		/* Window attribute demo */
				attr_demo();
				break;
			case 6 :		/* Set window colors */
				set_colors();
				break;
			case 7 :		/* Clear main window */
				wclwin();
				break; } }
	title("Demonstration Complete");
	wclose();
	wclose();
}

/*
 * Draw a title on the title window
 */
title(string)
	char *string;
{
	w_clwin(titlewin);
	w_gotoxy((78/2) - (strlen(string)/2), 0, titlewin);
	w_puts(string, titlewin);
}

/*
 * Pause for a key to be pressed
 */
pause()
{
	wopen(22, 12, 37, 5, WSAVE|WBOX2|WCOPEN|color[PAUSE]);
	wcursor_off();
	wgotoxy(5, 1);
	wputs("Press any key to proceed...");
	wgetc();
	wclose();
}

/*
 * Display help text on the screen
 */
help(text)
	char *text[];
{
	int i;
	char *ptr;

	wclwin();
	i = 0;
	while(ptr = text[i]) {
		wgotoxy(0, i++);
		wputs(ptr); }
}

/*
 * Set the window colors
 */
set_colors()
{
	int c, f, b;
	char buffer[50];

	c = 0;
	wopen(15, 9, 50, 11, WSAVE|WBOX1|WCOPEN|color[TEXT]);
	wcursor_off();
	wputs("Use PgUp and PgDn to select window\n\n");
	wputs("Up and Down arrows to select FOREGROUND color\n");
	wputs("Left and Right arrows to select BACKGROUND color\n");
	wputs("Press ESCAPE to exit this function");
	for(;;) {
		sprintf(buffer, "COLOR selection menu: %s", menu1[c]);
		title(buffer);
		b = color[c] >> 4;
		f = color[c] & 0x0F;
		wgotoxy(0, 6);
		wprintf("Foreground: %-15s", menu2[f]);
		wgotoxy(0, 8);
		wprintf("Background: %-15s", menu2[b]);
		wopen(45, 16, 15, 3, WBOX3|WCOPEN|((b<<4)+f));
		wputs("Sample window");
		wclose();
		switch(wgetc()) {
			case _KPU :
				c = (c > 0) ? c-1 : 3;
				break;
			case _KPD :
				c = (c+1) % 4;
				break;
			case 0x1B:		/* Escape key */
				wclose();
				return;
			case _KUA :
				f = (f > 0) ? f-1 : 7;
				goto newc;
			case _KDA :
				f = (f+1) % 8;
				goto newc;
			case _KLA :
				b = (b > 0) ? b-1 : 7;
				goto newc;
			case _KRA :
				b = (b+1) % 8;
			newc:
				color[c] = (b<<4) + f; } }
}

/*
 * Demo of Fast window creation
 */
create_demo()
{
	int i, j, k, l, x, y;

	title("Fast CREATION - 1000 Windows");
	for(i=0; i < 1000; ++i) {
		k = random(76 - (x = random(60)+1))+3;
		l = random(22 - (y = random(17)+4))+3;
		j = color[i%4];
		wopen(x, y, k, l, WBOX1|WCOPEN|j);
		wclose(); }
	pause();
}

/*
 * Demo of scrolling window
 */
scroll_demo()
{
	int i;

	title("Example SCROLLing window");
	wopen(15, 9, 50, 10, WSAVE|WBOX1|WCOPEN|WSCROLL|WWRAP|color[TEXT]);
	wcursor_off();
	for(i=0; i <= 5000; ++i)
		wprintf("%-6u", i);
	pause();
	wclose();
}

/*
 * Demo of an input form
 */
form_demo()
{
	title("Example FORM entry");
	wopen(15, 16, 50, 9, WSAVE|WBOX3|color[TEXT]);
	help(ftext);
	wform(15, 9, WSAVE|WBOX1|WCOPEN|color[FORM], form, software, author,
		directory, filename);
	wclose();
}

/*
 * Demo concurrent window access
 */
simul_demo()
{
	int i;
	struct WINDOW *w1, *w2, *w3, *w4;

	title("Multiple CONCURRENT open windows");

	w1 = wopen(10, 06, 20, 8, WSAVE|WCOPEN|WSCROLL|WWRAP|color[FORM]);
	w2 = wopen(50, 06, 20, 8, WSAVE|WBOX1|WCOPEN|WSCROLL|WWRAP|color[PAUSE]);
	w3 = wopen(10, 15, 20, 8, WSAVE|WBOX2|WCOPEN|WSCROLL|WWRAP|color[TEXT]);
	w4 = wopen(50, 15, 20, 8, WSAVE|WBOX3|WCOPEN|WSCROLL|WWRAP|color[MENU]);
	wcursor_off();
	for(i=0; i < 1000; ++i) {
		w_printf(w1, "\n%u", i);
		w_printf(w2, "\n%b", i);
		w_printf(w3, "\n%o", i);
		w_printf(w4, "\n%x", i); }

	pause();
	wclose();
	wclose();
	wclose();
	wclose();
}

/*
 * Demo of window attributes (for "wopen")
 */
attr_demo()
{
	unsigned i, attrs, x, y;

	title("Window OPEN attributes");
	wopen(10, 6, 60, 12, WSAVE|WBOX1|WCOPEN|color[TEXT]);
	wcursor_off();
	wputs("Select Attributes, ENTER=window, ESCAPE=exit");
	for(i=0; i < 8; ++i) {
		wgotoxy(1, i+2);
		wprintf("%u - XXX - %s\n", i+1, atext[i]); }
	for(;;) {
		for(i=0; i <8; ++i) {
			wgotoxy(5, i+2);
			wputs(wattrs[i] ? "ON " : "OFF"); }
		switch(i = wgetc()) {
			case 0x1B:
				wclose();
				return;
			case '\n' :
				attrs = 0;
				for(i=0; i < 8; ++i)
					attrs = (attrs << 1) | wattrs[i];
				wopen(42, 17, 34, 8, WSAVE|WBOX3|color[PAUSE]);
				help(awtext);
				wopen(48, 10, 20, 6, (attrs << 8) | color[FORM]);
				wcursor_line();
				while((i = wgetc()) != 0x1B) {
					x = W_OPEN->WINcurx;
					y = W_OPEN->WINcury;
					switch(i) {
						case _KUA:
							wgotoxy(x, y-1);
							break;
						case _KDA:
							wgotoxy(x, y+1);
							break;
						case _KRA:
							wgotoxy(x+1, y);
							break;
						case _KLA:
							wgotoxy(x-1, y);
							break;
						case _KHO:
							wclwin();
							break;
						case _KPU:
							wcleow();
							break;
						case _KPD:
							wcleol();
							break;
						case _KBS:
							i = 8;
						default:
							wputc(i); } }
				wclose();
				wclose();
				break;
			default:
				if((i -= '1') < 8) {
					wattrs[i] = !wattrs[i];
					break; }
				wputc(7); } }
}
