/*
 * Paper Tape Read
 *
 * Reads a paper tape from an OAE (Oliver Audio Engineering) OP-80A
 * paper tape reader. This is a manual feed reader, which presents
 * 8 data lines and a "Read Data Available" strobe, which is reset
 * by an ACK signal.
 *
 * Compile with DDS Micro-C/PC compiler available from: www.dunfield.com
 * Compile command: cc PTR -fop
 *
 * Sep 2006 - Dave Dunfield
 */
#include <stdio.h>

#define	TICK	peekw(0x40,0x6C)	// Read BIOS clock tick

unsigned
	seg,				// Memory segment
	stop,				// Position in memory segment
	Zrem,				// Removed zero count
	ppt;				// Parallel port

FILE
	*fp;				// File pointer

unsigned char
	*file,				// Filename
	*Ifile,				// Input file
	Mask = 0xFF,		// Receive mask
	Lz,					// Strip leading Zeros
	Tz,					// Strip trailing Zeros
	Az,					// Strip all zeros
	Rev,				// Reverse output
	Visual,				// Visual output
	Xack,				// XOR of ACK
	Xstat,				// XOR for Status
	Xdata;				// XOR for data

unsigned char wire[] = { "OP-80A parallel port wiring:\n\n\
OP-80A - LPT - Signal name\n\
   1       2	Data-0		\\\n\
  16       3	Data-1	 	|\n\
   2       4	Data-2		| Recommended to be connected through\n\
  15       5	Data-3		| 100 Ohm resistors to prevent excessive\n\
   3       6	Data-4		| current draw when the parallel port\n\
  14       7	Data-5		| is configured as output (default state)\n\
   4       8    Data-6		|\n\
  13       9    Data-7		/\n\
   5       1	-Reader Ack / Printer Strobe\n\
   6      10	-Reader Data Available / Printer ACK\n\
   8   18-25	Ground\n\
   9     n/a    +5vdc\n\n\
OP-80A should be jumpered for negative ACK.\n" };

/*
 * ---------------------------------------------------------------
 * Reader Interface Routines - modify these for different hardware
 * ---------------------------------------------------------------
 */

/*
 * Show visual display of tape bits
 */
void showbit(unsigned char c, unsigned char m, unsigned n)
{
	do {
		putc((c & m) ? '*' : ' ', stdout);
		putc(' ', stdout);
		m >>= 1; }
	while(--n);
}

/*
 * Pulse ACK line
 * Also serves to initialize port to all-inputs.
 */
void ack(void)
{
	out(ppt+2, 0x21 ^ Xack);	// Input + Strobe
	out(ppt+2, 0x20 ^ Xack);	// Input (without strobe)
}

/*
 * Poll for data from tape reader
 *   While no data, loop performing:
 *      Check for ESC from tty and return !0 if found
 *      Update screen display if count has changed
 *   Once data received:
 *      Handle removing Leading(LZ) or All(Az) zeros
 *		Acknowlege data as required by reader
 *		Store data in (seg,stop) and increment (stop)
 *		return 0;
 */
int getbyte()
{
	unsigned c;
	static unsigned l;

	// Make the following 'while(!(...))' if your reader
	// provides a positive RDA strobe
	while((in(ppt+1) ^ Xstat) & 0x40) {	// Wait for strobe
		if(kbtst() == 0x1B)
			return 255;
		if(l != stop)
			printf("%u\r", l=stop); }

	if(!(c = (in(ppt)^Xdata) & Mask)) {	// Handle zero bytes
		if(Lz | Az) {
			++Zrem;
			ack();
			return 0; } }

	ack();
	poke(seg, stop++, c);

	if(Visual) {
		fputs("| ", stdout);
		showbit(c, 0x80, 5);
		fputs(" . ", stdout);
		showbit(c, 0x04, 3);
		fputs("|\n", stdout); }
	return Lz = 0;		// Insure no further leading zeros removed
}

/* --------------------------------------------------------------- */

/*
 * Command line help text
 */
char help[] = { "\nPaper Tape Reader - "#__DATE__" - Dave Dunfield\n\n\
Use:	PTR output_file [input_file] [options]\n\n\
opts:	/A		= remove All 00 bytes\n\
	/L		= remove Leading 00 bytes\n\
	/T		= remove Trailing 00 bytes\n\
	/R		= write out in Reverse order (backward tape)\n\
	/V		= Visual tape display\n\
	/W		= display Wiring information\n\
	+M[mask]	= Mask input stream			[00 +7F]\n\
	+D[xor]		= Xor input stream			[00 +FF]\n\
	+A		= reader uses positive Ack		[-]\n\
	+S		= reader provides positive Strobe	[-]\n\
	P=1-3/addr	= specify Parallel port			[Highest]\n\
" };

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i, j;
	unsigned char *p, *p1;

	// Parse for command line arguments
	for(i=1; i < argc; ++i) {
		p = argv[i];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case '-R' :
		case '/R' : Rev = 255;				continue;
		case '-L' :
		case '/L' : Lz = 255;				continue;
		case '-T' :
		case '/T' : Tz = 255;				continue;
		case '-A' :
		case '/A' : Az = 255;				continue;
		case '-V' :
		case '/V' : Visual = 255;			continue;
		case '-W' :
		case '/W' : fputs(wire, stdout);	return;
		case '+A' : *(p1 = &Xack) = 1;		goto xp1;
		case '+S' : *(p1 = &Xstat) = 0x40;	goto xp1;
		case '+D' : *(p1 = &Xdata) = 0xFF;	goto xp1;
		case '+M' : *(p1 = &Mask) = 0x7F;
		xp1:	if(j = atox(p))
					*p1 = j;
				continue;
		case 'P=' : ppt = atoi(p);			continue;
		case '?'<<8:
		case '-?' :
		case '/?' : abort(help); }
		if(!file) {
			file = p-2;
			continue; }
		if(!Ifile) {
			Ifile = p-2;
			continue; }
		abort(help); }

	if(!file)		// incomplete arguments - issue command line help
		abort(help);

	seg = alloc_seg(4096);

	if(Ifile) {		// Input from file - read it
		if(ppt)
			abort("Cannot input from file AND reader\n");
		if((Mask==0xFF) && !(Lz|Tz|Az|Rev|Xdata))
			abort("You must specfify /A /L /T /R +M or +D to use an input file\n");
		fp = fopen(Ifile, "rvqb");
		while((i = getc(fp)) != EOF) {
			if(!(i = (i^Xdata)&Mask)) {
				if(Lz|Az) {
					++Zrem;
					continue; } }
				poke(seg, stop++, i);
				Lz = 0; } }
	else {			// Input from reader - locate parallel port
		i = 0;
		switch(ppt) {
		case 0 :			// Locate highest PPT
			for(i=3; i; --i) {
				if(ppt = peekw(0x40, (i*2)+6))
					break; }
			break;
		case 1 :			// Lookup PPT in BIOS
		case 2 :
		case 3 :
			ppt = peekw(0x40, ((i=ppt)*2)+6); }
		printf("Using parallel port ");
		if(i)
			printf("%u ", i);
		printf("at %04x", ppt);
		switch(ppt) {
		case 0  : printf(" - Not found\n"); return;
		default: printf(" - non-standard address", ppt);
		case 0x278:
		case 0x378:
		case 0x3BC: }
		putc('\n', stdout); }

	fp = fopen(file, "wvqb");

	if(!Ifile) {		// Reader - loop accepting data
		ack();	// This forces ppt to all inputs, and issues ACK to clear
		ack();	// reader - do twice in case initial state was invalid.
		printf("Begin reading - press ESC to end.\n");
		while(!getbyte()); }

	if(Tz) {			// Remove trailing zeros
		while(stop && !peek(seg, stop-1)) {
			++Zrem;
			--stop; } }
	if(Zrem)
		printf("%u zero byte(s) were removed.\n", Zrem);
	printf("Output file size: %u\n", stop);

	if(Rev) {			// Output in reverse order
		while(stop)
			putc(peek(seg, --stop), fp); }
	else {				// Output in normal order
		for(i=0; i < stop; ++i)
			putc(peek(seg, i), fp); }

	fclose(fp);
}
