//*****************************************************
//* Classic 99 - TI Emulator for Win32				  *
//* by M.Brent                                        *
//*                                                   *
//* Version QI3.00alpha6 -	28 Sep 02                 *
//*                                                   *
//* Thanks to:	Jeff Brown ('X' instruction)		  *
//*				Frank Palazolo, Ralph Nebet (speech)  *
//*			    Roland Meier (Disk)                   *
//*													  *
//* It should be noted that this Emulator is Dependant*
//* on running the original ROMs and expects to find  *
//* ROM routines at certain addresses. Custom ROMs    *
//* are very likely to act strangely or outright fail *
//* (actually, that may not be true except for paste) *
//*****************************************************

// Win32 Code for MSVC 6.0

// Scratchpad RAM is now at 0x8300, it was too confusing at 0x8000.
// any patches that want to access it directly (not through ROMWORD or RCPUBYTE)
// must take note of this or they will fail

#pragma warning (disable: 4113 4761 4101)

#define WIN32_LEAN_AND_MEAN
#define _WIN32_WINNT 0x0400

////////////////////////////////////////////
// Includes
////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <windows.h>
#include <mmsystem.h>
#include <process.h>
#include <malloc.h>
#ifndef GDIONLY
	#include <dsound.h>
#endif
#include <time.h>
#include <math.h>

#include "resource.h"
#include "tiemul.h"
#include "tms5220\5220intf.h"

////////////////////////////////////////////
// Globals
// These don't all NEED to be globals, but I'm only cleaning up the code, 
// not re-writing it all from scratch.
////////////////////////////////////////////

// Win32 Stuff
HINSTANCE hInstance;						// global program instance
HINSTANCE hPrevInstance;					// prev instance (always null so far)

// Memory
Byte CPU[65536];							// Main CPU Memory
Byte CPU2[8192];							// Cartridge space bank-switched
Byte GROM[65536];							// GROM space
Byte ROMMAP[65536];							// Write-protect map of CPU space
Byte CRU[4096];								// CRU space
Byte SPEECH[65536];							// Speech Synth ROM
Word GRMADD;								// GROM Address counter
Byte grmaccess,grmdata;						// GROM Prefetch emulation

// Keyboard
int KEYS[8][8]= { 
/* Joy 2 */		VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE,

/* 0 */			'M', 'J', 'U', '7', '4', 'F', 'R', 'V',
/* 1 */			191, 186, 'P', '0', '1', 'A', 'Q', 'Z',
/* 2 */			190, 'L', 'O', '9', '2', 'S', 'W', 'X',

/* Joy 1 */		VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE, VK_ESCAPE,
	
/* 3 */			188, 'K', 'I', '8', '3', 'D', 'E', 'C',
/* 4 */			'N', 'H', 'Y', '6', '5', 'G', 'T', 'B',
/* 5 */			187, VK_SPACE, VK_RETURN, VK_ESCAPE, VK_MENU, VK_SHIFT, VK_CONTROL, VK_ESCAPE 
};
char key[256];										// keyboard state buffer

// Win32 Joysticks
JOYINFOEX myJoy;
int fJoy;
int joy1mode, joy2mode;

// Audio
int need_byte;										// flag for whether waiting on a second byte
int last_byte;										// copy of last byte
int volume_mult;									// Volume multiplier
#ifndef GDIONLY
LPDIRECTSOUND lpds;									// DirectSound handle
LPDIRECTSOUNDBUFFER voice[3];						// Voice buffers
LPDIRECTSOUNDBUFFER noise[8];						// Noise buffers
LPDIRECTSOUNDBUFFER speechbuf;						// speech structure
#endif

// Assorted
char qw[80];										// temp string
int quitflag;										// quit flag
char lines[30][DEBUGLEN];							// debug lines
Word disasm[20];									// last 20 addresses for disasm
volatile int bank=0;								// Cartridge bank switch
int xb = 0;											// Is second bank (XB) loaded?
unsigned int index1;								// General counter variable
int drawspeed=0;									// flag used in display updating
int max_ipf=9999999;								// Maximum instructions per frame
int oldmax=9999999;									// copy of same
int idling=0;										// Set if an IDLE has occured
int speedup_float = 0;								// speedup floating point operations
int simulate_int = 0;								// simulate console interrupt (faster)
int instruction_count;								// Instruction counter for speed throttle
int cpucount, cpuframes;							// CPU counters for timing
int last_count=0;									// Separates video redraw and video interrupts
int timercount;										// Used to estimate runtime

int timer9901;										// 9901 interrupt timer
int starttimer9901;									// and it's set time

int retrace_count=0;								// count on the 60hz timer

int PauseInactive;									// what to do when the window is inactive
int CPUThrottle;									// Whether or not the CPU is throttled

time_t STARTTIME, ENDTIME;
long ticks;

ATOM myClass;										// Window Class
HWND myWnd, memWnd, regWnd, asmWnd, dbgWnd;			// Handle to windows
HDC myDC;											// Handle to Device Context
int fontX, fontY;									// Non-proportional font x and y size
int wndXtrim=0, wndYtrim=0;							// Window trim, used for sizing

char AVIFileName[256]="C:\\TI99AVI.AVI";			// AVI Filename

char *PasteString;									// Used for Edit->Paste
char *PasteIndex;
int PasteCount;

char diskpath[10][256];								// path to disk directories or images
char disktype[10];									// i=image, v=files with v9t9 header
													// t=files with TIFILES header,
													// r=raw program files without header
													// (load will attempt to autodetect)
unsigned long myThread;								// timer thread
HANDLE MyMutex;										// Synchronization mutex

// Console Interrupt Routine Simulator
int num_sprites;									// Number of active sprites
int y_speed;										// X pixel speed
int x_speed;										// Y pixel speed
int t1, t1a, t1b, t2, t2a, t2b;						// temp variables
int sprite_y, sprite_x, offset;						// Sprite location
int ty, tx;											// Temp location
int snd_address;									// Sound chip data address
Byte *sndlist;										// The list itself
Byte data;											// temp data

///////////////////////////////////
// Main
// Startup and shutdown system
///////////////////////////////////

int WINAPI WinMain( HINSTANCE hInst, HINSTANCE hInPrevInstance, LPSTR /*lpCmdLine*/, int /*nCmdShow*/)
{
	int idx;
	int err;
	char temp[255];
	WNDCLASS aclass;
	TEXTMETRIC myMetric;
	RECT myrect, myrect2;

	hInstance = hInst;
	hPrevInstance=hInPrevInstance;

	// Null the pointers
	myClass=0;
	myWnd=NULL;		// Classic99 Window
	memWnd=NULL;	// Memory Window
	regWnd=NULL;	// Register Window
	asmWnd=NULL;	// Assembly Window
	dbgWnd=NULL;	// Debug Window
#ifndef GDIONLY
	lpds=NULL;
	voice[0]=NULL;
	voice[1]=NULL;
	voice[2]=NULL;
#endif
	PasteString=NULL;
	PasteIndex=NULL;
	PasteCount=0;
	StretchMode=0;

	// Get the default np font dimensions with a dummy dc
	myDC=CreateCompatibleDC(NULL);
	SelectObject(myDC, GetStockObject(ANSI_FIXED_FONT));
	if (GetTextMetrics(myDC, &myMetric)) {
		fontX=myMetric.tmMaxCharWidth;
		fontY=myMetric.tmHeight;
	} else {
		fontX=20;
		fontY=20;
	}
	DeleteDC(myDC);

#ifndef GDIONLY
	for (idx=0; idx<8; idx++)
		noise[idx]=NULL;
#endif

	framedata=(unsigned short*)malloc((256+16)*(192+16)*2);	// This is where we draw everything - 8 pixel border
	framedata2=(unsigned short*)malloc((512+32)*(384+32)*2);// used for the 2xSaI filters - 16 pixel border (x2)

	MyMutex=CreateMutex(NULL, false, NULL);

	// create and register a class and open a window
	if (NULL == hPrevInstance)
	{
		aclass.style = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
		aclass.lpfnWndProc = myproc;
		aclass.cbClsExtra = 0;
		aclass.cbWndExtra = 0;
		aclass.hInstance = hInstance;
		aclass.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON1));
		aclass.hCursor = LoadCursor(NULL, IDC_ARROW);
		aclass.hbrBackground = NULL;
		aclass.lpszMenuName = MAKEINTRESOURCE(IDR_MENU1);
		aclass.lpszClassName = "TIWndClass";
		myClass = RegisterClass(&aclass);
		if (0 == myClass)
		{	
			err=GetLastError();
			sprintf(temp, "Can't create class: 0x%x", err);
			fail(temp);
		}

		// Info windows
		aclass.lpszClassName="Classic99Info";
		aclass.hbrBackground=NULL;
		aclass.lpszMenuName=NULL;
		myClass=RegisterClass(&aclass);
		if (0 == myClass) {
			debug_write("Couldn't register debug window class: 0x%x", GetLastError());
		}
	}

	myWnd = CreateWindow("TIWndClass", "Classic99", WS_OVERLAPPEDWINDOW | WS_SIZEBOX | WS_VISIBLE, CW_USEDEFAULT, 0, 280, 254, NULL, NULL, hInstance, NULL);
	if (NULL == myWnd)
	{	
		err=GetLastError();
		sprintf(temp, "Can't open window: %x", err);
		fail(temp);
	}
	ShowWindow(myWnd, SW_SHOWNORMAL);

	// Get window trimming size
	GetWindowRect(myWnd, &myrect);
	GetClientRect(myWnd, &myrect2);
	wndXtrim=((myrect.right - myrect.left)-(myrect2.right - myrect2.left))-1;
	wndYtrim=((myrect.bottom - myrect.top)-(myrect2.bottom - myrect2.top))-fontY;

	grmaccess=0;		// No GROM Access yet
	vdpaccess=0;		// No VDP address writes yet 
	vdpprefetch=0;		// Not really accurate, but eh
	tmpVDPADD=0;		// part of the vdpaccess
	interrupt_needed=0;	// No interrupt missed yet
	quitflag=0;			// no quit yet

	// set memory banks to zero. Real TIs don't do this here
	memset(CPU, 0, 65536);
	memset(CPU2, 0, 8192);
	memset(GROM, 0, 65536);
	memset(VDP, 0, 0x40000);
	memset(CRU, 1, 4096);
	memset(key, 0, 256);

	// set interrupt rate
	hzRate=60;

	// Use joystick
	fJoy=0;
	joy1mode=0;
	joy2mode=0;

	// video recording
	Recording=0;

	// window inactive
	PauseInactive=0;

	// CPU Throttling?
	CPUThrottle=0;

	// Volume multiplier
	volume_mult=-200;
	
	// clear debugging strings
	memset(lines, 0, sizeof(lines));
	memset(disasm, 0, sizeof(disasm));

	// Print some initial debug
	debug_write("---");
	debug_write("Classic99 version %s (c)2002 M.Brent", VERSION);

	// read the TIDATA file and load up the ROMs
	debug_write("Reading ROMs");
	readroms();

	// build the CPU table
	debug_write("Building CPU");
	buildcpu();

	// start sound
	debug_write("Starting Sound");
	startsound();
	
	// Init disk
	debug_write("Starting Disk");
	do_files(3);		// TI DSR sets up VDP for 3 files

	// prepare the emulation...
	cpucount=0;
	cpuframes=0;
	timercount=0;

	// start the video processor
	debug_write("Starting Video");
	startvdp();

	// set up 60hz timer
	myThread=NULL;
	myThread=_beginthread((void (__cdecl *)(void*))TimerThread, 0, NULL);
	if (myThread != -1)
		debug_write("Thread began...");
	else
		debug_write("Thread failed.");

	Sleep(100);			// time for thread to start

	// begin emulation - returns when it's time to exit
	debug_write("Starting Emulation");
	emulti();

	// Fail is the full exit
	debug_write("Shutting down");
	fail("Normal Termination");

	// good bye
	return 0;
}


//////////////////////////////////////////////////////
// start up the sound system
//////////////////////////////////////////////////////
void startsound()
{ /* start up the sound files */

#ifndef GDIONLY
	DSBUFFERDESC dsbd;
	WAVEFORMATEX pcmwf;
#endif
	unsigned int idx, idx2;
	UCHAR c;
	UCHAR *ptr1, *ptr2;
	unsigned long len1, len2, len;
	char buf[80];
	FILE *fp;
	
	need_byte=0;
	last_byte=0;

#ifndef GDIONLY
	if (DirectSoundCreate(NULL, &lpds, NULL) != DS_OK)
	{
		lpds=NULL;		// no sound
		return;
	}
	
	if (lpds->SetCooperativeLevel(myWnd, DSSCL_NORMAL) != DS_OK)
	{
		lpds->Release();
		return;
	}

	for (idx=0; idx<3; idx++)
	{
		ZeroMemory(&pcmwf, sizeof(pcmwf));
		pcmwf.wFormatTag = WAVE_FORMAT_PCM;		// wave file
		pcmwf.nChannels=1;						// 1 channel (mono)
		pcmwf.nSamplesPerSec= 22050;			// 22khz
		pcmwf.nBlockAlign=1;					// 1 byte per sample * 1 channel
		pcmwf.nAvgBytesPerSec=pcmwf.nSamplesPerSec * pcmwf.nBlockAlign;
		pcmwf.wBitsPerSample=8;					// 8 bit samples
		pcmwf.cbSize=0;							// always zero;

		ZeroMemory(&dsbd, sizeof(dsbd));
		dsbd.dwSize=sizeof(dsbd);
		dsbd.dwFlags=DSBCAPS_CTRLVOLUME | DSBCAPS_CTRLFREQUENCY | DSBCAPS_STATIC;
		dsbd.dwBufferBytes=1000;				// the sample is 1000 bytes long
		dsbd.lpwfxFormat=&pcmwf;

		if (lpds->CreateSoundBuffer(&dsbd, &voice[idx], NULL) != DS_OK)
		{
			lpds->Release();
			return;
		}
	
		// I have calculated that the square wav changed state every 20 samples. so, this
		// just generates the square wave. Apologies to those who liked the sine or triangle
		if (voice[idx]->Lock(0, 1000, (void**)&ptr1, &len1, (void**)&ptr2, &len2, DSBLOCK_ENTIREBUFFER) == DS_OK)
		{
			// since we haven't started the sound, hopefully the second pointer is nil
			if (len2 != 0)
				MessageBox(NULL, "Failed to lock tone buffer", "Classic99 Error", MB_OK);

			// unsigned
			c=255;
			for (idx2=0; idx2<len1; idx2++)
			{
				if ((idx2/20)*20 == idx2)
				{
					if (c==0)
						c=240;
					else
						c=0;
				}

				*(ptr1 + idx2)=c;
			}

			voice[idx]->Unlock(ptr1, len1, ptr2, len2);
		}

		if (voice[idx]->SetVolume(-10000) != DS_OK)
			debug_write("Set volume failed");

		if (voice[idx]->Play(0, 0, DSBPLAY_LOOPING) != DS_OK)
			debug_write("Voice DID NOT START");
	}

	// each noise channel now has it's own voice, to eliminate the need to change buffers
	// and mess with repeat lengths and the like

	for (idx=0; idx<8; idx++)
	{
		sprintf(buf,"noise%d.voc",idx+1);
		ZeroMemory(&pcmwf, sizeof(pcmwf));

		fp=fopen(buf,"rb");
		if (NULL==fp)
		{
			debug_write("Audio: Can't load %s", buf);
		}
		else
		{
			fread(buf,1,25,fp);
			// in a demonstration of bad programming, I'm just going to read the darn thing ;)
			// if it's not a VOC, you may be in trouble
			idx2=buf[20] + (256*buf[21]);			// header length
			idx2-=25;								// we've already read 25 bytes
			while (idx2--)
				fgetc(fp);

			// it's simple. I expect it to be a 22khz, 8 bit mono VOC with 1 part
			// good luck with anything else
			fgetc(fp);								// skip block type
			len=fgetc(fp)+(256*fgetc(fp))+(65536*fgetc(fp));
			fgetc(fp); fgetc(fp);					// two more bytes to skip

			pcmwf.wFormatTag = WAVE_FORMAT_PCM;		// wave file
			pcmwf.nChannels=1;						// 1 channel (mono)
			pcmwf.nSamplesPerSec= 22050;			// 22khz
			pcmwf.nBlockAlign=1;					// 1 byte per sample * 1 channel
			pcmwf.nAvgBytesPerSec=pcmwf.nSamplesPerSec * pcmwf.nBlockAlign;
			pcmwf.wBitsPerSample=8;					// 8 bit samples
			pcmwf.cbSize=0;							// always zero;

			ZeroMemory(&dsbd, sizeof(dsbd));
			dsbd.dwSize=sizeof(dsbd);
			dsbd.dwFlags=DSBCAPS_CTRLVOLUME | DSBCAPS_CTRLFREQUENCY | DSBCAPS_STATIC;
			dsbd.dwBufferBytes=len;
			dsbd.lpwfxFormat=&pcmwf;

			if (lpds->CreateSoundBuffer(&dsbd, &noise[idx], NULL) != DS_OK)
			{
				lpds->Release();
				return;
			}
	
			if (noise[idx]->Lock(0, len, (void**)&ptr1, &len1, (void**)&ptr2, &len2, DSBLOCK_ENTIREBUFFER) == DS_OK)
			{
				// since we haven't started the sound, hopefully the second pointer is nil
				if (len2 != 0)
					MessageBox(NULL, "Failed to lock noise buffer", "Classic99 Error", MB_OK);

				fread(ptr1, 1, len, fp);
			}
			noise[idx]->Unlock(ptr1, len1, ptr2, len2);
		
			if (noise[idx]->SetVolume(-10000) != DS_OK)
				debug_write("Set volume failed");

			if (noise[idx]->Play(0, 0, DSBPLAY_LOOPING) != DS_OK)
				debug_write("Noise DID NOT START");

			fclose(fp);
		}
	}

//	tms5220_sh_start();

#endif
}

//////////////////////////////////////////////////////////
// Start up the video system 
//////////////////////////////////////////////////////////
void startvdp()
{ 
	// call VDP Startup
	hVideoThread=NULL;
	hVideoThread=_beginthread((void (__cdecl *)(void*))VDPmain, 0, NULL);
	if (hVideoThread != -1)
		debug_write("Video Thread began...");
	else
		debug_write("Video Thread failed.");

	Sleep(100);

	// first retrace
	retrace_count=0;
}

//////////////////////////////////////////////////////////
// Non-fatal recoverable (?) error
//////////////////////////////////////////////////////////
void warn(char *x)
{ 
	// Warn will for now just dump a message into the log
	// eventually it should pop up a window and ask about
	// continuing

	debug_write(x);
}

//////////////////////////////////////////////////////////
// Fatal error - clean up and exit
// Note that normal exit is a fatal error ;)
//////////////////////////////////////////////////////////
void fail(char *x)
{ 
	// fatal error
	char buffer[1024];
	char buf2[256];

	// just in case it's not set yet
	quitflag=1;

	// add to the log - not useful now, but maybe in the future when it writes to disk
	debug_write(x);

	timeEndPeriod(1);

	sprintf(buffer,"\n%s\n",x);
	sprintf(buf2,"PC-%.4X  WP-%.4X  ST-%.4X  OP-%.4X\nGROM-%.4X VDP-%.4X\n",PC,WP,ST,in,GRMADD,VDPADD);
	strcat(buffer,buf2);
	sprintf(buf2,"Run Duration  : %d seconds\n",timercount/60);
	strcat(buffer,buf2);
	sprintf(buf2,"Operation time: %d instructions processed.\n",cpucount);
	strcat(buffer,buf2);
	sprintf(buf2,"Display frames: %d video frames displayed.\n",cpuframes);
	strcat(buffer,buf2);

	if (timercount<60) timercount=60;	// avoid divide by zero
	
	sprintf(buf2,"Average speed : %d instructions per second.\n",cpucount/(timercount/60));
	strcat(buffer,buf2);
	sprintf(buf2,"Frameskip     : %d\n",drawspeed);
	strcat(buffer,buf2);

	// the messagebox fails during a normal exit in WIN32.. why is that?
	MessageBox(myWnd, buffer, "Classic99 Exit", MB_OK);

	Sleep(600);			// give the threads a little time to shut down

	if (Recording) {
		CloseAVI();
	}

#ifndef GDIONLY
//	tms5220_sh_stop();

	if (voice[0])
	{
		voice[0]->Stop();
		voice[0]->Release();
	}

	if (lpds) {
		lpds->Release();
	}
#endif

	if (myWnd) {
		DestroyWindow(myWnd);
	}
	
	if (myClass) {
		UnregisterClass("TIWndClass", hInstance);
	}

	if (framedata) free(framedata);
	if (framedata2) free(framedata2);

	exit(0);
}

/////////////////////////////////////////////////////////
// Return a Word from CPU memory
/////////////////////////////////////////////////////////
Word romword(Word x)
{ 
	x&=0xfffe;		// drop LSB
	return((rcpubyte(x)<<8)+rcpubyte(x+1));
}

/////////////////////////////////////////////////////////
// Write a Word to CPU memory
/////////////////////////////////////////////////////////
void wrword(Word x, Word y)
{ 
	x&=0xfffe;		// drop LSB
	wcpubyte(x,(Byte)(y>>8));
	wcpubyte(x+1,(Byte)(y&0xff));
}

/////////////////////////////////////////////////////////
// Main loop for Emulation
/////////////////////////////////////////////////////////
void emulti()
{
	MSG msg;

	WP=romword(0);		// >0000 is the reset vector - read WP
	PC=romword(2);		// and then read the PC
	if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
	X_flag=0;			// not currently executing an X instruction
	quitflag=0;			// Don't quit

	while (!quitflag)
	{ 
		if (PC == 0x0904) {
			PC=PC;
		}

		// execute one opcode
		do1();

		// check for messages
		if (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			if (msg.message == WM_QUIT)
				quitflag=1;
			
			TranslateMessage(&msg);
			DispatchMessage(&msg);		// this will push it to the Window procedure
		}
	}
}

//////////////////////////////////////////////////////////
// Read and process the TIDATA file
//////////////////////////////////////////////////////////
void readroms()
{ 
	FILE *fp, *fp2;										// file pointers for TIDATA and binary file
	int flag, idx;										// condition flag
	char type;											// type of data being load
	unsigned short address, length;						// address and length for data
	char temp[1024], temp2[1024];						// temp string

	memset(ROMMAP, 0, 65536);							// set all CPU memory to r/w
	
	fp=fopen("tidata","r");								// open TIDATA file
	
	if (fp==NULL) 
		fail("Can't open TIDATA file!");
	
	flag=1;
	while (flag)										// till we're done reading
	{
		flag=(int)fgets(qw,80,fp);						// read the next line
		
		if (qw[0]=='#')									// check for Emulator options
		{
			if (_strnicmp(qw,"#FRAMESKIP",10) == 0)		// Frameskip
			{
				drawspeed=atoi(&qw[10]);
				debug_write("FrameSkip: %d", drawspeed);
			}

			if (_strnicmp(qw,"#MAXIPF",7) == 0)			// maximum speed
			{
				max_ipf=atoi(&qw[7]);
				oldmax=max_ipf;
				debug_write("Max IPF: %d", max_ipf);
			}

			if (_strnicmp(qw, "#AVI", 4) == 0)			// AVI filename
			{
				strncpy(AVIFileName, &qw[5], 255);
				idx=0;
				while (AVIFileName[idx])
				{
					if (AVIFileName[idx]<' ')
					{
						AVIFileName[idx]='\0';
						idx--;
					}
					idx++;
				}
				debug_write("AVI Filename: %s", AVIFileName);
			}

			if (_strnicmp(qw,"#speedup_float",14) == 0)	// ROM BIOS floating point speedups
			{
				speedup_float=atoi(&qw[14]);
				if (1==speedup_float)
					debug_write("Enabled Floating point speedups");
			}

			if (_strnicmp(qw,"#simulateint",12) == 0)	// console interrupt
			{
				simulate_int=atoi(&qw[12]);
				if (1 == simulate_int)
					debug_write("Enabled console interrupt simulator");
			}

			if (_strnicmp(qw,"#aspect",7)==0)			// aspect ratio setting
			{
				MaintainAspect=atoi(&qw[7]);
				if (1 == MaintainAspect) {
					debug_write("Locked aspect ratio");
				}
			}

			if (_strnicmp(qw,"#stretchmode",12)==0)		// stretch mode
			{
				StretchMode=atoi(&qw[12]);
				switch (StretchMode) {
				case 0:	debug_write("StretchMode: None"); break;
				case 1: debug_write("StretchMode: DIB"); break;
				case 2: debug_write("StretchMode: DX"); break;
				default:debug_write("StretchMode: Imaginary (bad value)"); break;
				}
			}

			if (_strnicmp(qw,"#fullscreenmode",15)==0)	// fullscreen mode
			{
				FullScreenMode=atoi(&qw[15]);
				switch (FullScreenMode) {
				case 1: debug_write("FullScreen: 320x240x8"); break;
				case 2: debug_write("FullScreen: 320x240x16"); break;
				case 3: debug_write("FullScreen: 320x240x24"); break;
				case 4: debug_write("FullScreen: 320x200x8"); break;
				case 5: debug_write("FullScreen: 640x480x8"); break;
				case 6: debug_write("FullScreen: 640x480x16"); break;
				case 7: debug_write("FullScreen: 640x480x24"); break;
				case 8: debug_write("FullScreen: 800x600x16"); break;
				case 9: debug_write("FullScreen: 800x600x24"); break;
				default: FullScreenMode=0;
						 debug_write("FullScreen: illegal mode"); break;
				}
			}
			
			if (_strnicmp(qw,"#joy1",5)==0)				// joystick 1
			{
				joy1mode=atoi(&qw[5]);
				switch (joy1mode) {
				case 0: debug_write("Joystick 1: Keyboard"); break;
				case 1: debug_write("Joystick 1: PC Joy1"); break;
				case 2: debug_write("Joystick 1: PC Joy2"); break;
				default: joy1mode=0;
				}
			}

			if (_strnicmp(qw,"#joy2",5)==0)				// joystick 2
			{
				joy2mode=atoi(&qw[5]);
				switch (joy2mode) {
				case 0: debug_write("Joystick 2: Keyboard"); break;
				case 1: debug_write("Joystick 2: PC Joy1"); break;
				case 2: debug_write("Joystick 2: PC Joy2"); break;
				default: joy2mode=0;
				}
			}

			if (_strnicmp(qw,"#disk",5) == 0)			// disk support (paths)
			{
				int drive;
				char path[256];
				char dtype;
				
				strcpy(path, "");
				sscanf(qw+6, "%d %1s %s", &drive, &dtype, &path);
				if ((strlen(path)) && (drive>=1 && drive<=9) && (dtype == 'i' || dtype == 'v' || dtype == 't' || dtype == 'r')) 
				{
					strcpy(&diskpath[drive][0], path);
					disktype[drive] = dtype;
					debug_write("Using path %s type %c as DSK%d.", path, dtype, drive);
				}
			}

			if (_strnicmp(qw,"#joystick",9) == 0)	// joystick
			{
				fJoy=atoi(&qw[9]);
				if (1 == fJoy)
					debug_write("Enabled joysticks");
			}

			if (_strnicmp(qw, "#volume", 7) == 0)	// audio curve
			{
				volume_mult=-atoi(&qw[7]);
				if (volume_mult>-1) volume_mult=-1;
				if (volume_mult<-700) volume_mult=-700;
				debug_write("Volume attenuation stepping set to %d db", (-volume_mult)/100);
			}
		}
		else
		{ 
			if ((flag!=(int)NULL)&&(qw[0]!=';'))		// if we have a line and it's not a comment
			{ 
				type=qw[0];								// read the type
				if ((type!='C')&&(type!='X')&&(type!='G')&&(type!='V')&&(type!='S'))
				{ 
					goto skipload;						// skip unrecognized lines
				}

				address=hex2dec(&qw[2]);				// get the address
				length=hex2dec(&qw[7]);					// and the length
      
				index1=11;					
				while (qw[++index1] >= ' ')	// get the last character of the filename
				  if (qw[index1] == '/') 
					qw[index1] = '\\';		// convert slashes to backslashes for windows
				qw[index1]=0;							// zero terminate it (in case of spaces)

				fp2=fopen(&qw[12],"rb");				// open the memory file

				if (fp2==NULL) {
					sprintf(temp2, "Can't open specified memory file: %s!", &qw[12]);
					fail(temp2);
				}
      
				sprintf(temp, "Loading %s to ",&qw[12]);// debug note
				switch (type)
				{ 
				case 'C': strcat(temp, "CPU ROM"); break;
				case 'X': strcat(temp, "XB ROM bank 2"); break;
				case 'G': strcat(temp, "GROM"); break;
				case 'V': strcat(temp, "VDP RAM"); break;
				case 'S': strcat(temp, "SPEECH ROM"); break;
				}
				sprintf(temp2, " at >%x, length >%x",address,length);
				strcat(temp, temp2);
				debug_write(temp);
      
				if (qw[6]=='-')							// TIFILES skip 128 bytes
				{ 
					debug_write("TIFILES header skip...");
					fseek(fp2, 128, SEEK_CUR);
				}

				if (qw[11]=='-')						// GRAM Kracker skip 6 bytes
				{ 
					debug_write("GRAM Kracker header skip...");
					fseek(fp2, 6, SEEK_CUR);
				}

				switch(type)
				{ 
				case 'C':	fread(&CPU[address], 1, length, fp2);
							memset(&ROMMAP[address], 1, length);
							break;
				case 'X':	fread(&CPU2[address-0x6000], 1, length, fp2);
							xb=1;
							break;
				case 'G':	fread(&GROM[address], 1, length, fp2);
							break;
				case 'V':	fread(&VDP[address], 1, length, fp2);
							break;
				case 'S':	fread(&SPEECH[address], 1, length, fp2);	// ignore speech for now
							break;
				}

				fclose(fp2);							// done this ROM
skipload: ;
			}
		}
	}
fclose(fp);												// done all ROMS
}

//////////////////////////////////////////////////////////
// Convert hex to decimal
//////////////////////////////////////////////////////////
Word hex2dec(char *x)
{
	Word z;

	z=0;
	sscanf(x, "%4hx", &z);

	return z;
}

//////////////////////////////////////////////////////////
// Speedup routine - convert from TI floating point
//////////////////////////////////////////////////////////
double tifloat2double(Byte *p) 
{
	short int firstword;
	int i, exp, sign=1;
	double res;
  
	firstword = *p++ << 8;
	firstword |= *p++;
	if (firstword==0)
	{
		return 0;
	}
	else 
	{
		if (firstword<0) 
		{								/* negative */
			sign=-1;
			firstword = -firstword;
		}
	}
	exp = ((firstword >> 8) - 0x40 - 6) << 1;
	res = firstword & 0xff;
	for (i=0; i<6; i++)
	{
		res = 100*res + *p++;
	}
	
	return (sign * res * pow(10,exp));
}
	
//////////////////////////////////////////////////////////
// Speedup routine - convert to TI floating point and set status+error flags
//////////////////////////////////////////////////////////
void double2tifloat(double d, Byte *p)
{
	short int firstword;
	int i, exp, sign=1;
  
	wcpubyte(0x8354, 0);		/* reset error flag */
	if (d==0)
	{
		*p++ = 0;
		*p = 0;
		wcpubyte(0x8354, 0x20);	/* zero flag in GPL status */
		return;
	} 
	else
	{
		if (d<0) 
		{						/* negative */
			sign=-1;
			d = -d;
		}
	}
	exp = (int) log10(d) >> 1;
	d /= pow(10,exp<<1);
	d += 0.0000000000005;		/* round last digit */
	if (d>=100) {				/* and handle overflow here... */
	  d /= 100;
	  exp++;
	}
	if (exp<=-128) {			/* underflow */
	  *p++ = 0;
	  *p = 0;
	  wcpubyte(0x8354, 0x20);	/* zero flag in GPL status */
	  return;
	} else if (exp>=128) {		/* overflow */
	  exp = 127;
	  d = 99;
	  wcpubyte(0x8354, 0x01);	/* set error flag (FIXME: 02 for division) */
	}
	firstword = (exp + 0x40) << 8 | (Byte) d;
	if (sign<0) {
	  firstword = -firstword;
	  wcpubyte(0x837c, 0x00);	/* GPL status: reset Arithmetic Greater Than */
	} else {
	  wcpubyte(0x837c, 0x40);	/* GPL status: set Arithmetic Greater Than */
	}
	*p++ = firstword >> 8;
	*p++ = firstword & 0xff;
	for (i=0; i<6; i++) 
	  {
		d = (d - (int) d) * 100;
		*p++ = (Byte) d;
	  }
}

//////////////////////////////////////////////////////////
// Interpret a single instruction
//////////////////////////////////////////////////////////
void do1()
{
	if ((PauseInactive)&&(myWnd != GetForegroundWindow()))
		return;										// exit if window inactive and we're supposed to pause

	cpucount++;										// increment instruction counter

	// check for video interrupt or redraw
	if ((interrupt_needed)&&(!skip_interrupt))
	{
		if ((retrace_count>drawspeed)&&(redraw_needed))	// only draw if VDP RAM has been changed, and it's time
		{ 
			if (DisplayEvent)
				SetEvent(DisplayEvent);

			redraw_needed=0;
		}
	
		timercount+=retrace_count;						// update timer statistics

		if (retrace_count>drawspeed)
		{
			cpuframes++;								// for accounting
			retrace_count=0;							// reset retrace counter
			instruction_count=0;						// for throttling
			last_count=0;								// to verify changes above

			if (Recording)								// for AVI recording
			{
				// write a frame to the AVI
				// I want only every 4th frame.
				RecordFrame++;
				if (RecordFrame >= 4/(drawspeed+1))
				{
					RecordFrame=0;
					WriteFrame();
				}
			}
		}

		if (((ST&0x000f) >= 2)&&(VDPREG[1]&0x20))		// if interrupt mask on and VDP ints on
		{	
			if (interrupt_needed)						// now flagged by the timer routine
			{	
				idling=0;
				if ((CPUThrottle==0)||(instruction_count < max_ipf-1)) {	// speed throttling
					if (simulate_int) {
						ConsoleInterrupt();				// Call console interrupt routine
					} else {
						interrupt_needed=0;
						wrword(0x83da,WP);				// WP in R13 
						wrword(0x83dc,PC);				// PC in R14 
						wrword(0x83de,ST);				// ST in R15 
						//ST=(ST&0xfff0);					// disable interrupts
						
						/* now load the correct workspace, and perform a branch and link to the address */
						WP=romword(0x0004);
						if ((WP>=0x8100)&&(WP<0x8400)) WP|=0x0300;
						PC=romword(0x0006);
						if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
					}
				}
			}
		}
	}

	// Control keys - must always be active
	if (key[VK_HOME]) 
	{
		// 39x19 window
		if (NULL == regWnd) {
			regWnd=CreateWindow("Classic99Info", "Classic99 - System Registers", WS_POPUPWINDOW | WS_VISIBLE | WS_CAPTION, 0, 0, 39*fontX+wndXtrim, 19*fontY+wndYtrim, myWnd, NULL, NULL, NULL);
			if (NULL == regWnd) {
				MessageBox(NULL, "Can't create register window", "D'oh", MB_OK);
			}
		}

		// DEBUGLEN-1x30 window
		if (NULL == dbgWnd) {
			dbgWnd=CreateWindow("Classic99Info", "Classic99 - Last 30 Debug Messages", WS_POPUPWINDOW | WS_VISIBLE | WS_CAPTION, 0, 0, (DEBUGLEN-1)*fontX+wndXtrim, 30*fontY+wndYtrim, myWnd, NULL, NULL, NULL);
			if (NULL == dbgWnd) {
				MessageBox(NULL, "Can't create debug window", "D'oh", MB_OK);
			}
		}

		// 40x30 window
		if (NULL == asmWnd) {
			asmWnd=CreateWindow("Classic99Info", "Classic99 - Disassembly", WS_POPUPWINDOW | WS_VISIBLE | WS_CAPTION, 0, 0, 40*fontX+wndXtrim, 30*fontY+wndYtrim, myWnd, NULL, NULL, NULL);
			if (NULL == dbgWnd) {
				MessageBox(NULL, "Can't create disassembly window", "D'oh", MB_OK);
			}
		}

		key[VK_HOME]=0;
	}

	if (key[VK_END]) {
		if (regWnd) {
			SendMessage(regWnd, WM_CLOSE, 0, 0);
		}
		if (dbgWnd) {
			SendMessage(dbgWnd, WM_CLOSE, 0, 0);
		}
		if (asmWnd) {
			SendMessage(asmWnd, WM_CLOSE, 0, 0);
		}
	}

	// breakpoint handling
//	if (PC & 0xf000 == 0x6000) {
//	  max_ipf=0;
//	  CPUThrottle=1;
//	}

	// Single-stepping
	if (key[VK_F1])
	{	
		if (0 == max_ipf)
		{
			max_ipf=oldmax;
			SetWindowText(myWnd, "Classic99");
		}
		else
		{
			max_ipf=0;
			SetWindowText(myWnd, "Classic99 - Paused. F1 to Continue, F2 to Step");
		}
		key[VK_F1]=0;
	}
	if (key[VK_F2])
	{
		instruction_count=-1;			// this will allow 1 instruction to run in pause
		key[VK_F2]=0;
	}									

	////////////////// System Patches
	if ((!idling)&&((CPUThrottle==0)||(instruction_count<max_ipf)))		// speed throttling
	{
		// Keyboard hacks - requires original ROM!
		if (PC==0x478) {	// This is where KSCAN begins to exit
			if (NULL != PasteString) {
				
				if (PasteString==PasteIndex) SetWindowText(myWnd, "Classic99 - Pasting (ESC Cancels)");	

				if (key[VK_ESCAPE]) {
					*PasteIndex='\0';
				}

				if ((rcpubyte(0x8374)==0)||(rcpubyte(0x8374)==5)) {		// Check for pastestring - note keyboard is still active
					if (*PasteIndex) {
						if (*PasteIndex==10) {
							// CRLF to CR, LF to CR
							if ((PasteIndex==PasteString)||(*(PasteIndex-1)!=13)) {
								*PasteIndex=13;
							}
						}

						if (PasteCount<1) {
							if ((*PasteIndex>31)||(*PasteIndex==13)) {
								wcpubyte(WP, *PasteIndex);					/* set R0 (byte) with keycode */
								wcpubyte(WP+12, rcpubyte(0x837c) | 0x20);	/* R6 must contain the status byte */
							}
							if (PasteCount<1) wcpubyte(0x837c, rcpubyte(0x837c)|0x20);
							PasteCount++;
						} else {
							PasteCount++;
							if (PasteCount==2) {
								PasteIndex++;
							}
							if (PasteCount==3) {
								PasteCount=0;
							}
						}

					} else {
						PasteIndex=NULL;
						free(PasteString);
						PasteString=NULL;
						SetWindowText(myWnd, "Classic99");
					}
				}
			}
		}

		// intercept patches for Classic99 DSK1 emulation
		if (PC==0x4800)									// special address to start DSK1 emulation
		{
			do_dsrlnk();
			PC=romword(WP+22)+2;						// return address in R11.. TI is a bit silly
			if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
														// and if we don't increment by 2, it's considered
														// an error condition
			return;
		}

		if (PC == 0x4810) 
		{												// powerup 
			wrword(0x8358, 0x3ef5);						// volume ID block
			wrword(0x8366, 0x3eea);						// VDP stack
			do_files(3);
			PC=romword(WP+22);
			if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
			return;
		}

		if (PC>=0x4820 && PC<=0x482c)					// special intercept address for emulated ROM
		{
			do_sbrlnk();
			PC=romword(WP+22)+2;						// return address in R11.. as above
			if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
			return;
		}

		if (speedup_float) 
		{									/* speedup floating point operations */
			if (PC==0x0d80) {					/* ADD */
				double2tifloat(tifloat2double(&CPU[0x835c]) + tifloat2double(&CPU[0x834a]), &CPU[0x834a]);
				PC=romword(WP+22);			// return address in R11.
				if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
				return;
			}
			if (PC==0x0d7c) {					/* SUBT */
				double2tifloat(tifloat2double(&CPU[0x835c]) - tifloat2double(&CPU[0x834a]), &CPU[0x834a]);
				PC=romword(WP+22);			// return address in R11.
				if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
				return;
			}
			if (PC==0xe88) {					/* MULT */
				double2tifloat(tifloat2double(&CPU[0x835c]) * tifloat2double(&CPU[0x834a]), &CPU[0x834a]);
				PC=romword(WP+22);			// return address in R11.
				if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
				return;
			}
			if (PC==0x0ff4) {					/* DIV */
				double den = tifloat2double(&CPU[0x834a]);
				if (den) {
				  double2tifloat(tifloat2double(&CPU[0x835c]) / den, &CPU[0x834a]);
				} else {
				  double2tifloat((CPU[0x835c] >= 0x80) ? -99e127 : 99e-127, &CPU[0x834a]);
				  CPU[0x8354] = 0x02; /* set error */
				}
				  PC=romword(WP+22);			// return address in R11.
				  if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
				return;
			}
			if (PC==0x12b8) {					/* CFI */
				wrword(0x834a, (Word) tifloat2double(&CPU[0x834a]));
				PC=romword(WP+22);			// return address in R11.
				if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
				return;
			}
		} /* speedup_float */

		///////////////// End of patches 

		instruction_count++;

		if (X_flag==0)
		{ 
			// Update the disassembly trace
			memmove(&disasm[0], &disasm[1], 38);//19*sizeof(Word));	// really should be a ring buffer
			disasm[19]=PC;

			in=romword(PC);							// ie: not an 'X' command
		    ADDPC(2);								// thanks to Jeff Brown for explaining that!
		}

		(*opcode[in])();							// 'in' contains the opcode to execute
	}

	skip_interrupt=0;
}

//////////////////////////////////////////////////////
// Read a single byte from CPU memory
//////////////////////////////////////////////////////
Byte rcpubyte(Word x)
{
	// TI CPU memory map
	// >0000 - >1fff  Console ROM
	// >2000 - >3fff  Low bank RAM
	// >4000 - >5fff  DSR ROMs
	// >6000 - >7fff  Cartridge expansion
	// >8000 - >9fff  Memory-mapped devices & CPU scratchpad
	// >a000 - >ffff  High bank RAM
	// 
	// All is fine and dandy, except the memory-mapped devices, and the
	// fact that writing to the cartridge port with XB in places causes
	// a bank switch. In this emulator, that will only happen if bank 2
	// has been loaded. A custom DSR will be written to be loaded which
	// will appear to the "TI" to support all valid devices, and it will
	// be loaded into the DSR space, which then will not be paged.

  switch (x & 0xe000) {
  case 0x8000:
	switch (x & 0xfc00) {
	case 0x8000:				// scratchpad RAM - 256 bytes repeating.
		return(CPU[x|0x0300]);	// I map it all to >83xx
	case 0x8400:				// Don't read the sound chip (can hang a real TI)
		return 0;
	case 0x8800:				// VDP read data
		return(rvdpbyte(x));
	case 0x8c00:				// VDP write data
		return 0;
	case 0x9000:				// Speech read data
		return(rspeechbyte(x));
	case 0x9400:				// Speech write data
		return 0;
	case 0x9800:				// read GROM data
		return(rgrmbyte(x));
	case 0x9c00:				// write GROM data
		return 0;
	default:					// We shouldn't get here, but just in case...
		return 0;
	}
  case 0x0000:					// console ROM
  case 0x2000:					// normal CPU RAM
  case 0x4000:					// DSR ROM (no bank switching) - FIXME: check CRU
  case 0xa000:					// normal CPU RAM
  case 0xc000:					// normal CPU RAM
  case 0xe000:					// normal CPU RAM
		return(CPU[x]);
  case 0x6000:					// cartridge ROM
		// XB is supposed to only page the upper 4k, but some other carts seem to like it all
		// paged? Dunno.. weird.
		if (xb && bank)
			return(CPU2[x-0x6000]);	// cartridge bank 2
		else 
			return(CPU[x]);			// cartridge bank 1
  default:						// We shouldn't get here, but just in case...
		return 0;
	}
}

//////////////////////////////////////////////////////////
// Write a byte to CPU memory
//////////////////////////////////////////////////////////
void wcpubyte(Word x, Byte c)
{
	rcpubyte(x);	// weak emulation of the read-before-write feature

	if (ROMMAP[x])	// trap ROM writes and check for XB bank switch
	{
		if ((x>=0x6000)&&(x<0x8000)&&(xb))		
		{ 
			bank=(x&0x0002)>>1;							// XB bank switch
		}
		return;											// can't write to ROM!
	}

  switch (x & 0xe000) {
  case 0x8000:
	switch (x & 0xfc00) {
	case 0x8000:				// scratchpad RAM - 256 bytes repeating.
		CPU[x|0x0300] = c;		// I map it all to >83xx
		break;
	case 0x8400:				// Sound write data
		wsndbyte(c);
		break;
	case 0x8800:				// VDP read data
		break;
	case 0x8c00:				// VDP write data
		wvdpbyte(x,c);
		break;
	case 0x9000:				// Speech read data
		break;
	case 0x9400:				// Speech write data
		wspeechbyte(x,c);
		break;
	case 0x9800:				// read GROM data
		break;
	case 0x9c00:				// write GROM data
		wgrmbyte(x,c);
		break;
	default:					// We shouldn't get here, but just in case...
		break;
	}
	break;
  case 0x0000:					// console ROM
  case 0x2000:					// normal CPU RAM
  case 0x4000:					// DSR ROM (no bank switching) - FIXME: check CRU
  case 0x6000:					// Cartridge RAM (ROM is trapped above)
  case 0xa000:					// normal CPU RAM
  case 0xc000:					// normal CPU RAM
  case 0xe000:					// normal CPU RAM
		CPU[x]=c;
		break;
  }
}

//////////////////////////////////////////////////////
// Read a byte from the speech synthesizer
//////////////////////////////////////////////////////
Byte rspeechbyte(Word x)
{
//	return tms5220_status_r(x);		// address passed is irrelevant
	return 0;
}

//////////////////////////////////////////////////////
// Write a byte to the speech synthesizer
//////////////////////////////////////////////////////
void wspeechbyte(Word x, Byte c)
{
//	tms5220_data_w(x, c);			// address passed is irrelevant
}

//////////////////////////////////////////////////////
// Increment VDP Address
//////////////////////////////////////////////////////
void increment_vdpadd() 
{
	VDPADD=(++VDPADD) & 0x3fff;
}

//////////////////////////////////////////////////////////////
// Read from VDP chip
//////////////////////////////////////////////////////////////
Byte rvdpbyte(Word x)
{ 
	unsigned short z;

	if ((x>=0x8c00) || (x&1))
	{
		return(0);											// write address
	}

	if (x&0x0002)
	{	/* read status */
		z=VDPS;
		VDPS&=0x5f;			// reset interrupt and collision bits
		vdpaccess=0;		// reset byte flag
		return((Byte)z);
	}
	else
	{ /* read data */
		z=vdpprefetch;
		vdpprefetch=VDP[VDPADD];
		increment_vdpadd();
		vdpaccess=0;
		return ((Byte)z);
	}
}

///////////////////////////////////////////////////////////////
// Write to VDP chip
///////////////////////////////////////////////////////////////
void wvdpbyte(Word x, Byte c)
{
	if (x<0x8c00 || (x&1)) 
	{
		return;							/* not going to write at that block */
	}

	if (x&0x0002)
	{	/* write address */
		tmpVDPADD=(tmpVDPADD>>8)|(c<<8);
		vdpaccess++;
		if (vdpaccess==2)
		{ 
			if (tmpVDPADD&0x8000)
			{ 
				wVDPreg((Byte)((tmpVDPADD&0x0700)>>8),(Byte)(tmpVDPADD&0x00ff));
				redraw_needed=1;
			} else {
				VDPADD=tmpVDPADD;
				if ((VDPADD&0x4000)==0) {	// set for reading?
					vdpprefetch=VDP[VDPADD];
					increment_vdpadd();
				} else {
					VDPADD&=0x3fff;			// writing, just mask the bit off
				}
			}
			vdpaccess=0;
		}
	}
	else
	{	/* write data */
		VDP[VDPADD]=c;
		vdpprefetch=c;
		increment_vdpadd();
		redraw_needed=1;
		vdpaccess=0;
	}
}

////////////////////////////////////////////////////////////////
// Write to VDP Register
////////////////////////////////////////////////////////////////
void wVDPreg(Byte r, Byte v)
{ 
	int t;

	VDPREG[r]=v;

	if (r==7)
	{	/* color of screen, set color 0 (trans) to match */
		t=v&0xf;
		if (t) {
			TIPALETTE[0]=TIPALETTE[t];
		} else {
			TIPALETTE[0]=0;
		}
		redraw_needed=1;
	}
}

////////////////////////////////////////////////////////////////
// Write a byte to the sound chip
////////////////////////////////////////////////////////////////
void wsndbyte(Byte c)
{
	unsigned int x, idx;								// temp variable
	static int noise_volume=-10000;						// volume of noise
	static int noise_type=0;							// type of noise
	static int freq3=22000;								// frequency calculator

#ifndef GDIONLY
	if (NULL == lpds) return;
#endif

	// 'c' contains the byte currently being written to the sound chip
	// all functions are 1 or 2 bytes long, as follows					
	//
	// BYTE		BIT		PURPOSE											
	//	1		0		always '1'										
	//			1-3		Operation:	000 - tone 1 frequency				
	//								001 - tone 1 volume					
	//								010 - tone 2 frequency				
	//								011 - tone 2 volume					
	//								100 - tone 3 frequency				
	//								101 - tone 3 volume					
	//								110 - noise control					
	//								111 - noise volume					
	//			4-7		Least sig. frequency bits for tone, or volume	
	//					setting (0-F), or type of noise.				
	//					(volume F is off)								
	//					Noise set:	4 - always 0						
	//								5 - 0=periodic noise, 1=white noise 
	//								6-7 - shift rate from table, or 11	
	//									to get rate from voice 3.		
	//	2		0-1		Always '0'. This byte only used for frequency	
	//			2-7		Most sig. frequency bits						
	//
	// Commands are instantaneous

	switch (c&0xf0)										// check command
	{	
	case 0x80:											// Voice 1 frequency
		last_byte=c;
		need_byte=1;									// needs a second byte
		break;

	case 0x90:
		x=(c&0x0f);										// Voice 1 volume
#ifndef GDIONLY
		if (x==0xf) {
			voice[0]->SetVolume(-10000);
		} else {
			voice[0]->SetVolume(x*(volume_mult));
		}
#endif
		break;

	case 0xa0:
		last_byte=c;									// Voice 2 frequency
		need_byte=2;									// needs a second byte
		break;

	case 0xb0:
		x=(c&0x0f);										// Voice 2 volume
#ifndef GDIONLY
		if (x==0xf) {
			voice[1]->SetVolume(-10000);
		} else {
			voice[1]->SetVolume(x*(volume_mult));
		}
#endif
		break;

	case 0xc0:
		last_byte=c;									// Voice 3 frequency
		need_byte=3;									// needs a second byte
		break;

	case 0xd0:
		x=(c&0x0f);										// Voice 3 volume
#ifndef GDIONLY
		if (x==0xf) {
			voice[2]->SetVolume(-10000);
		} else {
			voice[2]->SetVolume(x*(volume_mult));
		}
#endif
		break;

	case 0xe0:
		x=(c&0x07);										// Noise - get type
#ifndef GDIONLY
		for (idx=0; idx<8; idx++)
		{
			if (idx==x)
				noise[idx]->SetVolume(noise_volume);
			else
				noise[idx]->SetVolume(-10000);			// turn off noises, except the one we want
		}
#endif
		noise_type=x;
		break;

	case 0xf0:											// Noise volume
		x=(c&0x0f);
#ifndef GDIONLY
		if (x==0xf) {
			noise[noise_type]->SetVolume(-10000);
			noise_volume=-10000;
		} else {
			noise[noise_type]->SetVolume(x*(volume_mult));
			noise_volume=x*(volume_mult);
		}
#endif
		break;

	default:											// unknown byte
		if (need_byte)									// if we were waiting for it...
		{
			x=((last_byte&0x0f) | ((c&0x3f)<<4)) + 1;	// Get frequency code
			x=(111860/x)*22050;							// Make sample rate in HZ (22khz samples)
			x/=550;										// adjust to balance (sample recorded as 550hz)
#ifndef GDIONLY
			voice[need_byte-1]->SetFrequency(x);
			if (need_byte==3)							// If voice 3, save and adjust for noise channel
			{
				freq3=x;								
				noise[3]->SetFrequency(freq3);
				noise[7]->SetFrequency(freq3);
			}
#endif
			need_byte=0;								// don't need another byte
		}
		break;
	}
}

//////////////////////////////////////////////////////////////////
// Read a byte from GROM
//////////////////////////////////////////////////////////////////
Byte rgrmbyte(Word x)
{
	unsigned int z;										// temp variable

	if (x>=0x9c00)
	{
		return(0);										// write address
	}

	if (x&0x0002)
	{
		grmaccess=2;									// read GROM address
		z=(GRMADD&0xff00)>>8;
		GRMADD=(((GRMADD&0xff)<<8)|(GRMADD&0xff));		// read is destructive
		return((Byte)z);
	}
	else
	{
		grmaccess=2;									// read data
		z=grmdata;
		grmdata=GROM[GRMADD++];
		return((Byte)z);
	}
}

//////////////////////////////////////////////////////////////////
// Write a byte to GROM
//////////////////////////////////////////////////////////////////
void wgrmbyte(Word x, Byte c)
{
	if (x<0x9c00) 
	{
		return;											// read address
	}

	if (x&0x0002)
	{
		GRMADD=(GRMADD<<8)|(c&0x00ff);					// write GROM address
		grmaccess--;
		if (grmaccess==0)
		{ 
			grmaccess=2;								// prefetch emulation
			grmdata=GROM[GRMADD++];
		}
	}
	else
	{
		grmaccess=2;
		warn("Writing to GROM!!");						// write to GROM not supported
		//GROM[GRMADD++]=c;								// don't write data (it's G*ROM*)
		GRMADD++;
	}
}

//////////////////////////////////////////////////////////////////
// Write a bit to CRU
//////////////////////////////////////////////////////////////////
void wcru(Word ad, int bt)
{
	ad=(ad&0x0fff);										// get actual CRU line

	if (bt) {											// write the data
		CRU[ad]=1;
	} else {
		CRU[ad]=0;
		if (ad == 0) {
			// Turning off timer mode - start timer
			timer9901=CRU[1];
			timer9901|=CRU[2]<<1;
			timer9901|=CRU[3]<<2;
			timer9901|=CRU[4]<<3;
			timer9901|=CRU[5]<<4;
			timer9901|=CRU[6]<<5;
			timer9901|=CRU[7]<<6;
			timer9901|=CRU[8]<<7;
			timer9901|=CRU[9]<<8;
			timer9901|=CRU[10]<<9;
			timer9901|=CRU[11]<<10;
			timer9901|=CRU[12]<<11;
			timer9901|=CRU[13]<<12;
			timer9901|=CRU[14]<<13;
			timer9901|=CRU[15]<<14;
			starttimer9901=timer9901;
			debug_write("Setting 9901 timer to %d ticks", timer9901);
		}
	}
}

//////////////////////////////////////////////////////////////////
// Read a bit from CRU
// The CRU is kinda like black magic and is weird to emulate
//////////////////////////////////////////////////////////////////
int rcru(Word ad)
{
	int ret,col;										// temp variables
	int joyX, joyY, joyFire;

	if ((CRU[0]==0)&&(ad<=16)&&(ad>0)) {				// read elapsed time from timer
		if (ad==2) debug_write("Reading 9901 timer at %d ticks", timer9901);
		Word mask=0x01<<(ad-1);
		if (timer9901 & mask) {
			return 1;
		} else {
			return 0;
		}
	}

	// Read external hardware
	joyX=0;
	joyY=0;
	joyFire=0;

	// only certain CRU bits are implemented, check them

	ad=(ad&0x0fff);										// get actual CRU line
	ret=1;												// default return code (false)

	// keyboard/joystick. PC Keyboard is currently *NOT* remapped		
	// so that it's keys work. (TODO)
	
	// keyboard reads as an array. Bits 24, 26 and 28 set the line to	
	// scan (columns). Bits 6-14 are used for return. 0 means on.		
	// The address was divided by 2 before being given to the routine	

	if ((ad>=0x03)&&(ad<=0x0a))
	{	
		if ((ad==0x07)&&(CRU[0x15]==0))					// is it ALPHA LOCK?
		{	
			ret=1;
			if (GetKeyState(VK_CAPITAL) & 0x01)			// check CAPS LOCK (on?)
			{	
				ret=0;									// set Alpha Lock on
			}
		}
		else											// other key
		{
			col=(CRU[0x14]==0 ? 1 : 0) | (CRU[0x13]==0 ? 2 : 0) | (CRU[0x12]==0 ? 4 : 0);	// get column

			if ((col == 0) || (col == 4))				// reading joystick
			{	
				// TODO: This still reads the joystick many times for a single scan, but it's better ;)
				if (fJoy) {
					int device;

					device=-1;

					if (col==0) {
						switch (joy2mode) {
							case 1: device=JOYSTICKID1; break;
							case 2: device=JOYSTICKID2; break;
						}
					} else {
						switch (joy1mode) {
							case 1: device=JOYSTICKID1; break;
							case 2: device=JOYSTICKID2; break;
						}
					}

					if (device!=-1) {
						memset(&myJoy, 0, sizeof(myJoy));
						myJoy.dwSize=sizeof(myJoy);
						myJoy.dwFlags=JOY_RETURNBUTTONS | JOY_RETURNX | JOY_RETURNY | JOY_USEDEADZONE;
						if (JOYERR_NOERROR == joyGetPosEx(device, &myJoy)) {
							if (0!=myJoy.dwButtons) {
								joyFire=1;
							}
							if (myJoy.dwXpos<0x4000) {
								joyX=-4;
							}
							if (myJoy.dwXpos>0xC000) {
								joyX=4;
							}
							if (myJoy.dwYpos<0x4000) {
								joyY=4;
							}
							if (myJoy.dwYpos>0xC000) {
								joyY=-4;
							}
						}
					} else {	// read the keyboard
						if (key[VK_TAB]) {
							joyFire=1;
						}
						if (key[VK_LEFT]) {
							joyX=-4;
						}
						if (key[VK_RIGHT]) {
							joyX=4;
						}
						if (key[VK_UP]) {
							joyY=4;
						}
						if (key[VK_DOWN]) {
							joyY=-4;
						}
					}
				}

				if (ad == 3)
				{	
					if ((key[KEYS[col][0]])||(joyFire))	// button reads normally
					{
						ret=0;
					}
				}
				else
				{
					if (key[KEYS[col][ad-3]])			// stick return (*not* inverted. Duh)
					{	
						ret=0;
					}
					if (ret) {
						switch (ad-3) {						// Check real joystick
						case 1: if (joyX ==-4) ret=0; break;
						case 2: if (joyX == 4) ret=0; break;
						case 3: if (joyY ==-4) ret=0; break;
						case 4: if (joyY == 4) ret=0; break;
						}
					}
				}
			}
			else
			{
				if (key[KEYS[col][ad-3]])				// normal key
				{	
					ret=0;
				}
			}

		}
	}

	// are we checking VDP interrupt?
	if (ad == 0x02)
		ret=0;											// that's the only int we have

	// no other devices at this time

	return(ret);
}

/////////////////////////////////////////////////////////////////////////
// Write a line to the debug buffer displayed on the debug screen
/////////////////////////////////////////////////////////////////////////
void debug_write(char *s, ...)
{
	char buf[1024];

	vsprintf(buf, s, (char*)((&s)+1));

	if (!quitflag) {
		OutputDebugString(buf);
		OutputDebugString("\n");
	}

	buf[DEBUGLEN-1]='\0';

	memcpy(&lines[0][0], &lines[1][0], 29*DEBUGLEN);				// scroll data
	strncpy(&lines[29][0], buf, DEBUGLEN);							// copy in new line
	memset(&lines[29][strlen(buf)], 0x20, DEBUGLEN-strlen(buf));	// clear rest of line
	lines[29][DEBUGLEN-1]='\0';										// zero terminate
}

///////////////////////////////////////////////////////////////////
// Console interrupt simulator
///////////////////////////////////////////////////////////////////
void ConsoleInterrupt()
{
	Byte x1;											// temp var
	int t1,t2;											// temp vars

	interrupt_needed=0;
	x1=rcpubyte(0x83c2);
	if ((x1&0x80)==0)									// if console routine on
	{ 
		if ((x1&0x40)==0)								// update sprites
	    {												// tables are fixed in memory
			for (num_sprites=0; num_sprites<rcpubyte(0x837a); num_sprites++) // note: not masking the sprites!
			{	
				offset=num_sprites<<2;					// I didn't fully understand the sprite
				y_speed=(Byte)VDP[0x780+offset];		// movement routine, and my own simplified
				x_speed=(Byte)VDP[0x781+offset];		// version didn't work as expected. So this
				t1=(Byte)VDP[0x782+offset];				// code is more or less a straight translation
				t2=(Byte)VDP[0x783+offset];				// of the 9900 code ;)
				t1a=(y_speed&0xf0)>>4;
				t1b=(y_speed&0xf);
				t2a=(x_speed&0xf0)>>4;
				t2b=(x_speed&0xf);

				if (t1a>0x7)
				{	
					t1a-=0xf;
					if (t1b==0)
						t1a-=1;
				}
				if (t2a>0x7)
				{	
					t2a-=0xf;
					if (t2b==0)
						t2a-=1;
				}
	
				t1b=(y_speed&0xf);
				if ((y_speed>0x7f)&&(t1b))
					t1b=(t1b-0x10);
				if ((x_speed>0x7f)&&(t2b))
					t2b=(t2b-0x10);

				sprite_y=(Byte)VDP[0x300+offset];
				sprite_x=(Byte)VDP[0x301+offset];
				
				ty=sprite_y;
				tx=sprite_x;
			
				t1+=t1b;
				t2+=t2b;

				sprite_y+=t1a;
				sprite_x+=t2a;
	
				if (t1>0xf)
				{	
					t1-=0xf;
					sprite_y++;
				}
				if (t1<0)
				{	
					t1+=0xf;
					sprite_y--;
				}

				if (t2>0xf)
				{	
					t2-=0xf;
					sprite_x++;
				}
				if (t2<0)
				{	
					t2+=0xf;
					sprite_x--;
				}

				sprite_x&=0xff;
				sprite_y&=0xff;

				if ((sprite_y>=0xc0)&&(sprite_y<=0xe0))
				{	
					if (y_speed>0x7f)
						sprite_y-=0x20;
					else
						sprite_y+=0x20;
				}
	
				VDP[0x300+offset]=(Byte)sprite_y;
				VDP[0x301+offset]=(Byte)sprite_x;
				VDP[0x782+offset]=(Byte)t1;
				VDP[0x783+offset]=(Byte)t2;
			
				if ((ty!=sprite_y)||(tx!=sprite_x))
					redraw_needed=1;

			}
		}
    
		if ((x1&0x20)==0)								// update sound list
	    { 
	      if (rcpubyte(0x83ce))
		  {		wcpubyte(0x83ce, rcpubyte(0x83ce)-1);
				if (rcpubyte(0x83ce)==0)
				{	
					snd_address=romword(0x83cc);
					if (rcpubyte(0x83fd) & 0x01) {
						sndlist=(&VDP[snd_address]);	// VDP sound list
					} else
					{
						sndlist=(&GROM[snd_address]);	// GROM sound list
					}
					data=*sndlist++;
					snd_address+=data;
					if ((data!=0)&&(data!=0xff))
					{	
						while (data--)
						{	
							wsndbyte(*sndlist++);
						}
						data=*sndlist++;
						snd_address+=2;
						wrword(0x83cc, snd_address);
						wcpubyte(0x83ce, data);
					}	
					else
					{	
						if (data==0xff)
						{	
							if (rcpubyte(0x83fd) & 0x1)
								wcpubyte(0x83fd, rcpubyte(0x83fd)&0xfe);
							else
								wcpubyte(0x83fd, rcpubyte(0x83fd)|0x01);
						}
						wcpubyte(0x83cc, *sndlist++);
						wcpubyte(0x83cd, *sndlist++);
						wcpubyte(0x83ce, 0x01);
					}
				}
			}
		}
	
	    if ((x1&0x10)==0)								// Check QUIT key
		{ 
			if (((key[VK_MENU])||(key[VK_CONTROL]))&&(key[187]))
			{ 
				while (key['=']);						// wait for key release
				ST=(ST&0xfff0);							// disable interrupts
				WP=romword(0x0000);						// Reset = BLWP @>0000
				if ((WP>=0x8100)&&(WP<0x8400)) WP|=0x0300;
				wrword(WP+26,0xffff);
				wrword(WP+28,PC);
				wrword(WP+30,ST);
				PC=romword(0x0002);
				if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
				grmaccess=0;		// No GROM Access yet
				vdpaccess=0;		// No VDP address writes yet 
				interrupt_needed=0;	// No interrupt missed yet
			}
	    }
	}

	// the following parts can't be turned off, except by LIMI or disabling VDP ints 
  
  	wrword(0x83d6,romword(0x83d6)+2);	// this blanks the screen when it 
										// wraps around to zero 
	
	if (romword(0x83d6)==0)				// we're wrapped					
	{
		t1=rcpubyte(0x83d4);			// read the byte...					
		t1=(t1|0x20)&0xbf;				// int turns ON VDP interrupts, and	
										// turns OFF display bit			
		VDPREG[1]=(Byte)t1;				// write it							
	}
  
	wcpubyte(0x8379, rcpubyte(0x8379)+rcpubyte(0x83fc));// VDP Int Timer - not sure how it	
										// gets this, but here it is. Add	
										// byte from R14 of the GPL WP at
										// >83e0. Can't trace what it is.	
										// E/A manual calls it System Status

	wcpubyte(0x837b, VDPS);				// VDP Status register - copied to CPU RAM	
	VDPS=0;								// A read clears the register
    
	if (romword(0x83C4)) 				// call the user interrupt routine 
	{	// pretend that we really were called with BLWP by saving the values	
		// into the appropriate workspace (0x83c0)
		
		wrword(0x83da,WP);				// WP in R13 
		wrword(0x83dc,PC);				// PC in R14 
		wrword(0x83de,ST);				// ST in R15 

		ST=(ST&0xfff0);					// disable interrupts

		/* now load the correct workspace, and perform a branch and link to the address */
		WP=0x83e0;				/* interrupt workspace is 0x80e0						*/
		wrword(0x83f6,0x0ab8);	/* write into new R11 hard-coded return address in ROM	*/
		PC=romword(0x83c4);		/* off we go.. TI should return normally now			*/
		if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
	}
}

//////////////////////////////////////////////////////////////
// 'Retrace' counter for timing - runs at 50 or 60 hz
//////////////////////////////////////////////////////////////
void __cdecl TimerThread()
{
	unsigned __int64 freq;
	unsigned __int64 count;
	unsigned __int64 last;
	unsigned __int64 max;

	SetThreadAffinityMask(GetCurrentThread(), 0x01);		// Force this thread to CPU 1, just in case
	
	if (false == QueryPerformanceFrequency((LARGE_INTEGER*)&freq)) {
		freq=0;
	}

	time(&STARTTIME);

	if (freq==0) {
		debug_write("No performance counter! Faking it with Sleep...");
	} else {
		debug_write("Started Timer Thread - Counter is %I64d hz", freq);
	}
	
	last=0;
	count=0;

	// The 9901 timer decrements every 64 periods, with the TI having a 3MHz clock speed
	// Thus, it decrements 46875 times per second. If CRU bit 3 is on, we trigger an
	// interrupt when it expires

	while (quitflag==0)
	{	
		if (freq==0) {
			Sleep(hzRate==50 ? 20:16);
			// Fairly inaccurate 9901 hack ;)
			if (timer9901>0) {
				timer9901-=(hzRate==50 ? 937 : 781);  // really 937.5 and 781.25
				if (timer9901 < 1) {
					timer9901=starttimer9901;
					debug_write("9901 timer expired");
					if ((CRU[3])&&(ST&0x02)) {			// Timer interrupts, and interrupt mask
						wrword(0x83da,WP);				// WP in R13 
						wrword(0x83dc,PC);				// PC in R14 
						wrword(0x83de,ST);				// ST in R15 
//						ST=(ST&0xfff0);					// disable interrupts
						
						/* now load the correct workspace, and perform a branch and link to the address */
						WP=romword(0x0004);
						if ((WP>=0x8100)&&(WP<0x8400)) WP|=0x0300;
						PC=romword(0x0006);
						if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
					}
				}
			}
		} else {
			unsigned __int64 tmplast=0;

			max=freq/hzRate;		// High accuracy, high CPU use
			while (quitflag==0) {
				QueryPerformanceCounter((LARGE_INTEGER*)&count);
				if (count==0) {
					debug_write("Performance counter failed - falling back on Sleep()");
					freq=0;
					break;
				}

				if (tmplast == 0) tmplast=count;
				
				// Somewhat better 9901 timing
				if (timer9901>0) {
					unsigned __int64 tmpcnt;

					tmpcnt=count-tmplast;
					tmpcnt=((tmpcnt<<8)/((freq<<8)/(46875<<8)))>>8;	// 8 bits of fraction
					if (tmpcnt) {			
						timer9901-=(int)(tmpcnt&0xfffff);
						tmplast=count;						// This way, we'll never spin on the difference being too small
					}

					if (timer9901 < 1) {
						timer9901=starttimer9901;
						debug_write("9901 timer expired");
						if ((CRU[3])&&(ST&0xF)) {			// Timer interrupts, and interrupt mask
							wrword(0x83da,WP);				// WP in R13 
							wrword(0x83dc,PC);				// PC in R14 
							wrword(0x83de,ST);				// ST in R15 
							//ST=(ST&0xfff0);					// disable interrupts
							
							/* now load the correct workspace, and perform a branch and link to the address */
							WP=romword(0x0004);
							if ((WP>=0x8100)&&(WP<0x8400)) WP|=0x0300;
							PC=romword(0x0006);
							if ((PC>=0x8100)&&(PC<0x8400)) PC|=0x0300;
						}
					}
				}
				
				if (count>=last+max) {
					break;
				}
				SwitchToThread();	// give someone else a go ;) This prevents hogging the PC
			}
			last=count;
		}
		Counting();
	}

	time(&ENDTIME);

	debug_write("Seconds: %ld, ticks: %ld", (long)ENDTIME-STARTTIME, ticks);

	debug_write("Ending Timer Thread");
}

////////////////////////////////////////////////////////////////
// Timer calls this function
////////////////////////////////////////////////////////////////
void Counting()
{
	ticks++;
	retrace_count++;
	if (CRU[2]) {			// VDP interrupts enabled?
		interrupt_needed=1;
	}
	VDPS |= 0x80;

	if (memWnd || regWnd || asmWnd || dbgWnd) {
		draw_debug();
	}
}

////////////////////////////////////////////////////////////////
// stpcpy - strcpy then return the address of the \0
////////////////////////////////////////////////////////////////
char *stpcpy(char *dest, char *src)
{
	while (*src) {
		*dest++ = *src++;
	}
	*dest++ = *src++;
  return (dest-1);
}

////////////////////////////////////////////////////////////////
// stpncpy - strncpy then return the address of the \0
////////////////////////////////////////////////////////////////
char *stpncpy(char *dest, char *src, int n)
{
	while (n--) {
		*dest++=*src;
		if (!(*src)) {
			break;
		}
		src++;
	}
	return (dest-1);
}

