/**********************************************************************************************

     TMS5220 interface

     Written for MAME by Frank Palazzolo
     With help from Neill Corlett
     Additional tweaking by Aaron Giles
     Speech ROM support and a few bug fixes by R Nabet

***********************************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <windows.h>
#include <mmsystem.h>
#include <dsound.h>
#include <string.h>

#include "../tiemul.h"
#include "tms5220.h"
#include "5220intf.h"

extern LPDIRECTSOUND lpds;							// directsound object
extern LPDIRECTSOUNDBUFFER speechbuf;				// speech structure
extern Byte SPEECH[];								// extern to Speech ROM

/* these describe the current state of the output buffer */
static DWORD sample_pos;
static DWORD buffer_len;
static int emulation_rate;
static int INUSE=0;

extern struct TMS5220interface *intf;

struct TMS5220interface *intf;

/* static function prototypes */
static void tms5220_update (int force);

/**********************************************************************************************

     tms5220_sh_start -- allocate buffers and reset the 5220

***********************************************************************************************/

int tms5220_sh_start ()
{

	DSBUFFERDESC dsbd;
	WAVEFORMATEX pcmwf;
	UCHAR *ptr1, *ptr2;
	unsigned long len1, len2;

	if (SPEECH[0] == 0xaa)		// is the speech synth ROM present?
	{
		tms5220_speechrom = &SPEECH[0];
		tms5220_speechROMlen = 0x8000;	// Set to 32k (TODO: read from the init file)
	} else
	{
		tms5220_speechrom = NULL;
		tms5220_speechROMlen = 0;
	}

    /* determine the output sample rate and buffer size */
    buffer_len = 4000;
    emulation_rate = 8000;
    sample_pos = 0;
	
    /* start audio stream - buffer_len buffer, 8 bit, 8khz, max vol, center */
    /* note! this is unsigned mono, not signed. Routines modified accordingly */
	/* QUERY.. *is* it unsigned? */

	ZeroMemory(&pcmwf, sizeof(pcmwf));
	pcmwf.wFormatTag = WAVE_FORMAT_PCM;		// wave file
	pcmwf.nChannels=1;						// 1 channel (mono)
	pcmwf.nSamplesPerSec= emulation_rate;	// Should be 8khz
	pcmwf.nBlockAlign=1;					// 1 byte per sample * 1 channel
	pcmwf.nAvgBytesPerSec=pcmwf.nSamplesPerSec * pcmwf.nBlockAlign;
	pcmwf.wBitsPerSample=8;					// 8 bit samples
	pcmwf.cbSize=0;							// always zero;

	ZeroMemory(&dsbd, sizeof(dsbd));
	dsbd.dwSize=sizeof(dsbd);
	dsbd.dwFlags=DSBCAPS_CTRLPOSITIONNOTIFY | DSBCAPS_GETCURRENTPOSITION2;
	dsbd.dwBufferBytes=buffer_len;
	dsbd.lpwfxFormat=&pcmwf;

	speechbuf=NULL;

	if (NULL == lpds) return 1;

	if (lpds->CreateSoundBuffer(&dsbd, &speechbuf, NULL) != DS_OK)
	{
		return 1;
	}
	
	if (speechbuf->Lock(0, buffer_len, (void**)&ptr1, &len1, (void**)&ptr2, &len2, DSBLOCK_ENTIREBUFFER) == DS_OK)
	{
		// since we haven't started the sound, hopefully the second pointer is nil
		if (len2 != 0)
			MessageBox(NULL, "Failed to lock speech buffer", "Classic99 Error", MB_OK);

		// unsigned - zero the sample
		ZeroMemory(ptr1, len1);
		
		speechbuf->Unlock(ptr1, len1, ptr2, len2);
	}

	if (speechbuf->Play(0, 0, DSBPLAY_LOOPING) != DS_OK)
		debug_write("Speech DID NOT START");

    /* reset the 5220 */
    tms5220_reset ();
    tms5220_set_irq (NULL);

    return 0;
}

/**********************************************************************************************

     tms5220_sh_stop -- free buffers

***********************************************************************************************/

void tms5220_sh_stop (void)
{
    if (speechbuf)
    {
		speechbuf->Stop();
		speechbuf->Release();
	}

	speechbuf=NULL;
}



/**********************************************************************************************

     tms5220_sh_update -- update the speech chip

***********************************************************************************************/

void tms5220_sh_update (void)
{
    DWORD iWrite, iRead;
	DWORD num;
	unsigned char *ptr1, *ptr2;
	unsigned long len1, len2;

	if (quitflag)
		return;				// if we're exitting, don't do it

	if (INUSE)
		return;				// synchronization

	// fill the buffer stream if it's time 
	// first check how much space we have

	INUSE=1;

	if (speechbuf==NULL)
	{
		INUSE=0;
		return;
	}

	speechbuf->GetCurrentPosition(&iRead, &iWrite);
	if (sample_pos < iRead) 
		num=iRead-sample_pos;
	else 
		num=iRead+(buffer_len - sample_pos);
	
	num--;

	if (num>=0)		
	{
		if (speechbuf->Lock(sample_pos, num, (void**)&ptr1, &len1, (void**)&ptr2, &len2, DSBLOCK_ENTIREBUFFER) == DS_OK) // 160 samples is 20 milliseconds
		{
			tms5220_process (ptr1, len1);					// fill new buffer 
			if (len2 > 0)									// handle wraparound
				tms5220_process (ptr2, len2);				// fill new buffer 
			speechbuf->Unlock(ptr1, len1, ptr2, len2);	

			sample_pos+=num;
			while (sample_pos >= buffer_len)
				sample_pos-=buffer_len;
		}
	}

	INUSE=0;
}



/**********************************************************************************************

     tms5220_data_w -- write data to the sound chip

***********************************************************************************************/

void tms5220_data_w (int offset, int data)
{
    /* bring up to date first */
    tms5220_update (0);
    tms5220_data_write (data);
}



/**********************************************************************************************

     tms5220_status_r -- read status from the sound chip

***********************************************************************************************/

int tms5220_status_r (int offset)
{
    /* bring up to date first */
    tms5220_update (1);
    return tms5220_read ();
}



/**********************************************************************************************

     tms5220_ready_r -- return the not ready status from the sound chip

***********************************************************************************************/

int tms5220_ready_r (void)
{
    /* bring up to date first */
    tms5220_update (0);
    return tms5220_ready_read ();
}



/**********************************************************************************************

     tms5220_int_r -- return the int status from the sound chip

***********************************************************************************************/

int tms5220_int_r (void)
{
    /* bring up to date first */
    tms5220_update (0);
    return tms5220_int_read ();
}



/**********************************************************************************************

     tms5220_update -- update the sound chip so that it is in sync with CPU execution

***********************************************************************************************/

static void tms5220_update (int force)
{

// this was all about filling the output buffer, so I'll just use the
// 'sh' update routine 

//tms5220_sh_update();
}
