
/***************************************************************************
                          bitmapprinter.cpp
                          -----------------
    begin                : February, 2001
    author               : Damon Chandler
    email                : dmc27@cornell.edu
    
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//---------------------------------------------------------------------------
#pragma hdrstop

#include "bitmapprinter.h"
#pragma package(smart_init)
//-----------------------------------------------------------------------

__fastcall TBitmapPrinter::TBitmapPrinter() :
   pDIB_(NULL), pBits_(NULL), hPal_(NULL), scale_(bpsCustom)
{
   memset(&RPrint_, 0, sizeof(RECT));
}
//-----------------------------------------------------------------------

__fastcall TBitmapPrinter::~TBitmapPrinter()
{
   DeleteObject(hPal_);
   delete [] pBits_;
   delete [] reinterpret_cast<unsigned char*>(pDIB_);
}
//-----------------------------------------------------------------------

void __fastcall TBitmapPrinter::DoCalculatePrintRect(
    HDC hPrinterDC
   )
{
   // page width
   const int page_width = GetDeviceCaps(hPrinterDC, HORZRES);
   // page height
   const int page_height = GetDeviceCaps(hPrinterDC, VERTRES);
   // pixels per inch horizontally
   const int pix_inchX = GetDeviceCaps(hPrinterDC, LOGPIXELSX);
   // pixels per inch vertically
   const int pix_inchY = GetDeviceCaps(hPrinterDC, LOGPIXELSY);

   // grab a pointer to the DIB's header
   LPBITMAPINFOHEADER pHeader =
      reinterpret_cast<LPBITMAPINFOHEADER>(pDIB_);

   //
   // adjust the target rectangle according
   // to the specified scaling options...
   //
   switch (scale_)
   {
      // fit horizontally
      case bpsHorz:
      {
         RPrint_.left = RPrint_.top = 0;
         RPrint_.right = page_width;
         RPrint_.bottom =
            0.5 + abs(pHeader->biHeight) *
            (page_width / static_cast<float>(pHeader->biWidth)) *
            (pix_inchY / static_cast<float>(pix_inchX));
         break;
      }
      // fit vertically
      case bpsVert:
      {
         RPrint_.left = RPrint_.top = 0;
         RPrint_.bottom = page_height;
         RPrint_.right =
            0.5 + pHeader->biWidth *
            (page_height / static_cast<float>(abs(pHeader->biHeight))) *
            (pix_inchX / static_cast<float>(pix_inchY));
         break;
      }
      // fit to page
      case bpsPage:
      {
         RPrint_.left = RPrint_.top = 0;
         RPrint_.right = page_width;
         RPrint_.bottom = page_height;
         break;
      }
   }
}
//-----------------------------------------------------------------------

bool __fastcall TBitmapPrinter::DoCalculateBand(
    RECT& RBand, RECT& RImage
   )
{
   // compute the printable band area
   if (IntersectRect(&RBand, &RBand, &RPrint_))
   {
      // ratio of the print width to the image width
      const double ratioX =
         static_cast<double>(RPrint_.right - RPrint_.left) /
         static_cast<double>(RImage.right - RImage.left);

      // ratio of the print height to the image height
      const double ratioY =
         static_cast<double>(RPrint_.bottom - RPrint_.top) /
         static_cast<double>(RImage.bottom - RImage.top);

      //
      // compute the scaled band (this will tell
      // us where in the image to blit from)...
      //
      RImage.left = static_cast<int>(
         0.5 + (RBand.left - RPrint_.left) / ratioX
         );  
      RImage.right = RImage.left + static_cast<int>(
         0.5 + (RBand.right - RBand.left) / ratioX
         );
      RImage.top = static_cast<int>(
         0.5 + (RBand.top - RPrint_.top) / ratioY
         );
      RImage.bottom = RImage.top + static_cast<int>(
         0.5 + (RBand.bottom - RBand.top) / ratioY
         );
      return true;
   }
   return false;
}
//-----------------------------------------------------------------------

void __fastcall TBitmapPrinter::DoInternalPrintBitmap(
    HDC hPrnDC
   )
{
   if (IsRectEmpty(&RPrint_))
   {
      throw EBitmapPrinterError("Invalid target rectangle.");
   }

   if (pDIB_ && pBits_)
   {
      // extract the width and the height from the DIB's header
      const int dib_width = pDIB_->bmiHeader.biWidth;
      const int dib_height = pDIB_->bmiHeader.biHeight;

      // query the printer for the NEXTBAND capability
      // (not really needed, but better safe than sorry)
      const int query_cap = NEXTBAND;
      int do_bands = Escape(
         hPrnDC, QUERYESCSUPPORT, sizeof(int),
         reinterpret_cast<LPCSTR>(&query_cap), NULL
         );

      // test the OS version...
      OSVERSIONINFO osvi = {sizeof(OSVERSIONINFO)};
      if (GetVersionEx(&osvi))
      {
         do_bands &= (osvi.dwPlatformId != VER_PLATFORM_WIN32_NT);
      }

      //
      // proceed with printing...
      //
      bool result = true;
      if (do_bands)
      {
         RECT RBand = {0};
         while (Escape(hPrnDC, NEXTBAND, 0, NULL, &RBand))
         {
            if (IsRectEmpty(&RBand)) break;

            const int band_height =
              min(75L, RBand.bottom - RBand.top);
            const int num_bands =
              0.5 + (RBand.bottom - RBand.top) /
              static_cast<double>(band_height);

            for (int iBand = 0; iBand < num_bands; ++iBand)
            {
               RBand.top = iBand * band_height;
               RBand.bottom = RBand.top + band_height;

               RECT RImage = {0, 0, dib_width, dib_height};
               if (DoCalculateBand(RBand, RImage))
               {
                  // set the stretching mode to preserve colors
                  SetStretchBltMode(hPrnDC, COLORONCOLOR);

                  // support palettized printers
                  bool palettized =
                     (GetDeviceCaps(hPrnDC, RASTERCAPS) & RC_PALETTE);
                  if (palettized)
                  {
                     hPal_ = SelectPalette(hPrnDC, hPal_, FALSE);
                     RealizePalette(hPrnDC);
                  }

                  const int dst_width = RBand.right - RBand.left;
                  const int dst_height = RBand.bottom - RBand.top;
                  const int src_width = RImage.right - RImage.left;
                  const int src_height = RImage.bottom - RImage.top;

                  result &= GDI_ERROR !=
                     StretchDIBits(
                        // draw to the printer's DC
                        hPrnDC,
                        // dest rect
                        RBand.left, RBand.top,
                        dst_width, dst_height,
                        // source rect
                        RImage.left, dib_height - RImage.bottom,
                        src_width, src_height,
                        // source bits
                        pBits_,
                        // source DIB
                        pDIB_,
                        // DIB's color table contains RGBQUADs
                        DIB_RGB_COLORS,
                        // copy source
                        SRCCOPY
                        );

                  // restore the original palette
                  if (palettized)
                  {
                     hPal_ = SelectPalette(hPrnDC, hPal_, TRUE);
                  }
               }
            }
         }
      }
      else // no banding--draw the DIB in its entirety
      {
         // support palettized printers
         bool palettized = 
            (GetDeviceCaps(hPrnDC, RASTERCAPS) & RC_PALETTE);
         if (palettized)
         {
            hPal_ = SelectPalette(hPrnDC, hPal_, FALSE);
            RealizePalette(hPrnDC);
         }
         
         // set the stretching mode to preserve colors
         SetStretchBltMode(hPrnDC, COLORONCOLOR);

         result &= GDI_ERROR !=
            StretchDIBits(
               // draw to the printer's DC
               hPrnDC,
               // dest rect
               RPrint_.left, RPrint_.top,
               RPrint_.right - RPrint_.left,
               RPrint_.bottom - RPrint_.top,
               // source rect (entire DIB)
               0, 0, dib_width, dib_height,
               // source bits
               pBits_,
               // source DIB
               pDIB_,
               // DIB's color table contains RGBQUADs
               DIB_RGB_COLORS,
               // copy source
               SRCCOPY
               );

         // restore the original palette
         if (palettized)
         {
            hPal_ = SelectPalette(hPrnDC, hPal_, TRUE);
         }
      }

      // report errors
      if (!result)
      {
         throw EBitmapPrinterError("Error printing DIB.");
      }
   }
   // report errors
   else
   {
      throw EBitmapPrinterError("Error getting DIB info.");
   }
}
//-----------------------------------------------------------------------

void __fastcall TBitmapPrinter::DoPrintBitmap(
    HDC hPrnDC
   )
{
   if (!pDIB_)
   {
      throw EBitmapPrinterError("No bitmap defined.");
   }
   if (!hPrnDC)
   {
      throw EBitmapPrinterError("Invalid printer DC.");
   }

   // compute the rectangle to print to
   DoCalculatePrintRect(hPrnDC);
   
   // print the bitmap
   DoInternalPrintBitmap(hPrnDC);
}
//-----------------------------------------------------------------------

void __fastcall TBitmapPrinter::DoSetRect(
    RECT RPrint
   )
{
   RPrint_ = RPrint;
   scale_ = bpsCustom;   
}
//-----------------------------------------------------------------------

void __fastcall TBitmapPrinter::DoSetScale(
    TBitmapPrinterScale scale
   )
{
   if (scale >= bpsHorz && scale <= bpsCustom)
   {
      scale_ = scale;
   }
   else throw EBitmapPrinterError("Invalid scale.");
}
//-----------------------------------------------------------------------

void __fastcall TBitmapPrinter::DoSetBitmap(
    const Graphics::TBitmap* Bitmap
   )
{
   DeleteObject(hPal_);
   hPal_ = NULL;
   
   delete [] pBits_;
   pBits_ = NULL;

   delete [] reinterpret_cast<unsigned char*>(pDIB_);
   pDIB_ = NULL;

   if (Bitmap)
   {
      pDIB_ = DoCreateDIB(*Bitmap, pBits_);
      hPal_ = CopyPalette(
         const_cast<Graphics::TBitmap*>(Bitmap)->Palette
         );
   }
}
//-----------------------------------------------------------------------

LPBITMAPINFO __fastcall TBitmapPrinter::DoCreateDIB(
    const Graphics::TBitmap& Bitmap,
    unsigned char*& pBits
   )
{
   //
   // use the GetDIBSizes() function to
   // gauge the size of the DIB's parts
   //
   size_t header_ct_size = 0; // BCB1 requires 'int' instead of 'size_t'
   size_t image_size = 0;     // BCB1 requires 'int' instead of 'size_t'
   GetDIBSizes(
      const_cast<Graphics::TBitmap&>(Bitmap).Handle,
      header_ct_size, image_size
      );

   // compute the total size
   const size_t total_size =
     header_ct_size + image_size;

   if (total_size > 0)
   {
      // memory for the header
      // and the color table
      unsigned char* pHeaderAndCT = new unsigned char[header_ct_size];

      // memory for the pixels
      pBits = new unsigned char[image_size];

      try
      {
         //
         // use the GetDIB() function
         // to get the header, color table,
         // and a copy of Bitmap's pixels
         //
         bool got_dib =
            GetDIB(
               const_cast<Graphics::TBitmap&>(Bitmap).Handle,
               const_cast<Graphics::TBitmap&>(Bitmap).Palette,
               pHeaderAndCT, pBits
               );
         if (got_dib)
         {
            return reinterpret_cast<LPBITMAPINFO>(pHeaderAndCT);
         }
         throw EBitmapPrinterError("Error creating DIB.");
      }
      catch (...)
      {
         // clean up
         delete [] pHeaderAndCT;
         delete [] pBits;
         throw;
      }
   }
   return NULL;
}
//-----------------------------------------------------------------------



