/*
    This source code was written by Jeffrey J. Peters (c) 1999.
    It is hereby released into the public domain, for use by anyone
    for anything.

*/

//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop
#include <direct.h>
#include <dos.h>
#include <stdio.h>
#include <io.h>

#include "fmain.h"
#include "fabout.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma link "PCRegExp"
#pragma resource "*.dfm"
TMainForm *MainForm;
//---------------------------------------------------------------------------
__fastcall TMainForm::TMainForm(TComponent* Owner)
        : TForm(Owner)
{
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::FileExitItemClick(TObject *Sender)
{
  Close();
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::SearchButtonClick(TObject *Sender)
{
  // Prepare the two buttons so that Cancel and Search are mutually
  // exclusive.
  SearchButton->Enabled = false;
  CancelButton->Enabled = true;
  SearchAnimate->Visible = true;
  SearchAnimate->Active = true;

  BeginGrep();
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::CancelButtonClick(TObject *Sender)
{
  // Keep the buttons mutually exclusive.
  SearchButton->Enabled = true;
  CancelButton->Enabled = false;
  SearchAnimate->Visible = false;
  SearchAnimate->Active = false;

  // Blank out the status bar text
  StatusBar->SimpleText = "";
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::ProcessFile(AnsiString &fname)
{
  // This function takes a file name (possibly with a path too) and opens
  // the file and performs the actual grep process on each line of the file.
  FILE *fp;
  char buf[512];
  int line;

  // Update the status bar with the name of the file we're analyzing
  StatusBar->SimpleText = fname;

  // Now open the file
  fp = fopen(fname.c_str(), "rt");

  // If it failed, give a good error report to the user
  if (!fp)
  {
    MessageDlg(AnsiString().sprintf("Error: %s while opening file: %s",
                            sys_errlist[errno], fname.c_str()),
                            mtError, TMsgDlgButtons()<<mbOK, 0);
    return;
  }

  // The line numbers in files are one based
  line = 1;

  // read each line until end of file...
  while (!feof(fp))
  {
    if (!fgets(buf, sizeof(buf), fp))
      continue;

    // Call PCRE to determine if there is a hit on this line
    if (PCRE->Execute(buf) == true)
    {
      // If so, update our ListView
      AddHit(fname, buf, line);
    }
    line++;
  }

  // Close the file
  fclose(fp);

  // Check to see if the user has suggested that we terminate the current
  // search that is in progress.
  if (CancelButton->Enabled == false)
    // We'll signal this situation by raising an exception
    throw new Exception("User Break");

  // Give the UI some cycles so that everything looks smoothe.
  Application->ProcessMessages();
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::AddHit(AnsiString &fname, char *buf, int line)
{
  // Update our ListView with the info about this hit

  char *p;
  TListItem *li;

  // Remove any \r and \n chars from the string before they're added to the
  // ListView
  while ((p = strrchr(buf, '\n')) != NULL)
    p[0] = '\0';

  while ((p = strrchr(buf, '\r')) != NULL)
    p[0] = '\0';

  // Add a new element to the ListView
  li = ResultListView->Items->Add();

  // This will contain the string of the first column (file name)
  li->Caption = fname;

  // Now add the text for the second column (line #)
  li->SubItems->Add(AnsiString().sprintf("%d", line));

  // And finally the text for the third column (string) which is the text of
  // the entire line where we had the match.
  li->SubItems->Add(buf);
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::ProcessDir(AnsiString &fname)
{
  // This routine iterates through the specified directory looking for
  // files to process and possibly other subdirectories.

  ffblk ff;
  int done;
  AnsiString path, filespec;
  AnsiString item;

  // Extract and save the path information that was passed in
  path = ExtractFilePath(fname);

  // Extract and save the filespec (wildcards, or filename)
  filespec = ExtractFileName(fname);

  // Now, first look for all the files that match in this directory
  done = findfirst(fname.c_str(),&ff,FA_NORMAL);

  while(!done)
  {
    // set up the full path of this item
    item = AnsiString().sprintf("%s%s",path.c_str(), ff.ff_name);

    // Now process this file
    ProcessFile(item);

    // Cycle to the next file in the directory
    done = findnext(&ff);
  }

  // If we're in subdir mode, search for all the subdirs here
  if (OptionsSubdirsItem->Checked)
  {
    done = findfirst((path+AnsiString("*.*")).c_str(),&ff,FA_DIREC);

    while(!done)
    {
      // set up the full path of this item
      item = AnsiString().sprintf("%s%s\\",path.c_str(), ff.ff_name);

      // Skip everything that's not a directory.  Also, always skip
      // the "." and ".." directories
      if (ff.ff_attrib & FA_DIREC &&
          strcmp(ff.ff_name, ".")  != 0 &&
          strcmp(ff.ff_name, "..") != 0) // if this is a subdir and it's not "."
                                         // or ".."
      {

        // Add in the filespec for the subdir and process it
        item.sprintf("%s", filespec.c_str());
        ProcessDir(item);
      }

      // Cycle to the next directory
      done = findnext(&ff);
    }
  }
}
//---------------------------------------------------------------------------

void __fastcall TMainForm::BeginGrep(void)
{
  // This function begins the searching proceedure

  AnsiString Error;
  int ErrorOffset;
  char *p, *q, *str;

  // Update the pattern history
  if (PatternCombo->Items->IndexOf(PatternCombo->Text) == -1)
    PatternCombo->Items->Add(PatternCombo->Text);

  // Update the filespec history
  if (FileCombo->Items->IndexOf(FileCombo->Text) == -1)
    FileCombo->Items->Add(FileCombo->Text);


  // Set the PCRE flag for case sensitivity
  if (OptionsCaseSensitiveItem->Checked)
    PCRE->CompileFlags = PCRE->CompileFlags >> pcfCASELESS;
  else
    PCRE->CompileFlags = PCRE->CompileFlags << pcfCASELESS;

  // Set the PCRE pattern string
  PCRE->PatternString = PatternCombo->Text;

  // Cause the PCRE component to compile the pattern to an internal
  // representation that can then be used to test against strings that
  // we'll read out of the specified files.
  if (!PCRE->Compile(Error, ErrorOffset))
  {
    // If an error occurs, notify the user and skip this search.
    MessageDlg(Error + AnsiString(" occured while compiling regexp pattern: ") +
               PatternCombo->Text, mtError, TMsgDlgButtons()<<mbOK, 0);
    return;
  }

  // Erase previous results in the ListView
  ResultListView->Items->Clear();

  // We wrap all of this up in a try/__finally so that we can delete str
  // properly no matter when or how we exit.
  try
  {
    // Allocate a new string so that we can bounce our way through it,
    // writing 0's into the first whitespace chars that seperate each term
    str = new char[FileCombo->Text.Length()+2];

    // Copy the file name into our new memory
    strcpy(str, FileCombo->Text.c_str());

    // Add a double null char to the end so we can detect the real end of
    // the string without walking off the end.
    str[strlen(str)+1] = 0;

    q = p = str;

    // iterate through all the terms listed which are seperated by whitespace
    // This means that you can specify filespecs like:
    //   *.cpp *.h
    // or
    //   myprog.* *.h *.hpp
    while (p && *p)
    {
      // At this point, q points to the beginning of the current term

      // Find the first whitespace (if any) after the current term
      p = strchr(q, ' ');
      if (p)
      {
        // end the current term at the first space following it
        *p = 0;
      }

      // Guard against the exception that is thrown when the Cancel button is
      // pressed
      try
      {
        // Now process the current term
        ProcessDir(AnsiString(q));
      }
      catch (Exception *e)
      {
        if (e->Message == "User Break")
        {
          // Stop the animation and reset the buttons.
          CancelButton->Click();
          StatusBar->SimpleText = e->Message;
        }
        else
        {
          // Stop the animation and reset the buttons.
          CancelButton->Click();

          StatusBar->SimpleText = e->Message;

          // re-throw the exception
          throw;
        }
      }

      // If there are subsequent terms, bump p past the null char
      if (p)
        p++;

      // Skip any repeating whitespace so that "*.cpp     *.h" works
      while (p && *p && *p == ' ')
        p++;

      // Now set q to the head of the next term (if any)
      q = p;
    }
  }
  __finally
  {
    // release our extra memory
    delete str;
  }

  // Stop the animation and reset the buttons.
  CancelButton->Click();
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::OptionsMenuClick(TObject *Sender)
{
  // This handler will implement toggelable check marks in the options menu
  TMenuItem *mi = ((TMenuItem*)Sender);

  mi->Checked = !mi->Checked;
}
//---------------------------------------------------------------------------
void __fastcall TMainForm::ResultListViewDblClick(TObject *Sender)
{
  // This function is called when an item in the ListView is double-clicked
  // We'll simply put up a message box that contains the filename and the
  // line number in it.
  // Eventually it would be nice to spawn the user's editor of choice with
  // this file and position to the specified line number.

  TListItem *li;
  AnsiString line;
  li = ResultListView->Selected;
  if (li)
  {
    line = li->SubItems->Strings[0];
    ShowMessage(AnsiString("File: ") + li->Caption + "\nLine: " + line);
  }
}
//---------------------------------------------------------------------------

void __fastcall TMainForm::About1Click(TObject *Sender)
{
  AboutBox->ShowModal();         
}
//---------------------------------------------------------------------------

