#ifndef ListManagerH
#define ListManagerH

// ====================================================================
// List Manager Class
// ====================================================================
template <class T> class ListViewManager : public TComponent
{
  protected:
  TLVDeletedEvent     OldOnDelete;
  TLVCompareEvent     OldOnCompare;
  TLVColumnClickEvent OldOnColumnClick;
  int iCurrentSort;

  public:

  TListView* lvList;

  __fastcall ListViewManager(TListView* pList);

  TListItem* AddObject(T* pObj);
  void RefreshObject(T* pObj);
  void RefreshItem(TListItem* item);

  void __fastcall OnListViewDeletion(TObject *Sender, TListItem *Item);

  void __fastcall OnListViewColumnClick(TObject *Sender, TListColumn *Column);

  void __fastcall OnListViewCompare(TObject *Sender, TListItem *Item1,
                                    TListItem *Item2, int Data, int &Compare);
};

// =========================================================================
// ListViewManager Implementation
// =========================================================================
template <class T>
__fastcall ListViewManager<T>::ListViewManager(TListView* pList)
  : TComponent(pList)
{
  lvList = pList;
  iCurrentSort = -1;

  // Save previous event handlers defined in DFM
  OldOnDelete = lvList->OnDeletion;
  OldOnCompare = lvList->OnCompare;
  OldOnColumnClick = lvList->OnColumnClick;

  // Assign new event handlers
  lvList->OnDeletion    = OnListViewDeletion;
  lvList->OnCompare     = OnListViewCompare;
  lvList->OnColumnClick = OnListViewColumnClick;
}

// This function updates the column text of the specified object
template<class T>
void ListViewManager<T>::RefreshObject(T* pObj)
{
  TListItem* pItem = FindObject(pObj);
  if (pItem) {
    RefreshItem(pItem);
  }
}

// Function RefreshItem() updates the columns of text
// and the icon for an object in the listview.
template<class T>
void ListViewManager<T>::RefreshItem(TListItem* item)
{
  if (item) {
    T* pObject = (T*)(item->Data);
    if (pObject) {
      item->SubItems->Clear();
      for (int i = 0; i < lvList->Columns->Count; i++) {
        if (i == 0) {
          item->Caption = pObject->GetColumnString(i);
        }
        else {
          item->SubItems->Add(pObject->GetColumnString(i));
        }
      }
      item->ImageIndex = pObject->GetImageIndex();
    }
  }
}

// Function AddItem() adds an object to the listview and
// automatically sets the text of all the columns.
template <class T>
TListItem* ListViewManager<T>::AddObject(T* pObject)
{
  TListItem* item = lvList->Items->Add();
  item->Data = (TObject*)pObject;
  RefreshItem(item);
  return item;
}

// This event ensures that the objects will get deleted
// when the listview is deleted.
template <class T>
void __fastcall ListViewManager<T>::OnListViewDeletion(TObject *Sender, TListItem *Item)
{
  T* pObject = (T*)(Item->Data);
  if (pObject) {
    delete pObject;
    Item->Data = NULL;
  }
}

// This event handles automatic sorting.
template <class T>
void __fastcall ListViewManager<T>::OnListViewColumnClick( TObject *Sender,
                                                       TListColumn *Column)
{
  int iColumn = 0;
  for (int i = 0; i < lvList->Columns->Count; i++) {
    if (lvList->Columns->Items[i] == Column) {
      iColumn = i;
      break;
    }
  }

  lvList->CustomSort(NULL, iColumn);

  if (iCurrentSort == iColumn) {
    iCurrentSort = -1;
  }
  else {
    iCurrentSort = iColumn;
  }

  if (OldOnColumnClick) {
    OldOnColumnClick(Sender, Column);
  }
}

// This event sorts columns by doing a string comparison.
template <class T>
void __fastcall ListViewManager<T>::OnListViewCompare( TObject *Sender,
                                                TListItem *Item1,
                                                TListItem *Item2,
                                                int Data,
                                                int &Compare)
{
  String sValue1;
  String sValue2;
  if (Data == 0) {
    sValue1 = Item1->Caption;
    sValue2 = Item2->Caption;
  }
  else {
    sValue1 = Item1->SubItems->Strings[Data - 1];
    sValue2 = Item2->SubItems->Strings[Data - 1];
  }

  Compare = sValue1.AnsiCompareIC(sValue2);

  if (iCurrentSort == Data) {
    Compare = -Compare;
  }
}

#endif


