/****************************************************************************
 * Copyright (c) 2000 - 2001
 * by Cristobal J. Gallegos
 *
 * The above copyright notice must appear in any and all copies of this code
 * and in all supporting documentation.  I make no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 ***************************************************************************/
#ifndef DiskListH
#define DiskListH
//---------------------------------------------------------------------------
#include "TempList.h"//Contains TssTempList.
//------------------NOTES:---------------------------------------------------
//If detailed failure information is not desired, SetFileName() can be
//shortened to the single line:
//    pFS = new TFileStream(NewFileNameStr, fmCreate | fmShareExclusive);
//WARNING:  If reading object information from a disk file, objects are
//auto-created regardless of the value of F_ALLOW_CREATE.  This means that
//the value of F_ALLOW_DELETE must be considered carefully.
//---------------------------------------------------------------------------
//First version.  See notes in TempList.h for explanation.
template <class T> class TssDiskList : System::TObject
{
private://Not needed, just here for clarity.
  TssDiskList();//Constructor not defined.
  TssDiskList(const TssDiskList&);//Copy Constructor not defined.
};//End of TssDiskList (first version).
//---------------------------------------------------------------------------
//Second version.  See notes in TempList.h for explanation.
template <class T> class TssDiskList< T * > : public TssTempList< T * >
{
  private:
    String FFileName;

    void SetFileName(const String NewValue);

  public:
    TssDiskList(bool AllowCreation, bool AllowDeletion, bool AllowNULL) :
      TssTempList< T * >(AllowCreation, AllowDeletion, AllowNULL){}
      //The initializer list is the only way to pass this constructor's
      //parameters on to the base class' constructor.

  /*
    No need for destructor.  TObject's destructor is virtual.  This means
    that the TssTempList (which is derived from TObject) destructor will
    be called whenever a TssDiskList instance is destroyed.
  */

  __property String FileName = {read=FFileName, write=SetFileName};

  bool ReadFromFile();

  bool WriteToFile();
};//End of TssDiskList.
//---------------------------------------------------------------------------
//NOTE:  The testing of drive name validity below will throw an exception
//if the string passed in contains a network drive name and the computer
//is not currently connected to the network.  This function cannot trap
//single backslashes contained in file names without drive names.  In this
//situation, the creation of NewNameStr simply concatenates what comes
//before and after the single backslash.  Assumes all error messages will
//be seen by the end user.
template <class T>
        void TssDiskList< T * >::SetFileName(const String NewFileNameStr)
{
  if(FileExists(NewFileNameStr))//NewFileNameStr must be good.
  {
    FFileName = NewFileNameStr;
    return;
  }//End of if(FileExists(NewFileNameStr)).

  //If the file doesn't exist, test if NewFileNameStr is OK.  Provide a
  //descriptive message if it isn't.
  TFileStream * pFS = NULL;
  try
  {
    if(NewFileNameStr.IsEmpty())
        throw EInvalidOperation
                ("The specified file name cannot be blank.");
    //Test the validity of any drive names that may have been passed in.
    String TempStr = ExtractFileDrive(NewFileNameStr);//Need in GetDriveType().
    if(TempStr.IsEmpty())
      TempStr = NewFileNameStr;
    else//There is a drive name to test.
    {
      switch(::GetDriveType(TempStr.c_str()))//Test for invalid drive name.
      {
        //Can't use break statements in switch due to exception generation.
        //The compiler warns of unreachable code.  Allow read/write to
        //DRIVE_REMOVABLE, DRIVE_FIXED, DRIVE_REMOTE, or DRIVE_RAMDISK.
        //Also allow DRIVE_CDROM even though it can't be written to.
        case 0://The drive type cannot be determined.
        case 1://The root directory does not exist.
          throw EInvalidOperation
                  ("The file name contains invalid drive specifier.");
      }//End of switch(::GetDriveType(TempStr.c_str())).
      if(NewFileNameStr[3] != '\\')
        //Have to specifically test for this.  ExtractFileDrive() doesn't
        //care if there is no double backslash after the drive name.
        throw EInvalidOperation//User/programmer warning.
          ("The file name contains invalid path information.");
      //Eliminate the drive name to help improve the accuracy of what follows.
      //Eliminating the first backslash just for efficiency.
      TempStr = NewFileNameStr.SubString(4, NewFileNameStr.Length());
        //Excludes previously tested characters.  Length() is too much but
        //over-runs char array safely.  Ensures complete copy.
    }//End of else.

    //Test the new filename for invalid file name characters.
    int StrLength = TempStr.Length();//Avoids continual re-evaluating.
    for(int i = 1; i <= StrLength; i++)
    {
      if(TempStr.IsDelimiter("/:*?\"<>|", i))//Checks for a second colon.
        throw EInvalidOperation("The file name contains illegal characters.");
    }//End of for(int i = 1; i <= StrLength; i++).

    //Test for a valid file name and extension.
    if(ExtractFileName(NewFileNameStr).IsEmpty())
      throw EInvalidOperation
              ("The file name does not contain an actual valid file name.");
    if(ExtractFileExt(NewFileNameStr).IsEmpty())
      throw EInvalidOperation
              ("The file name does not contain a valid file extension.");

    //One last overall test.  Try to create the file temporarily.
    pFS = new TFileStream(NewFileNameStr, fmCreate | fmShareExclusive);
      //Throws an Exception if NewFileNameStr is no good.  Generates it's
      //own descriptive message string.

    //Assuming nothing has thrown an exception so far... (whew!)
    FFileName = NewFileNameStr;
  }//End of try.
  catch(Exception &E)
  {
    Beep(1000, 1000);
    ShowMessage("The current attempt to change the file name cannot be "
            "completed.  The process was halted with a message of: \""
            + E.Message + "\"");
  }//End of catch.
  if(pFS != NULL)
  {
    delete pFS;
    DeleteFile(NewFileNameStr);//File was just created above.
  }//End of if(pFS != NULL).
}//End of TssDiskList::SetFileName().
//---------------------------------------------------------------------------
//This method reads the information written to the end of the specified file
//by WriteToFile().  It then performs validity checks using that information.
//If the checks pass, all pointers (and possibly the objects they represent)
//are cleared from memory.  The correct number of actual objects are
//allocated, then object information is read from disk.  NULL pointers are
//added to the list on the fly as needed.
template <class T> bool TssDiskList< T * >::ReadFromFile()
{
  bool Result = false, NULLRecordFlag;//Initialized later before use.
  int RecordCount, TotalCount, TypeNameLength;//Initialized before use later.
  char * TypeNameBuf = NULL;//Holds object type name.
  TFileStream * pFS = NULL;
  try
  {
    if(FFileName.IsEmpty())
      throw EInvalidOperation//Programmer warning.  Set HelpContext.
              ("The FFileName data member has not been initialized.", 1);
    pFS = new TFileStream(FileName, fmOpenRead | fmShareExclusive);
    pFS->Position = pFS->Size - (3 * sizeof(int));
    pFS->Read(&TypeNameLength, sizeof(int));
    pFS->Read(&TotalCount, sizeof(int));
    pFS->Read(&RecordCount, sizeof(int));

    //Do error checking.
    if(TypeNameLength == 0)
      throw EInvalidOperation//End user warning.  Reset HelpContext.
              ("The file contains invalid object information.", 0);

    if(TotalCount < 0 || TotalCount > MaxListSize//Avoids EListError excptn.
            || RecordCount < 0 || RecordCount > TotalCount)
      //MaxListSize is defined in Classes.pas as MaxInt / 16 (134217727).
      throw EInvalidOperation//End user warning.  Reset HelpContext.
              ("The file contains corrupted object information.", 0);

    if(!F_ALLOW_NULL && (TotalCount > RecordCount))
      throw EInvalidOperation//Programmer warning.  Set HelpContext.
              ( "The file contains NULL record information.  This "
                "instance of TssDiskList cannot store NULL records.", 1);

    TypeNameBuf = new char[TypeNameLength];//No need to initialize.
    pFS->Position = pFS->Size - (3 * sizeof(int)) - TypeNameLength;
    pFS->Read(TypeNameBuf, TypeNameLength);
    if(ObjectType != AnsiString(TypeNameBuf))
      //The following exception can be seen by the end user.
      throw EInvalidOperation//End user warning.  Reset HelpContext.
              ("The file contains the wrong type of object information.", 0);

    //Everything must be OK.  Continue on.
    pFS->Position = 0;//Must move back to the start of the file stream.
    Clear();//Empty list.
    AllocateNewObjects(RecordCount);//Allocates correct number of objects.
    for(int i = 0; i < TotalCount; i++)
    {
      pFS->Read(&NULLRecordFlag, sizeof(bool));
      if(NULLRecordFlag)
        Insert(i, NULL);//Insert NULL pointers where needed.
      else
        pFS->Read(FList->Items[i], sizeof(T));
    }//End of for(int i = 0; i < RecordCount; i++).
    Result = true;
  }//End of try.
  catch(EInvalidOperation &E)
  {
    //Catches problems and reports to programmer or end user.
    String MsgStr;
    if(E.HelpContext == 1)//Programmer warning.
      MsgStr =  "The current call to TssDiskList->ReadFromFile() "
                "has been halted.  ";
    MsgStr += "The disk file: \"" + FFileName + "\" cannot be opened.  "
              "The process was halted with an error message of:  \""
              + E.Message + "\"";
    Beep(1000, 1000);
    ShowMessage(MsgStr);
  }//End of catch(EInvalidOperation &E).
  catch(...){}//Trap filestream problems.  Return value does the talking.
  if(pFS != NULL)
    delete pFS;
  if(TypeNameBuf != NULL)
    delete[] TypeNameBuf;
  return Result;
}//End of TssDiskList::ReadFromFile().
//---------------------------------------------------------------------------
//This method prefixes every record written to the file with a tag that
//describes whether the record that follows it represents a NULL pointer
//that was in the list that wrote the file, or whether the record that
//follows is a valid (non-NULL) record.  It also tracks how many valid
//(non-NULL) records are written and how many records are written total.
//It appends this information to the end of the file produced.  This
//information is then used in ReadFromFile().
template <class T> bool TssDiskList< T * >::WriteToFile()
{
  bool Result = false;
  TFileStream * pFS = NULL;//Writes to disk file.
  char * TypeNameBuf = NULL;//Stores TypeName chars.
  if(FFileName.IsEmpty())
  {
    Beep(1000, 1000);
    ShowMessage("TssDiskList->WriteToFile() cannot be completed as the "
    "private FFileName data member has not been initialized.  The current "
    "call to WriteToFile() will now abort.");
  }//End of if(FFileName.IsEmpty()).
  else
  {
    bool NULLRecordFlag;//Initialized later before use.
    T * pT = NULL;
    //Next three integers provide true buffers for pFS->Write to read from.
    const int TypeNameLength = ObjectType.Length() + 1;//Can set now.
    const int TotalRecords = FList->Count;//Can set now.
    int ValidRecords = 0;//Initialize now for incrementing later.
    try
    {
      TypeNameBuf = new char[TypeNameLength];
      ::ZeroMemory(TypeNameBuf, TypeNameLength);//Ensure terminator.
      strncpy(TypeNameBuf, ObjectType.c_str(), TypeNameLength);
      pFS = new TFileStream(FFileName, fmCreate | fmShareExclusive);
      for(int i = 0; i < FList->Count; i++)
      {
        pT = Get(i);
        if(pT == NULL)
        {
          NULLRecordFlag = true;//Marks a NULL pointer.
          pFS->Write(&NULLRecordFlag, sizeof(bool));//Write the tag to file.
        }//End of if(pT == NULL).
        else
        {
          NULLRecordFlag = false;//Marks a valid record.
          pFS->Write(&NULLRecordFlag, sizeof(bool));//Write the tag to file.
          pFS->Write(pT, sizeof(T));//Write the record.
          ++ValidRecords;//Track valid record count.
        }//End of else.
      }//End of for(int i = 0; i < FList->Count; i++).

      pFS->Write(TypeNameBuf, TypeNameLength);//Write object type name.
      pFS->Write(&TypeNameLength, sizeof(int));//Write length of type name.
      pFS->Write(&TotalRecords, sizeof(int));//Write total number of records.
      pFS->Write(&ValidRecords, sizeof(int));//Write number of valid records.
      Result = true;
    }//End of try.
    catch(...)//Catch any and all problems.
    {
      if(pFS != NULL)//Write known invalid conditions to file.
      {
        ValidRecords = 0;//Set to safe value.
        pFS->Write(&ValidRecords, sizeof(int));//Write length of object name.
        pFS->Write(&ValidRecords, sizeof(int));//Write for TotalRecords.
        pFS->Write(&ValidRecords, sizeof(int));//Write for ValidRecords.
      }//End of if(pFS != NULL).
    }//End of catch(...).
  }//End of else.
  if(pFS != NULL)
    delete pFS;
  if(TypeNameBuf != NULL)//Release resources.
    delete[] TypeNameBuf;
  return Result;
}//End of TssDiskList::WriteToFile().
//---------------------------------------------------------------------------
#endif

