/****************************************************************************
 * Copyright (c) 2000 - 2001
 * by Cristobal J. Gallegos
 *
 * The above copyright notice must appear in any and all copies of this code
 * and in all supporting documentation.  I make no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 ***************************************************************************/
#ifndef RegListH
#define RegListH
//---------------------------------------------------------------------------
#include <registry.hpp>//Needed for TRegistry.
#include "TempList.h"//Contains TssTempList.
//------------------NOTES:---------------------------------------------------
//The RootKey property allows direct write access to the private FRootKey
//data member.  The compiler handles the verification of the HKEY parameter
//if it is assigned (can't convert "int" or "unsigned long" to "void *").
//The RegKey property also allows direct write access to the private FRegKey
//data member.  Any printable character is allowed for use in a registry
//value name (unlike a disk file's name).  Let the user beware!  TRegistry's
//ReadBinaryData() and TRegistry->ReadInteger() are case insensitive.
//WARNING:  If reading object information from the registry, objects are
//auto-created regardless of the value of F_ALLOW_CREATE.  This means that
//the value of F_ALLOW_DELETE must be considered carefully.
//---------------------------------------------------------------------------
//First version.  See notes in TempList.h for explanation.
template <class T> class TssRegList : System::TObject
{
private://Not needed, just here for clarity.
  TssRegList();//Constructor not defined.
  TssRegList(const TssRegList&);//Copy Constructor not defined.
};//End of TssRegList (first version).
//---------------------------------------------------------------------------
//Second version.  See notes in TempList.h for explanation.
template <class T> class TssRegList< T * > : public TssTempList< T * >
{
private:

  const String OBJECT_VALUE_NAME,
          RECORDCOUNT_VALUE_NAME, TOTCOUNT_VALUE_NAME, TYPENAME_VALUE_NAME;

  String FRegKey;

  HKEY FRootKey;

public:
  TssRegList(bool AllowCreation, bool AllowDeletion, bool AllowNULL) :
    //Initialize TssRegList const data members.
    OBJECT_VALUE_NAME("Record #"), RECORDCOUNT_VALUE_NAME("RecordCount"),
    TOTCOUNT_VALUE_NAME("TotalCount"), TYPENAME_VALUE_NAME("TypeName"),
    TssTempList< T * >(AllowCreation, AllowDeletion, AllowNULL)
      //The initializer list is the only way to pass this constructor's
      //parameters on to the base class' constructor.
  {
    FRootKey = NULL;
  }//End of TssRegList constructor.

/*
  No need for destructor.  TObject's destructor is virtual.  This means
  that the TssTempList (which is derived from TObject) destructor will
  be called whenever a TssRegList instance is destroyed.
*/

__property String RegKey = {read=FRegKey, write=FRegKey};

__property HKEY RootKey = {read=FRootKey, write=FRootKey};

bool ReadFromRegistry();

bool WriteToRegistry();
};//End of TssRegList.
//---------------------------------------------------------------------------
//This method reads the object summary information written to the Registry
//by WriteToRegistry().  It then performs validity checks using that
//information.  If the checks pass, all pointers (and possibly the objects
//that they represent) are cleared from memory.  The correct number of actual
//objects are then allocated, then object information is read from the
//Registry.  NULL pointers are added to the list on the fly as needed.
template <class T> bool TssRegList< T * >::ReadFromRegistry()
{
  bool Result = false;
  int RecordCount, TotalCount, DataSize;
  String TypeNameStr;
  TRegistry * pReg = NULL;
  try
  {
    if(FRegKey.IsEmpty())
      throw EInvalidOperation//Programmer warning.
              ("The FRegKey data member has not been initialized.", 1);
    pReg = new TRegistry;
    if(FRootKey != NULL)
      pReg->RootKey = FRootKey;
    if(pReg->OpenKeyReadOnly(FRegKey))
    {
      TypeNameStr = pReg->ReadString(TYPENAME_VALUE_NAME);
      RecordCount = pReg->ReadInteger(RECORDCOUNT_VALUE_NAME);
      TotalCount = pReg->ReadInteger(TOTCOUNT_VALUE_NAME);

      //Do error checking.
      if(TypeNameStr.IsEmpty())
        throw EInvalidOperation//End user warning.
              ("The data source contains invalid object information.", 0);

      if(TotalCount < 0 || TotalCount > MaxListSize
              || RecordCount < 0 || RecordCount > TotalCount)
        throw EInvalidOperation//End user warning.
              ("The data source contains corrupted object information.", 0);

      if(!F_ALLOW_NULL && (TotalCount > RecordCount))
        throw EInvalidOperation//Programmer warning.
              ("The data source contains NULL record information.  "
               "This instance of TssRegList cannot store NULL records.", 1);

      if(ObjectType != TypeNameStr)
        throw EInvalidOperation//End user warning.
              ("The specified data source "
               "contains the wrong type of object information.", 0);

      //Everything must be OK.  Continue on.
      Clear();
      AllocateNewObjects(RecordCount);//Allocates correct number of objects.
      for(int i = 0; i < TotalCount; i++)
      {
        //Find the size of the next object to be read.
        DataSize = pReg->ReadBinaryData//Must pass NULL for the Buffer param.
                (OBJECT_VALUE_NAME + IntToStr(i), NULL, sizeof(T));
        if(DataSize == 0)//Insert NULL pointers where needed.
          Insert(i, NULL);
        else//Read object data to the pointer's address.
          pReg->ReadBinaryData(OBJECT_VALUE_NAME
                  + IntToStr(i), FList->Items[i], sizeof(T));
      }//End of for(int i = 0; i < RecordCount; i++).
      Result = true;
    }//End of if(pReg->OpenKeyReadOnly(FRegKey)).
  }//End of try.
  catch(EInvalidOperation &E)
  {
    //Catches problems and reports to programmer or end user.
    String MessageStr;
    if(E.HelpContext == 1)
      MessageStr =  "The current call to TssRegList->ReadFromFile() "
                    "has been halted.  ";
    MessageStr +=   "The object information requested from the \"" + RegKey +
                    "\" Registry Key cannot be read.  The read operation was "
                    "halted with an error message of:  \"" + E.Message + "\"";
    Beep(1000, 1000);
    ShowMessage(MessageStr);
  }//End of catch(EInvalidOperation &E).
  catch(ERegistryException &E)//Traps all problems with pReg.
  {
    Beep(1000, 1000);
    ShowMessage("The call to TssRegList->ReadFromRegistry() cannot be "
                "completed at this time.  The requested Registry operation "
                "failed with a message of \"" + E.Message + "\"");
  }//End of catch(ERegistryException &E).
  if(pReg != NULL)
    delete pReg;
  return Result;
}//End of TssRegList::ReadFromRegistry().
//---------------------------------------------------------------------------
//This method writes a zero length binary entry to the registry for every
//NULL pointer that exists in the list.  It also tracks how many valid
//(non-NULL) records are written and how many records are written total.
//This information is used in ReadFromRegistry().
template <class T> bool TssRegList< T * >::WriteToRegistry()
{
  bool Result = false;
  TRegistry * pReg = NULL;
  if(FRegKey.IsEmpty())
  {
    ::Beep(1000, 1000);
    ShowMessage("TssRegList->WriteToRegistry() cannot be completed as the "
              "private FRegKey data member has not been initialized yet.  "
              "The current call to WriteToRegistry() will now abort.");
  }//End of if(FRegKey.IsEmpty()).
  else
  {
    T * pT = NULL;
    try
    {
      int ValidRecords = 0;//Initialize now for incrementing later.

      pReg = new TRegistry;
      if(FRootKey != NULL)
        pReg->RootKey = FRootKey;
      if(pReg->OpenKey(FRegKey, true))//Create if needed.
      {
        for(int i = 0; i < FList->Count; i++)
        {
          pT = Get(i);
          if(pT == NULL)//Write a zero length binary value.
            pReg->WriteBinaryData(OBJECT_VALUE_NAME + IntToStr(i), NULL, 0);
          else
            {//Write a record.  Increment valid record counter.
            pReg->WriteBinaryData
                    (OBJECT_VALUE_NAME + IntToStr(i), pT, sizeof(T));
            ++ValidRecords;//Track valid records.
            }//End of else.
        }//End of for(int i = 0; i < FList->Count; i++).
        //Write the record and total records counters.
        pReg->WriteString(TYPENAME_VALUE_NAME, ObjectType);
        pReg->WriteInteger(RECORDCOUNT_VALUE_NAME, ValidRecords);
        pReg->WriteInteger(TOTCOUNT_VALUE_NAME, FList->Count);
      }//End of if(pReg->OpenKey(FRegKey).
      Result = true;
    }//End of try.
    catch(ERegistryException &E)//Traps all problems with pReg.
    {
      Beep(1000, 1000);
      ShowMessage("The call to TssRegList->WriteToRegistry() cannot be "
                  "completed at this time.  The requested Registry operation "
                  "failed with a message of \"" + E.Message + "\"");
      if(pReg != NULL)
      {
        //Write known safe values to the Registry.  See ReadFromRegistry().
        pReg->WriteString(TYPENAME_VALUE_NAME, "");//
        pReg->WriteInteger(RECORDCOUNT_VALUE_NAME, 0);
        pReg->WriteInteger(TOTCOUNT_VALUE_NAME, 0);
      }//End of if(pReg != NULL).
    }//End of catch(ERegistryException &E).
  }//End of else.
  if(pReg != NULL)
    delete pReg;
  return Result;
}//End of TssRegList::WriteToRegistry().
//---------------------------------------------------------------------------
#endif
