/****************************************************************************
 * Copyright (c) 2000 - 2001
 * by Cristobal J. Gallegos
 *
 * The above copyright notice must appear in any and all copies of this code
 * and in all supporting documentation.  I make no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 ***************************************************************************/
#ifndef TempListH
#define TempListH
#include <new.h>
  //Needed to catch std::bad_alloc in second versions of Add() and Insert().
  //std::bad_alloc exception also possible in AllocateNewObjects().
//#include <exception>//Specified by online help to catch std::bad_alloc.
//---------------------------------------------------------------------------
//This first implementation provides the basis for declaring the second
//version.  Without it, assuming that the second is declared the way it is
//(TssTempList< T * >), the compiler will complain about "Undefined symbol
//'TssTempList'".  The first version will be called if one tries to
//instantiate a TssTempList using intrinsic data types (bool, byte, int,
//float, etc).  Due to the private constructor and copy constructor, the
//compiler will not allow any use of the first version.  This version is
//never created by the compiler even if someone does try to reference it.
template <class T> class TssTempList : System::TObject
{
private://Just for clarity.
  TssTempList();//Constructor not defined.
  TssTempList(const TssTempList&);//Copy Constructor not defined.
};//End of TssTempList (first version).
//---------------------------------------------------------------------------
//Second version.  Insures that TssTempList can only be instantiated by
//specifying pointers to "T" objects (TssTempList< T * >).  This is ensured
//by partial specialization.  Must have the first version declared before
//compiler will allow this version (see notes above).
template <class T> class TssTempList< T * > : System::TObject
{
protected://Allows use by descendants.
  const bool F_ALLOW_CREATE, F_ALLOW_DELETE, F_ALLOW_NULL;

  TList * FList;

  void AllocateNewObjects(int NewCount) const;
    //Used by SetListCount() and descendant classes.

  T * Get(int Index) const
    //Read method of Items[].  Used throughout TssTempList and descendants.
  {
    return reinterpret_cast< T * >(FList->Items[Index]);
  }

  String GetClassName() const;//Defined later.
    //Returns correct name for this class and descendants.

  int GetListCount() const//Read method of Count property.
  {
    return FList->Count;
  }

  String GetObjectType() const;//Defined later.
    //Returns type name for objects stored by TssTempList and descendants.

  inline void Put(int Index, T * Item) const;//Needed by Items[] property.
    //NOTE:  Items[i] = Item automatically passes both Index and Item.

  void ReportBad_Alloc(String FunctName, String Msg) const;
    //Reports failed resource allocation attempts to end user.

  void SetListCount(int NewCount) const;//Write method of Count property.

  static int __fastcall SortingFunct(void * Item1, void * Item2)
  {
    //This function is passed into the call to FList->Sort().  Must
    //dereference pointers so that user defined sorting operator is called.
    return *reinterpret_cast< T * >(Item1) ^ *reinterpret_cast< T * >(Item2);
    //      Code for using with user supplied Compare() function.
    //return reinterpret_cast< T * >(Item1)->
    //        QSortCompare(reinterpret_cast< T * >(Item2));
    //This assumes user's sorting function accepts pointers to objects.
    //If sorting fuction accepts only objects, pointers must be dereferenced.
  }//End of SortingFunct().

  public:
  TssTempList(bool AllowCreation, bool AllowDeletion, bool AllowNULL) :
    F_ALLOW_CREATE(AllowCreation), F_ALLOW_DELETE(AllowDeletion),
    F_ALLOW_NULL(AllowNULL)//Initialize all const data members.
  {
    FList = new TList;
  }//End of class constructor.

  virtual __fastcall ~TssTempList()
  {
    //Have to declare using __fastcall or compiler halts with the message
    //"Virtual function 'TssTempList< T > ::~TssTempList()' conflicts
    //with base class TObject."  NOTE:  TObject destructor is virtual.
    Clear();//Have to call this manually.  TObject's destructor doesn't
      //contain an explicit call to a virtual TObject->Clear() method.
      //Deletes objects from memory if F_ALLOW_DELETE is true.
    delete FList;//Redundantly calls TList::Clear().
  }//End of class destructor.

  inline int Add(T * Item) const;//First form.  Defined later.

  inline T * Add() const;//Second form.  Defined later.

  String __property ClassName = {read=GetClassName};
    //NOTE:  Excludes name mangling characters assigned by compiler.

  inline void Clear() const;
    //Deletes objects from memory if F_ALLOW_DELETE is true.

  int __property Count = {read=GetListCount, write=SetListCount};

  inline void Delete(int Index) const;
    //Deletes objects from memory if F_ALLOW_DELETE is true.

  void Exchange(int Index1, int Index2) const
  {
    FList->Exchange(Index1, Index2);
  }

  //In BCB 5.0 TList->Delete() and TList->Extract() only remove pointers.
  //Descendant classes re-define Delete() to free object memory.
  void Extract(int Index) const
  {
    FList->Delete(Index);
  }

  T * First(void) const
  {
    return reinterpret_cast< T * >(FList->First());
  }

  int IndexOf(T * Item) const
  {
    return FList->IndexOf(Item);
  }

  inline void Insert(int Index, T * Item) const;//First form.  Defined later.

  inline T * Insert(int Index) const;//Second form.  Defined later.

  __property T * Items[int Index] = {read=Get, write=Put};
    //NOTE: calling Put() automatically passes both Index and Item.

  T * Last(void) const
  {
    return reinterpret_cast< T * >(FList->Last());
  }

  void Move(int CurIndex, int NewIndex) const
  {
    FList->Move(int CurIndex, int NewIndex);
  }

  String __property ObjectType = {read=GetObjectType};
    //NOTE:  Excludes name mangling characters assigned by compiler.

  void Pack(void) const
  {
    FList->Pack(void);
  }

  inline int Remove(T * Item) const;//Defined later.

  void Sort()
  {
    FList->Sort(SortingFunct);//Passes private member function.  See above.
  }
};//End of TssTempList.
//---------------------------------------------------------------------------
//Used by SetListCount() and descendant classes.  This method assumes that
//NewCount is within reasonable limits.
template <class T> void TssTempList< T * >
        ::AllocateNewObjects(int NewCount) const
{
  try
  {
    int OldCount = FList->Count;
    T * ArrayOfT = new T[NewCount - OldCount];//Allocate objects.
      //The global operator new[]() throws a std::bad_alloc exception
      //if memory allocation attempt fails.  Not true in BCB 4.0??
    FList->Count = NewCount;//Allocate space in the list.
    //OldCount now indexes the first new pointer in FList.
    for(int i = OldCount; i < FList->Count; i++)//Add objects to list.
      FList->Items[i] = &ArrayOfT[i - OldCount];//Take address.
    //ArrayOfT pointer now falls out of scope.  Objects managed by FList.
  }//End of try.
  catch(std::bad_alloc &E)//Exception implicitly passed by reference.
  //The call to ::operator new [] failed.  Warn the end user.
  {
    //The E.what() below returns a char array that shows the failure
    //message reported by the memory management system.
    ReportBad_Alloc//First the function name, then the extra message.
            (ClassName + "->ReportBad_Alloc()", String(E.what()));
  }//End of catch(std::bad_alloc).
}//End of TssTempList< T >::AllocateNewObjects().
//---------------------------------------------------------------------------
//First form.  This method warns the programmer and aborts it's execution
//if Item == NULL and F_ALLOW_NULL == false.
template <class T> int TssTempList< T * >::Add(T * Item) const
{
  int Result = -1;
  if(Item == NULL && !F_ALLOW_NULL)
  {
    ::Beep(1000, 1000);
    ShowMessage("This instance of " + ClassName + " was not created with "
            "the ability to store NULL objects.  The current call to Add() "
            "(the first form) will now abort.");
  }//End of if(Item == NULL && !F_ALLOW_NULL).
  else
    Result = FList->Add(Item);
  return Result;
}//End of TssTempList< T >::Add() (the first form).
//---------------------------------------------------------------------------
//Second form.  Returns the pointer to an automatically created "T"
//instance.  This method warns the programmer and aborts it's execution
//if F_ALLOW_CREATE is false.
template <class T> T * TssTempList< T * >::Add() const
{
  T * pT = NULL;
  try
  {
    if(F_ALLOW_CREATE)
    {
      pT = new T;
      FList->Add(pT);
    }//End of if(F_ALLOW_CREATE).
    else
    {
      ::Beep(1000, 1000);
      ShowMessage("This instance of " + ClassName + " was not created with "
              "the ability to create objects for storage.  The current call "
              "to Add() (the second form) will now terminate.");
    }//End of else.
  }//End of try.
  catch(std::bad_alloc &e)//Exception implicitly passed by reference.
  //The call to ::operator new [] failed.  Warn the end user.
  {
    //The E.what() below returns a char array that shows the failure
    //message reported by the memory management system.
    ReportBad_Alloc(ClassName + "->Add() (the second form)",
            ::AnsiString(E.what()));
      //First the function name, then the extra message.
    pT = NULL;//Just in case.
  }//End of catch(std::bad_alloc).
  return pT;
}//End of TssTempList< T >::Add() (the second form).
//---------------------------------------------------------------------------
template <class T> void TssTempList< T * >::Clear() const
{
  if(F_ALLOW_DELETE)
  {
    for(int i = 0; i < FList->Count; i++)
      delete Get(i);//Delete the object.
        //Calling Delete() would add a redundant test of F_ALLOW_DELETE.
  }//End of if(F_ALLOW_DELETE).
  FList->Clear();//Delete pointers.
}//End of TssTempList< T >::Clear().
//---------------------------------------------------------------------------
template <class T> void TssTempList< T * >::Delete(int Index) const
{
  if(F_ALLOW_DELETE)
    delete Get(Index);//Delete the object.  OK to call delete on a NULL.
  FList->Delete(Index);//Delete pointer.
}//End of TssTempList< T >::Delete().
//---------------------------------------------------------------------------
//ClassName() returns the dynamic class instance name in the form of a Delphi
//ShortString: "@%TssTempList$p9TMeterCfg%" or "@%TssDiskList$p11TTimeRecord%"
//(quotation marks added by me).  The compiler appends the object type name
//with the number of characters that it contains, hence the "9TMeterCfg" 
//and "11TTimeRecord" parts of the strings.
//@%TssTempList$pg% for long double.  @%TssDiskList$pd% for double.
//@%TssDiskList$pi% for int.  @%TssTempList$pul% for DWORD or ULONG.
template <class T> String TssTempList< T * >::GetClassName() const
{
  String ClassName = TObject::ClassType()->ClassName();
    //Provides the correct class type information for this and descendant
    //classes.  Converts a Delphi ShortString to a BCB VCL AnsiString.
  return ClassName.SubString(3, ClassName.Pos("$p") - 3);
    //Removes leading "@%" and stops before the "$p" separator.  SubString()
    //apparently calls Delete(1, 3) (due to starting the SubString() process
    //from ClassName[3]) before calling SetLength() with the length value
    //returned from the call to Pos().  Need to Delete two characters from
    //the front of the string, hence the "- 3" instead of "- 1".
}//End of TssTempList< T >::GetClassName().
//---------------------------------------------------------------------------
//Removes the numbers appended to the actual object name part of the string
//due to compiler name mangling.  See notes in GetClassName().
template <class T> String TssTempList< T * >::GetObjectType() const
{
  String ClassName = TObject::ClassType()->ClassName();
  ClassName.Delete(1, ClassName.Pos("$p") + 1);
  //The compiler does not allow variable or class names to begin with numbers.
  //Therefore, the following shoule be fairly safe.
  while(::IsCharAlpha(ClassName[1]) == 0)//Thanks Win32 API.
    ClassName.Delete(1,1);//Get rid of numbers added due to name mangling.
  ClassName.SetLength(ClassName.Length() - 1);
  return ClassName;
}//End of TssTempList< T >::GetObjectType().
//---------------------------------------------------------------------------
//First form.  This method warns the programmer and aborts it's execution
//if Item == NULL and F_ALLOW_NULL == false.
template <class T> void TssTempList< T * >::Insert(int Index, T * Item) const
{
  if(Item == NULL && !F_ALLOW_NULL)
  {
    ::Beep(1000, 1000);
    ShowMessage("This instance of " + ClassName + " was not created with "
            "the ability to store NULL objects.  The current call to "
            "Insert() will now abort.");
  }//End of if(Item == NULL && !F_ALLOW_NULL).
  else
    FList->Insert(Index, Item);
}//End of TssTempList< T >::Insert() (the first form).
//---------------------------------------------------------------------------
//Second form.  Returns the pointer to an automatically created "T"
//instance.  This method warns the programmer and aborts it's execution
//if F_ALLOW_CREATE is false.
template <class T> T * TssTempList< T * >::Insert(int Index) const
{
  T * pT = NULL;
  try
  {
    if(F_ALLOW_CREATE)
    {
      pT = new T;
      FList->Insert(Index, pT);
    }//End of if(F_ALLOW_CREATE).
    else
    {
      ::Beep(1000, 1000);
      ShowMessage("This instance of " + ClassName + " was not created with "
              "the ability to create objects for storage.  The current call "
              "to Insert() (the second form) will now terminate.");
    }//End of else.
  }//End of try.
  catch(std::bad_alloc &e)//Exception implicitly passed by reference.
  //The call to ::operator new [] failed.  Warn the end user.
  {
    //The E.what() below returns a char array that shows the failure
    //message reported by the memory management system.
    ReportBad_Alloc(ClassName + "->Insert() (the second form)",
            ::AnsiString(E.what()));
      //First the function name, then the extra message.
    pT = NULL;//Just in case.
  }//End of catch(std::bad_alloc).
  return pT;
}//End of TssTempList< T >::Insert() (the second form).
//---------------------------------------------------------------------------
//Needed by Items[].  Can't call Delete().  That deletes the FList[Index]
//pointer.  Need to re-assign the pointer and possibly delete the object.
//Note:  Items[Index] = Item; automatically passes both Index and Item.
template <class T> void TssTempList< T * >::Put(int Index, T * Item) const
{
  if(Item == NULL && !F_ALLOW_NULL)
  {
    ::Beep(1000, 1000);
    ShowMessage("This instance of " + ClassName + " was not created with "
              "the ability to store NULL objects.  The current call to "
              "Put() (used by Items[]) will now abort.");
  }//End of if(Item == NULL && !F_ALLOW_NULL).
  else
  {
    if(F_ALLOW_DELETE)
      delete Get(Index);//Delete the object.  OK to call delete on a NULL.
    FList->Items[Index] = Item;
  }//End of else.
}//End of TssTempList< T >::Put().
//---------------------------------------------------------------------------
//TList::Remove() calls TList::IndexOf() internally.  Therefore,
//TList::Remove() returns -1 if the pointer didn't exist in the list.
template <class T> int TssTempList< T * >::Remove(T * Item) const
{
  int Result = FList->Remove(Item);
  if(F_ALLOW_DELETE && Result >= 0)
    delete Item;
  return Result;
}//End of TssTempList< T >::Remove().
//---------------------------------------------------------------------------
//Should be called after any failed call to new() or new[]. 
template <class T> void TssTempList< T * >::
        ReportBad_Alloc(String FunctName, String Msg) const
{
  ::Beep(1000, 1000);
  ShowMessage("The operating system could not allocate the memory needed "
          "for the objects requested in the current call to " + FunctName
          + ".  The attempted memory allocation failed with a message of "
          "\"" + Msg + "\".  The current call to " + FunctName +
          " will now terminate.");
}//End of TssTempList< T * >::ReportBad_Alloc().
//---------------------------------------------------------------------------
//This method tests both the F_ALLOW_CREATE and F_ALLOW_NULL members and
//acts accordingly.  This method does nothing if validity checks pass and
//NewCount is equal to FList->Count.
template <class T> void TssTempList< T * >::SetListCount(int NewCount) const
{
  if(!(F_ALLOW_CREATE || F_ALLOW_NULL))
  {
    ::Beep(1000, 1000);
    ShowMessage("This instance of " + ClassName + " does not allow the "
            "direct manipulation of it's Count property.  The current "
            "call to SetListCount() will now terminate.");
    return;
  }//End of if(!(F_ALLOW_CREATE || F_ALLOW_NULL)).
  if(NewCount < 0 || (NewCount > (MaxListSize - Count)))
    //Avoids EListError exception.  The second term prevents rollover
    //errors if Count + NewCount sums up greater than MaxInt.
    //MaxListSize is defined in Classes.pas as MaxInt / 16 (134217727).
  {
    ::Beep(1000, 1000);
    ShowMessage("The parameter \"NewCount\" being passed into the current "
            "call to " + ClassName + "->SetListCount() is out of bounds "
            "or may cause a List over-run.\n\nNewCount = " +
            IntToStr(NewCount) + "\nCount = " + IntToStr(Count) +
            "\nMaximum total allowed is:  " + IntToStr(MaxListSize) +
            "\n\nThe current call to SetListCount() will now terminate.");
    return;
  }//End of if(NewCount < 0 || (NewCount > (MaxListSize - Count))).
  if(NewCount > FList->Count)//Try to add something to the list.
  {
    if(F_ALLOW_CREATE)
      AllocateNewObjects(NewCount - Count);//Increase Count.
    else//If F_ALLOW_CREATE is false, F_ALLOW_NULL must be true.
      FList->Count = NewCount;
  }//End of if(NewCount > FList->Count).
  else if(NewCount < FList->Count)
  {
    while(NewCount < FList->Count)//If the list needs to be smaller.
      Delete(NewCount);//Delete decides whether to delete objects or not.
  }//End of else if(NewCount < FList->Count).
}//End of TssTempList< T >::SetListCount().
//---------------------------------------------------------------------------
#endif

