
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include "Unit1_ToolBar.h"
#include "Unit2.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TForm1 *Form1;
//---------------------------------------------------------------------------

//////////////////////////////////////////////////////////
//                                                      //
//  Streaming utility functions                         //
//  (See the article "Saving and loading components".)  //
//                                                      //
//////////////////////////////////////////////////////////

int __fastcall SaveComponent(
   AnsiString filename,
   TComponent* Component
  )
{
  assert(Component != NULL);
  assert(Component->Owner != NULL);

  std::auto_ptr<TFileStream> fs(
    new TFileStream(filename, fmCreate)
    );
  std::auto_ptr<TWriter> Writer(
    new TWriter(fs.get(), 4096)
    );

  // specify the Root component
  Writer->Root = Component->Owner;
  // write the component
  Writer->WriteComponent(Component);
  // return the number of bytes written
  return Writer->Position;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

int __fastcall LoadComponent(
   AnsiString filename,
   TComponent*& Component
  )
{
  assert(Component != NULL);
  assert(Component->Owner != NULL);

  std::auto_ptr<TFileStream> fs(
    new TFileStream(filename, fmOpenRead)
    );
  std::auto_ptr<TReader> Reader(
    new TReader(fs.get(), 4096)
    );

  // set Root, Owner, and Parent
  Reader->Root = Component->Owner;
  Reader->Owner = Component->Owner;
  TControl* Control =
    dynamic_cast<TControl*>(Component);
  if (Control) {
    Reader->Parent = Control->Parent;
  }

  // remove the existing component
  delete Component; Component = NULL;

  // load the stored component
  Reader->BeginReferences();
  try {
    Component =
      Reader->ReadComponent(NULL);
  }
  __finally {
    Reader->FixupReferences();
    Reader->EndReferences();
  }

  // return the number of bytes read
  return Reader->Position;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void RegisterOwnedComponents(
   TComponent& Component
  )
{
  const int count =
    Component.ComponentCount;
  for (int idx = 0; idx < count; ++idx)
  {
    TComponent& OwnedComponent =
      *Component.Components[idx];
    const TComponentClass classes[1] =
      {OwnedComponent.ClassType()};

    // register OwnedComponent
    RegisterClasses(classes, 0);

    // recursively register all of
    // OwnedComponent's components
    RegisterOwnedComponents(
      OwnedComponent);
  }
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



//////////////////////////////////////////////////////////
//                                                      //
//  Toolbar utility functions                           //
//                                                      //
//////////////////////////////////////////////////////////

void ShowInsertMark(
   TToolBar& ToolBar, TToolButton& Btn,
   bool after)
{
  TBINSERTMARK tbim;
  tbim.iButton = Btn.Index;
  tbim.dwFlags = after;

  SNDMSG(
    ToolBar.Handle, TB_SETINSERTMARK, 0,
    reinterpret_cast<LPARAM>(&tbim)
    );
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void HideInsertMark(
   TToolBar& ToolBar
  )
{
  const TBINSERTMARK tbim = {-1};
  SNDMSG(
    ToolBar.Handle, TB_SETINSERTMARK, 0,
    reinterpret_cast<LPARAM>(&tbim)
    );
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

void ResetCommandIDs(TToolBar& ToolBar)
{
  const HWND hToolBar = ToolBar.Handle;
  const int count = ToolBar.ButtonCount;
  for (int idx = 0; idx < count; ++idx)
  {
    SNDMSG(
      hToolBar, TB_SETCMDID, idx, idx);
  }
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

int PtOnSeparator(
   TToolBar& ToolBar,
   int X, int Y
  )
{
  RECT RBtn;
  const POINT PMouse = {X, Y};
  const HWND hToolBar = ToolBar.Handle;
  const int count = ToolBar.ButtonCount;

  for (int idx = 0; idx < count; ++idx)
  {
    TToolButton& Btn =
      *ToolBar.Buttons[idx];
    if (Btn.Style == tbsDivider ||
        Btn.Style == tbsSeparator)
    {
      SNDMSG(
        hToolBar, TB_GETITEMRECT, idx,
        reinterpret_cast<LPARAM>(&RBtn)
        );
      if (PtInRect(&RBtn, PMouse))
      {
        return idx;
      }
    }
  }
  return -1;
}
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//////////////////////////////////////////////////////////




__fastcall TForm1::TForm1(
  TComponent* Owner) : TForm(Owner),
  DragTBObject_(new TDragTBObject(NULL))
{
//
// TIP: Use this code to change the color
// of the toolbar's insertion mark:
//
//  SNDMSG(
//    SomeToolBar->Handle, TB_SETINSERTMARKCOLOR,
//    0, RGB(128, 128, 128) // new color
//    );
//

  // load the custom drag-and-drop cursors 
  Screen->Cursors[1] = LoadCursor(
    HInstance, MAKEINTRESOURCE(crTBDropAdd)
    );
  Screen->Cursors[2] = LoadCursor(
    HInstance, MAKEINTRESOURCE(crTBDropMove)
    );
  Screen->Cursors[3] = LoadCursor(
    HInstance, MAKEINTRESOURCE(crTBDropRemove)
    );

  // load the toolbars    
  LoadToolBars("complete_toolbars.tbi");
}
//---------------------------------------------------------------------------

void __fastcall TForm1::LoadToolBars(
   AnsiString filename
  )
{
  // register the TToolButton class
  const TComponentClass classes[1] =
    {__classid(TToolButton)};
  RegisterClasses(classes, 0);

  // load the entire rebar control
  // (i.e., the toolbars's parent)
  LoadComponent(filename, static_cast<TComponent*>(CoolBar1));
}
//---------------------------------------------------------------------------

void __fastcall TForm1::SaveToolBars(
   AnsiString filename
  )
{
  // save the entire rebar control
  // (i.e., the toolbars's parent)
  SaveComponent(filename, CoolBar1);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::CustomizeToolBar(
   TToolBar& ToolBar,
   bool customize
  )
{
  const int count = ToolBar.ButtonCount;
  for (int idx = 0; idx < count; ++idx)
  {
    TToolButton& Btn =
      *ToolBar.Buttons[idx];
    Btn.DragMode = customize ?
      dmAutomatic : dmManual;
    Btn.OnStartDrag = ToolBtnsStartDrag;
    Btn.OnEndDrag = ToolBtnsEndDrag;
    Btn.OnDragOver = ToolBtnsDragOver;
    Btn.OnDragDrop = ToolBtnsDragDrop;
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::
  CustomizeMode(bool customize)
{
  CustomizeToolBar(
    *FileToolBar, customize);
  CustomizeToolBar(
    *EditToolBar, customize);
  CustomizeToolBar(
    *FrmtToolBar, customize);
  CustomizeToolBar(
    *ViewToolBar, customize);

  CustomizeDlg->UpdateToolBarsCheckBox();  
  CustomizeMenuItem->Enabled = !customize;
  SaveMenuItem->Enabled = !customize;
  LoadMenuItem->Enabled = !customize;
}
//---------------------------------------------------------------------------

void __fastcall TForm1::
  CustomizeMenuItemClick(TObject *Sender)
{
  CustomizeDlg->Show();
  CustomizeMode(true);
}
//---------------------------------------------------------------------------

TToolButton* __fastcall TForm1::
  NewButton(
    TToolBar& ToolBar,
    AnsiString caption,
    int image_index
   )
{
  TToolButton* Btn =
    new TToolButton(this);
  Btn->Parent = &ToolBar;
  Btn->OnStartDrag = ToolBtnsStartDrag;
  Btn->OnEndDrag = ToolBtnsEndDrag;
  Btn->OnDragOver = ToolBtnsDragOver;
  Btn->OnDragDrop = ToolBtnsDragDrop;
  Btn->DragMode = dmAutomatic;

  // separator image is in last position
  if (image_index == TBImageList->Count - 1)
  {
    Btn->Style = tbsSeparator;
    Btn->Width = 8;
  }
  else
  {
    Btn->Caption = caption;
    Btn->ImageIndex = image_index;
  }
  return Btn;
}
//---------------------------------------------------------------------------

void __fastcall TForm1::DoDrop(
   TToolBar& ToolBar, TToolButton* Btn,
   TControl* SrcCtl, bool after
  )
{
  // if SrcCtl is a toolbutton
  if (dynamic_cast<TToolButton*>(SrcCtl))
  {
    SrcCtl->Parent = &ToolBar;
    if (Btn)
    {
      SrcCtl->Left = after ?
        Btn->Left + Btn->Width :
        Btn->Left;
    }
  }
  // otherwise, we're adding a new button
  else
  {
    // grab a pointer to
    // the dropped list item
    TListView& SrcListView = static_cast<TListView&>(*SrcCtl);
    TListItem* DragItem = SrcListView.Selected;

    // create a corresponding toolbutton
    TToolButton* NewBtn =
      new TToolButton(this);
    NewBtn->Parent = &ToolBar;
    NewBtn->DragMode = dmAutomatic;
    NewBtn->OnStartDrag = ToolBtnsStartDrag;
    NewBtn->OnEndDrag = ToolBtnsEndDrag;
    NewBtn->OnDragOver = ToolBtnsDragOver;
    NewBtn->OnDragDrop = ToolBtnsDragDrop;
    
    // (separator is the last image)
    if (DragItem->ImageIndex == TBImageList->Count - 1)
    {
      NewBtn->Style = tbsSeparator;
      NewBtn->Width = 8;
    }
    else
    {
      NewBtn->Caption = DragItem->Caption;
      NewBtn->ImageIndex = DragItem->ImageIndex;
      NewBtn->OnClick = ToolBtnsClick;
    }
    // position the new button
    if (Btn)
    {
      NewBtn->Left = after ?
        Btn->Left + Btn->Width : Btn->Left;
    }
  }
  // fix the command IDs
  ResetCommandIDs(ToolBar);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::ToolBtnsDragOver(
   TObject *Sender, TObject *Source,
   int X, int Y, TDragState State,
   bool &Accept
  )
{
  // determine whether to accept the drop
  Accept = dynamic_cast<TDragTBObject*>(Source);
  if (!Accept) return;

  // grab a reference to the
  // button that's being dragged over
  TToolButton& Btn = static_cast<TToolButton&>(*Sender);

  // grab a reference to the
  // button's toolbar
  TToolBar& ToolBar = static_cast<TToolBar&>(*Btn.Parent);

  // if the cursor is leaving the button
  if (State == dsDragLeave)
  {
    // hide the insertion mark
    HideInsertMark(ToolBar);
  }
  // otherwise
  else
  {
    // display an insertion mark
    // before or after the button
    const bool after = X > Btn.Width / 2;
    ShowInsertMark(ToolBar, Btn, after);
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::ToolBarsDragOver(
   TObject *Sender, TObject *Source,
   int X, int Y, TDragState State,
   bool &Accept
  )
{
  // determine whether to accept the drop
  Accept = dynamic_cast<TDragTBObject*>(Source);
  if (!Accept) return;

  // grab a reference to the
  // toolbar that's being dragged over
  TToolBar& ToolBar =
    static_cast<TToolBar&>(*Sender);

  // if the cursor is leaving the toolbar
  if (State == dsDragLeave)
  {
    // hide the insertion mark
    HideInsertMark(ToolBar);
  }
  // otherwise, if the
  // toolbar contains buttons
  else if (ToolBar.ButtonCount > 0)
  {
    const int index =
      PtOnSeparator(ToolBar, X, Y);
    if (index >= 0)
    {
      // show an insertion mark before
      // or after the separator
      TToolButton& SepBtn = *ToolBar.Buttons[index];
      const bool after = X - SepBtn.Left > SepBtn.Width / 2;
      ShowInsertMark(ToolBar, SepBtn, after);
    }
    else
    {
      // show an insertion mark
      // after the last button
      TToolButton& LastBtn = *ToolBar.
        Buttons[ToolBar.ButtonCount - 1];
      ShowInsertMark(
        ToolBar, LastBtn, true);
    }
  }
  // otherwise
  else
  {
    // ToolBar contains no buttons so
    // accept drops anywhere on it
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::ToolBtnsDragDrop(
   TObject *Sender, TObject *Source,
   int X, int Y
  )
{
  // grab a reference to the
  // control that's being dragged
  TControl& SrcCtl =
    *static_cast<TDragTBObject*>(Source)->Control;

  // grab a reference to the button
  // that's being dragged over
  TToolButton& Btn = static_cast<TToolButton&>(*Sender);

  // grab a reference to the
  // button's parent (toolbar)
  TToolBar& ToolBar = static_cast<TToolBar&>(*Btn.Parent);

  // move/insert the button
  const bool after = X > 0.5 * Btn.Width;
  DoDrop(ToolBar, &Btn, &SrcCtl, after);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::ToolBarsDragDrop(
   TObject *Sender, TObject *Source,
   int X, int Y
  )
{
  // grab a reference to the
  // control that's being dragged
  TControl& SrcCtl =
    *static_cast<TDragTBObject*>(Source)->Control;

  // grab a reference to the toolbar
  // that's being dragged over
  TToolBar& ToolBar = static_cast<TToolBar&>(*Sender);

  // if the toolbar contains buttons
  if (ToolBar.ButtonCount > 0)
  {
    // find the index of the separator
    // that's being dragged over
    const int index = PtOnSeparator(ToolBar, X, Y);
    if (index >= 0)
    {
      // grab a reference to the button
      // that's being dragged over
      TToolButton& SepBtn = *ToolBar.Buttons[index];

      // move/insert the source button
      const bool after = X - SepBtn.Left > 0.5 * SepBtn.Width;
      DoDrop(ToolBar, &SepBtn, &SrcCtl, after);
    }
    else
    {
      // grab a reference to the
      // last button on the toolbar
      TToolButton& LastBtn =
        *ToolBar.Buttons[ToolBar.ButtonCount - 1];

      // move/insert the source button
      DoDrop(ToolBar, &LastBtn, &SrcCtl, true);
    }
  }
  // otherwise
  else
  {
    // accept drops anywhere
    DoDrop(ToolBar, NULL, &SrcCtl, false);
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::
  ToolBarsMouseDown(
    TObject *Sender, TMouseButton Button,
    TShiftState Shift, int X, int Y
  )
{
  if (Button == mbLeft)
  {
    // grab a reference to the toolbar on
    // which the mouse button was pressed
    TToolBar& ToolBar = static_cast<TToolBar&>(*Sender);

    // if the cursor is over a separator
    const int sep_index = PtOnSeparator(ToolBar, X, Y);
    if (sep_index >= 0)
    {
      ToolBar.Buttons[sep_index]->
        Perform(WM_LBUTTONDOWN, 0, 0);
    }
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::
  ToolBtnsStartDrag(TObject *Sender,
    TDragObject *&DragObject
  )
{
  TToolButton& Btn = static_cast<TToolButton&>(*Sender);
  Btn.Marked = true;
  DragTBObject_->Control = &Btn;
  DragObject = DragTBObject_.get();
}
//---------------------------------------------------------------------------

void __fastcall TForm1::ToolBtnsEndDrag(
   TObject *Sender, TObject *Target, int X, int Y
  )
{
  TToolButton& Button = static_cast<TToolButton&>(*Sender);
  Button.Marked = false;
  if (Target == NULL)
  {
    delete &Button;
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::TBPopupMenuPopup(TObject *Sender)
{
  FileMenuItem->Checked = FileToolBar->Visible;
  EditMenuItem->Checked = EditToolBar->Visible;
  FrmtMenuItem->Checked = FrmtToolBar->Visible;
  ViewMenuItem->Checked = ViewToolBar->Visible;
}
//---------------------------------------------------------------------------

void __fastcall TForm1::TBMenuItemsClick(TObject *Sender)
{
  TMenuItem& MenuItem = static_cast<TMenuItem&>(*Sender);
  switch (MenuItem.MenuIndex)
  {
    case 0:
      FileToolBar->Visible =
        !FileToolBar->Visible;
      break;
    case 1:
      EditToolBar->Visible =
        !EditToolBar->Visible;
      break;
    case 2:
      FrmtToolBar->Visible =
        !FrmtToolBar->Visible;
      break;
    case 3:
      ViewToolBar->Visible =
        !ViewToolBar->Visible;
      break;
  }
  CustomizeDlg->UpdateToolBarsCheckBox();
}
//---------------------------------------------------------------------------

void __fastcall TForm1::
  LoadMenuItemClick(TObject *Sender)
{
  if (TBOpenDialog->Execute())
  {
    LoadToolBars(TBOpenDialog->FileName);
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::
  SaveMenuItemClick(TObject *Sender)
{
  if (TBSaveDialog->Execute())
  {
    SaveToolBars(TBSaveDialog->FileName);
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::
  ToolBtnsClick(TObject *Sender)
{
  TToolButton& Btn = static_cast<TToolButton&>(*Sender);
  ShowMessage(
    "Caption = " +  Btn.Caption +
    "\nImageIndex = " +
    IntToStr(Btn.ImageIndex)
    );
}
//---------------------------------------------------------------------------

void __fastcall TForm1::Button1Click(TObject *Sender)
{
  for (int index = 0; index < FileToolBar->ButtonCount; ++index)
  {
    FileToolBar->Buttons[index]->OnClick = ToolBtnsClick;
  }
  for (int index = 0; index < EditToolBar->ButtonCount; ++index)
  {
    EditToolBar->Buttons[index]->OnClick = ToolBtnsClick;
  }
  for (int index = 0; index < FrmtToolBar->ButtonCount; ++index)
  {
    FrmtToolBar->Buttons[index]->OnClick = ToolBtnsClick;
  }
  for (int index = 0; index < ViewToolBar->ButtonCount; ++index)
  {
    ViewToolBar->Buttons[index]->OnClick = ToolBtnsClick;
  }
}
//---------------------------------------------------------------------------

