/****************************************************************************
 * Copyright (c) 2000 - 2001
 * by Cristobal J. Gallegos
 *
 * The above copyright notice must appear in any and all copies of this code
 * and in all supporting documentation.  I make no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 ***************************************************************************/
#ifndef DiskListH
#define DiskListH
//---------------------------------------------------------------------------
#include "TempList.h"//Contains TssTempList.
//---------------------------------------------------------------------------
//First version.  See notes in TempList.h for explanation.
template <class T> class TssDiskList : System::TObject
{
private://Not needed, just here for clarity.
  TssDiskList();//Constructor not defined.
  TssDiskList(const TssDiskList&);//Copy Constructor not defined.
};//End of TssDiskList (first version).
//---------------------------------------------------------------------------
//Second version.  See notes in TempList.h for explanation.
template <class T> class TssDiskList< T * > : public TssTempList< T * >
{
  private:
    const byte F_NULL_RECORD, F_VALID_RECORD;

    String FFileName;

    void SetFileName(const String NewValue);

  public:
    TssDiskList(bool AllowCreation, bool AllowDeletion, bool AllowNULL) :
      F_NULL_RECORD(0x0F), F_VALID_RECORD(0xF0),
      //Initialize TssDiskList const data members.
      TssTempList< T * >(AllowCreation, AllowDeletion, AllowNULL)
      //The initializer list is the only way to pass this constructor's
      //parameters on to the base class' constructor.
    {
    }

  /*
    No need for destructor.  TObject's destructor is virtual.  This means
    that the TssTempList (which is derived from TObject) destructor will
    be called whenever a TssDiskList instance is destroyed.
  */

  __property String FileName = {read=FFileName, write=SetFileName};

  int GetRecordCountFromFile();

  bool ReadFromFile();

  bool WriteToFile();
};//End of TssDiskList.
//---------------------------------------------------------------------------
//This method definition returns a signed integer for two reasons.  The
//first is that I have been told that TList is only efficient for up to
//5000 pointers.  Therefore, and unsigned int return value is less than
//necessary.  Returning a signed int also allows for returning an error
//value (as a negative number).
template <class T> int TssDiskList< T * >::GetRecordCountFromFile()
{
  int Result = -1;//A know error value.  Could be modified further.
  TFileStream * pFS = NULL;
  try
  {
    if(FFileName.IsEmpty())
      throw EInvalidOperation("TssDiskList->GetRecordCountInFile\(\) "
              "cannot complete as the private FFileName data member has not "
              "been initialized.");
    pFS = new TFileStream(FFileName, fmOpenRead | fmShareExclusive);
    pFS->Position = pFS->Size - sizeof(int);//Position properly.
    pFS->Read(&Result, sizeof(int));//Just read the last 32 bits.
  }//End of try.
  catch(EInvalidOperation &E)
  {
    Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch(EInvalidOperationError &E).
  catch(...){}//Here in case there are problems with the file stream.
    //Simply let the return value do the talking.
  if(pFS != NULL)
    delete pFS;
  return Result;
}//End of TssDiskList::GetRecordCountInFile().
//---------------------------------------------------------------------------
//NOTE:  The testing of drive name validity below will throw an exception
//if the string passed in contains a network drive name and the computer
//is not currently connected to the network.  This function cannot trap
//single backslashes contained in file names without drive names.  In this
//situation, the creation of NewNameStr simply concatenates what comes
//before and after the single backslash.
template <class T>
        void TssDiskList< T * >::SetFileName(const String NewNameStr)
{
  try
  {
    int TempInt;//Used for various things.
    String TempStr;//Used at various times.
    if(NewNameStr.IsEmpty())
      throw EInvalidOperation("The string passed into TssDiskList->"
              "SetFileName\(\) cannot be empty.  ");

    //Test the validity of any drive names that may have been passed in.
    TempStr = ExtractFileDrive(NewNameStr);//Need memorized for GetDriveType().
    if(TempStr.IsEmpty())
      TempStr = NewNameStr;
    else//There is a drive name to test.
    {
      switch(::GetDriveType(TempStr.c_str()))
      {
        //Can't use break statements.  Compiler flags them as unreachable code
        //due to the exception generation and produces warnings.  Don't worry
        //about DRIVE_REMOVABLE, DRIVE_FIXED, DRIVE_REMOTE, or DRIVE_RAMDISK.
        case 0://The drive type cannot be determined.
        case 1://The root directory does not exist.
          throw EInvalidOperation("The string passed into TssDiskList->"
                  "SetFileName\(\) contains the drive name of a drive whose "
                  "type either cannot be determined, or whose root directory "
                  "does not exist.  ");
        case DRIVE_CDROM://The drive is a CD-ROM drive.
          throw EInvalidOperation("The string passed into TssDiskList->"
                  "SetFileName\(\) contains the drive name of a CDROM "
                  "drive.  This template connot write information to a "
                  "CDROM drive.  ");
      }//End of switch(::GetDriveType(TempStr.c_str())).
      if(NewNameStr[3] != '\\')
        //Have to specifically test for this.  ExtractFileDrive() doesn't
        //care if there is no double backslash after the drive name.  This
        //also throws the exception if NewNameStr is derived programatically
        //without double backslashes (assuming a drive name is supplied).
        throw EInvalidOperation("The string passed into TssDiskList->"
                "SetFileName\(\) contains a valid drive name but the rest "
                "of the path information is invalid.  ");
      //Eliminate the drive name to help improve the accuracy of what follows.
      //Eliminating the first backslash just for efficiency.
      TempStr = NewNameStr.SubString(4, NewNameStr.Length());
        //Excludes previously tested characters.  NewNameStr.Length() is
        //too much but insures complete copy.
    }//End of else.

    //Test the new filename for invalid file name characters.
    TempInt = TempStr.Length();//Avoids continually re-evaluating Length().
    for(int i = 1; i <= TempInt; i++)
    {
      if(TempStr.IsDelimiter("/:*?\"<>|", i))
        //Delimit list does check for a possible second colon.
        throw EInvalidOperation("The string passed into TssDiskList->"
                "SetFileName\(\) contains characters that are not legal "
                "for a DOS/Windows filename.  ");
    }//End of for(int i = 1; i <= StrLength; i++).

    //Test for a valid file name and extension.
    if(ExtractFileName(NewNameStr).IsEmpty())
      throw EInvalidOperation("The string passed into TssDiskList->"
              "SetFileName\(\) does not contain a valid file name.  ");
    if(ExtractFileExt(NewNameStr).IsEmpty())
      throw EInvalidOperation("The string passed into TssDiskList->"
              "SetFileName\(\) does not contain a valid file extension.  ");

    //Assuming nothing has thrown an exception so far... (whew!)
    FFileName = NewNameStr;
  }//End of try.
  catch(EInvalidOperation &E)
  {
    Beep(1000, 1000);
    ShowMessage(E.Message
                + "The private FFileName data member will not be changed.");
  }//End of catch.
}//End of TssDiskList::SetFileName().
//---------------------------------------------------------------------------
//If the current TssDiskList instance is created such that F_ALLOW_CREATE
//is true, this function takes responsibility for creating/deleting objects
//as needed.  If F_ALLOW_CREATE is false, the user is forced to call
//GetRecordCountInFile() and allocate/delete objects as needed before calling
//this function.  This methodology allows the objects to have more than a
//default (no argument) constructor.
//NOTE:  This method reads the tags placed in the file by WriteToFile() to
//determine if there is a valid record to be read next.  If the tag was set
//to F_NULL_RECORD and F_ALLOW_NULL is true, a NULL pointer is added to
//FList.  If F_ALLOW_NULL is false, an exception is thrown.  If the record
//tag was set to F_VALID_RECORD, the record information is read into the 
//appropriate object as referenced in FList.
template <class T> bool TssDiskList< T * >::ReadFromFile()
{
  bool Result = false;
  byte RecordFlag;//Initialized before use later.
  int RecordCount, TotalCount;//Initialized before use later.
  TFileStream * pFS = NULL;
  try
  {
    if(FFileName.IsEmpty())
      throw EInvalidOperation("TssDiskList->ReadFromFile\(\) cannot complete "
              "as the private FFileName data member has not been "
              "initialized.");
    pFS = new TFileStream(FileName, fmOpenRead | fmShareExclusive);
    pFS->Position = pFS->Size - (2 * sizeof(int));
    pFS->Read(&TotalCount, sizeof(int));
    pFS->Read(&RecordCount, sizeof(int));
    if(!F_ALLOW_NULL && (TotalCount > RecordCount))
      throw EInvalidOperation("The disk file specified contains NULL "
              "records.  This instance of TssDiskList was not created with "
              "the ability to store NULL records.  ");
    if(F_ALLOW_CREATE)
      SetListCount(RecordCount);//Adds or removes objects as needed.
    if(FList->Count != RecordCount)//Also tests success of SetListCount().
      throw EInvalidOperation("The current TssDiskList class instance "
              "currently has enough room allocated for " + IntToStr(FList->
              Count) + " records.  Room for " + IntToStr(RecordCount) +
              " is required if this function call is to execute "
              "properly.  ");
    pFS->Position = 0;//Must move back to the start of the file stream. 
    for(int i = 0; i < TotalCount; i++)
    {
      pFS->Read(&RecordFlag, sizeof(byte));
      if(RecordFlag == F_NULL_RECORD)
        FList->Insert(i, NULL);//Add a NULL pointer.
      else
        pFS->Read(FList->Items[i], sizeof(T));//Must read to FList pointers.
    }//End of for(int i = 0; i < RecordCount; i++).
  Result = true;
  }//End of try.
  catch(EInvalidOperation &E)
  {
    //Catches problems that needs to be reported to the programmer.
    Beep(1000, 1000);
    ShowMessage(E.Message
            + "The current call to ReadFromFile\(\) will now abort.");
  }//End of catch.
  catch(...){}//Here in case there are problems with the file stream.
    //Simply let the return value do the talking.
  if(pFS != NULL)
    delete pFS;
  return Result;
}//End of TssDiskList::ReadFromFile().
//---------------------------------------------------------------------------
//This method prefixes every record written to the file with a tag that
//describes whether the record that follows it represents a NULL pointer
//that was in the list that wrote the file, or whether the record that
//follows is a valid (non-NULL) record.  It also tracks how many valid
//(non-NULL) records are written and how many records are written total.
//It appends this information to the end of the file produced.  This
//information is then used in GetRecordCountFromFile() and ReadFromFile().
template <class T> bool TssDiskList< T * >::WriteToFile()
{
  bool Result = false;
  byte RecordFlag;//Initialized before use later.
  int ValidRecords = 0;//Initialize now for use later.
  const int TotalRecords = FList->Count;//Must initialize now.
    //Provides a true buffer for pFS->Write to read from.
  TFileStream * pFS = NULL;
  T * pT = NULL;
  try
  {
    if(FFileName.IsEmpty())
      throw EInvalidOperation("TssDiskList->WriteToFile\(\) cannot "
      "complete as the private FFileName data member has not been "
      "initialized.  The current call to WriteToFile\(\) will now "
      "abort.");
    pFS = new TFileStream(FFileName, fmCreate | fmShareExclusive);
    for(int i = 0; i < FList->Count; i++)
    {
      pT = reinterpret_cast< T * >(FList->Items[i]);
      if(pT == NULL)
      {
        RecordFlag = F_NULL_RECORD;//Marks a NULL pointer.
        pFS->Write(&RecordFlag, sizeof(byte));//Write the tag to the file.
      }
      else
      {
        RecordFlag = F_VALID_RECORD;//Marks a valid record.
        pFS->Write(&RecordFlag, sizeof(byte));//Write the tag to the file.
        pFS->Write(pT, sizeof(T));//Write the record.
        ++ValidRecords;//Track valid record count.
      }
    }//End of for(int i = 0; i < FList->Count; i++).
    pFS->Write(&TotalRecords, sizeof(int));//Write total number of records.
    pFS->Write(&ValidRecords, sizeof(int));//Write number of valid records.
    Result = true;
  }//End of try.
  catch(EInvalidOperation &E)
  {//Catches problems that needs to be reported to the programmer.
    Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch.
  catch(...){}//Here in case there are problems with the file stream.
    //Simply let the return value do the talking.
  if(pFS != NULL)
    delete pFS;
  return Result;
}//End of TssDiskList::WriteToFile().
//---------------------------------------------------------------------------
#endif
