/****************************************************************************
 * Copyright (c) 2000 - 2001
 * by Cristobal J. Gallegos
 *
 * The above copyright notice must appear in any and all copies of this code
 * and in all supporting documentation.  I make no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 ***************************************************************************/
#ifndef RegListH
#define RegListH
//---------------------------------------------------------------------------
#include <registry.hpp>//Needed for TRegistry.
#include "TempList.h"//Contains TssTempList.
//---------------------------------------------------------------------------
//First version.  See notes in TempList.h for explanation.
template <class T> class TssRegList : System::TObject
{
private://Not needed, just here for clarity.
  TssRegList();//Constructor not defined.
  TssRegList(const TssRegList&);//Copy Constructor not defined.
};//End of TssRegList (first version).
//---------------------------------------------------------------------------
//Second version.  See notes in TempList.h for explanation.
//NOTE:
//TRegistry->ReadBinaryData() and TRegistry->ReadInteger() are case
//insensitive.
template <class T> class TssRegList< T * > : public TssTempList< T * >
{
private:

  const String VALUE_NAME, RECORDCOUNT_VALUE_NAME, TOTCOUNT_VALUE_NAME;

  String FRegKey;

  HKEY FRootKey;

public:
  TssRegList(bool AllowCreation, bool AllowDeletion, bool AllowNULL) :
    VALUE_NAME("Record #"), RECORDCOUNT_VALUE_NAME("Record Count"),
    TOTCOUNT_VALUE_NAME("Total Count"),
    //Initialize TssRegList const data members.
    TssTempList< T * >(AllowCreation, AllowDeletion, AllowNULL)
      //The initializer list is the only way to pass this constructor's
      //parameters on to the base class' constructor.
  {
    FRootKey = NULL;
  }//End of TssRegList constructor.

/*
  No need for destructor.  TObject's destructor is virtual.  This means
  that the TssTempList (which is derived from TObject) destructor will
  be called whenever a TssRegList instance is destroyed.
*/

__property String RegKey = {read=FRegKey, write=FRegKey};

__property HKEY RootKey = {read=FRootKey, write=FRootKey};

int GetRecordCountFromRegistry();

bool ReadFromRegistry();

bool WriteToRegistry();
};//End of TssRegList.
//---------------------------------------------------------------------------
template <class T> int TssRegList< T * >::GetRecordCountFromRegistry()
{
  int Result = -1;//A know error value.  Could be modified further.
  TRegistry * pReg = NULL;
  try
  {
    if(FRegKey.IsEmpty())
      throw EInvalidOperation("TssRegList->GetRecordCountInRegistry\(\) "
              "cannot complete as the RegKey property has not been "
              "initialized.");
    pReg = new TRegistry;
    if(FRootKey != NULL)
      pReg->RootKey = FRootKey;
    if(pReg->OpenKeyReadOnly(FRegKey))
      Result = pReg->ReadInteger(RECORDCOUNT_VALUE_NAME);
    else
      throw EInvalidOperation("TssRegList->GetRecordCountInRegistry\(\) "
              "cannot complete as the Windows Registry key specified by the  "
              "RegKey property does not exist.");
  }//End of try.
  catch(EInvalidOperation &E)
  {
    Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch(EInvalidOperationError &E).
  catch(...){}//Here in case there are problems with TRegistry allocation.
    //Simply let the return value do the talking.
  if(pReg != NULL)
    delete pReg;
  return Result;
}//End of TssRegList::GetRecordCountInRegistry().
//---------------------------------------------------------------------------
//If the current TssRegList instance is created such that F_ALLOW_CREATE
//is true, this function takes responsibility for creating/deleting objects
//as needed.  If F_ALLOW_CREATE is false, the user is forced to call
//GetRecordCountInRegistry() and allocate/delete objects as needed before
//calling this function.  This methodology allows the objects to be created
//by more than a default (no argument) constructor.
//NOTE:  This method will read a zero length binary entry from the registry
//for every NULL pointer that was written to it by WriteToRegistry().  If
//F_ALLOW_NULL is true, a NULL pointer will be added to the list that called
//this method.  If it is false, an exception will be thrown.
template <class T> bool TssRegList< T * >::ReadFromRegistry()
{
  bool Result = false;
  int RecordCount, TotalCount;//Initialized before use later.
  TRegistry * pReg = NULL;
  try
  {
    if(FRegKey.IsEmpty())
      throw EInvalidOperation("TssRegList->ReadFromRegistry\(\) cannot "
              "complete as the private FRegKey data member has not been "
              "initialized.");
    pReg = new TRegistry;
    if(FRootKey != NULL)
      pReg->RootKey = FRootKey;
    if(pReg->OpenKeyReadOnly(FRegKey))
    {
      RecordCount = pReg->ReadInteger(RECORDCOUNT_VALUE_NAME);
      TotalCount = pReg->ReadInteger(TOTCOUNT_VALUE_NAME);
      if(!F_ALLOW_NULL && (TotalCount > RecordCount))
        throw EInvalidOperation("The specified Registry information "
              "contains NULL records.  This instance of TssRegList was not "
              "created with the ability to store NULL records.  ");
      if(F_ALLOW_CREATE)
        SetListCount(RecordCount);//Adds or removes objects as needed.
      if(FList->Count != RecordCount)//Also tests success of SetListCount().
        throw EInvalidOperation("The current TssRegList class instance "
                "currently has enough room allocated for " + IntToStr(FList->
                Count) + " records.  Room for " + IntToStr(RecordCount) +
                " records is required if this function call is to execute "
                "properly.  ");
      for(int i = 0; i < TotalCount; i++){
        if(pReg->ReadBinaryData
                (VALUE_NAME + IntToStr(i), NULL, sizeof(T)) == 0)
          //This call simply returns the size of the requested data.
          FList->Insert(i, NULL);//Add a NULL pointer.
        else
          pReg->ReadBinaryData
                  (VALUE_NAME + IntToStr(i), FList->Items[i], sizeof(T));
            //Must read to FList pointers.
        }//End of for(int i = 0; i < RecordCount; i++).
    Result = true;
    }//End of if(pReg->OpenKeyReadOnly(FRegKey)).
  }//End of try.
  catch(EInvalidOperation &E)
  {//Catches problems that needs to be reported to the programmer.
    Beep(1000, 1000);
    ShowMessage(E.Message
            + "The current call to ReadFromRegistry\(\) will now abort.");
  }//End of catch(EInvalidOperation &E).
  catch(...){}//Here in case there are problems with the file stream.
    //Simply let the return value do the talking.
  if(pReg != NULL)
    delete pReg;
  return Result;
}//End of TssRegList::ReadFromRegistry().
//---------------------------------------------------------------------------
//This method writes a zero length binary entry to the registry for every
//NULL pointer that exists in the list.  It also tracks how many valid
//(non-NULL) records are written and how many records are written total.
//This information is used in GetRecordCountFromRegistry() and
//ReadFromRegistry(). 
template <class T> bool TssRegList< T * >::WriteToRegistry()
{
  bool Result = false;
  int ValidRecords = 0;//Initialize now for use later.
  TRegistry * pReg = NULL;
  T * pT = NULL;
  try
  {
    if(FRegKey.IsEmpty())
      throw EInvalidOperation("TssRegList->WriteToRegistry\(\) cannot "
      "complete as the private FRegKey data member has not been initialized."
      "  The current call to WriteToRegistry\(\) will now abort.");
    pReg = new TRegistry;
    if(FRootKey != NULL)
      pReg->RootKey = FRootKey;
    if(pReg->OpenKey(FRegKey, true))
    {
      for(int i = 0; i < FList->Count; i++)
      {
        pT = reinterpret_cast< T * >(FList->Items[i]);
        if(pT == NULL)//Write a zero length binary value.
          pReg->WriteBinaryData(VALUE_NAME + IntToStr(i), NULL, 0);
        else
          {//Write a record.  Increment valid record counter.
          pReg->WriteBinaryData(VALUE_NAME + IntToStr(i), pT, sizeof(T));
          ++ValidRecords;//Track valid records.
          }
      }//End of for(int i = 0; i < FList->Count; i++).
    //Write the record and total records counters.
    pReg->WriteInteger(RECORDCOUNT_VALUE_NAME, ValidRecords);
    pReg->WriteInteger(TOTCOUNT_VALUE_NAME, FList->Count);
    }//End of if(pReg->OpenKey(FRegKey).
    Result = true;
  }//End of try.
  catch(EInvalidOperation &E)
  {
    //Catches problems that needs to be reported to the programmer.
    Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch.
  catch(...){}//Here in case there are problems with the file stream.
    //Simply let the return value do the talking.
  if(pReg != NULL)
    delete pReg;
  return Result;
}//End of TssRegList::WriteToRegistry().
//---------------------------------------------------------------------------
#endif
