/****************************************************************************
 * Copyright (c) 2000 - 2001
 * by Cristobal J. Gallegos
 *
 * The above copyright notice must appear in any and all copies of this code
 * and in all supporting documentation.  I make no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 ***************************************************************************/
#ifndef TempListH
#define TempListH
#include <new.h>
  //Needed to catch std::bad_alloc in Add(), Insert(), and SetListCount().
//#include <exception>//Specified by online help to catch std::bad_alloc.
//---------------------------------------------------------------------------
//The first implementation provides the basis for declaring the second
//version.  Without it, assuming that the second is declared the way it is
//(TssTempList< T * >), the compiler will complain about "Undefined symbol
//'TssTempList'".  The first version will be called if one tries to
//instantiate a TssTempList using intrinsic data types (bool, byte, int,
//float, etc).  Due to the private constructor and copy constructor, the
//compiler will not allow any use of the first version.  This version is
//never created by the compiler even if someone does try to reference it.
template <class T> class TssTempList : System::TObject
{
private://Not needed, just here for clarity.
  TssTempList();//Constructor not defined.
  TssTempList(const TssTempList&);//Copy Constructor not defined.
};//End of TssTempList (first version).
//---------------------------------------------------------------------------
//Second version.  Insures that TssTempList can only be instantiated by
//specifying pointers to "T" objects (TssTempList< T * >).  This is ensured
//by partial specialization.  Must have the first version declared before
//compiler will allow the second (see notes above).
template <class T> class TssTempList< T * > : TObject
{
protected://Allows use by descendants.
  const bool F_ALLOW_CREATE, F_ALLOW_DELETE, F_ALLOW_NULL;

  TList * FList;

  T * Get(int Index)//Needed by Items[].  Used throughout TssTempList.
  {
    return reinterpret_cast< T * >(FList->Items[Index]);
  }

  int GetListCount()
  {
    return FList->Count;
  }

  inline void Put(int Index, T * Item);//Needed by Items[] property.
    //Items[i] = Item;//Automatically passes both Index and Item.

  void SetListCount(int NewCount);

  static int __fastcall SortingFunc(void * Item1, void * Item2)
  {
    //This function is passed into the call to FList->Sort().  Must 
    //dereference pointers so that user defined sorting operator is called.
    return *reinterpret_cast< T * >(Item1) ^ *reinterpret_cast< T * >(Item2);
    //      Code for using with user supplied Compare() function.
    //T * NewItem1 = reinterpret_cast< T * >(Item1);//Need object to work with.
    //return NewItem1->Compare(NewItem1, reinterpret_cast< T * >(Item2));
    //This assumes user's sorting function accepts pointers to objects.
    //If sorting fuction accepts only objects, pointers must be dereferenced.
  }//End of SortingFunc().

  public:
  TssTempList(bool AllowCreation, bool AllowDeletion, bool AllowNULL) :
    F_ALLOW_CREATE(AllowCreation), F_ALLOW_DELETE(AllowDeletion),
    F_ALLOW_NULL(AllowNULL)//Initialize all const data members.
  {
    FList = new TList;
  }//End of class constructor.

  virtual __fastcall ~TssTempList()
  {
    //Have to declare using __fastcall or compiler halts with the message
    //"Virtual function 'TssTempList< T > ::~TssTempList()' conflicts
    //with base class TObject.
    Clear();//Have to call this manually.  TObject's destructor doesn't
      //contain an explicit call to a virtual TObject->Clear() method.
    delete FList;//Will redundantly yet automatically delete void pointers.
  }//End of class destructor.

  inline int Add(T * Item);//First form.  Defined later in file.

  inline T * Add();//Second form.  Defined later in file.

  inline void Clear();

  int __property Count = {read=GetListCount, write=SetListCount};
    //Have to use a function to access FList->Count.

  inline void Delete(int Index);

  void Exchange(int Index1, int Index2)
  {
    FList->Exchange(Index1, Index2);
  }

  void Extract(int Index)
    //In BCB 5.0 TList->Delete() and TList->Extract() only remove pointers.
    //Descendant classes define Delete() to free object memory.
  {
    FList->Delete(Index);
  }

  T * First(void)
  {
    return reinterpret_cast< T * >(FList->First());
  }

  int IndexOf(T * Item)
  {
    return FList->IndexOf(Item);
  }

  inline void Insert(int Index, T * Item);//First form.  Defined later.

  inline T * Insert(int Index);//Second form.  Defined later.

  __property T * Items[int Index] = {read=Get, write=Put};
    //Notice that a call to Put() automatically passes both Index and Item.

  T * Last(void)
  {
    return reinterpret_cast< T * >(FList->Last());
  }

  void Move(int CurIndex, int NewIndex)
  {
    FList->Move(int CurIndex, int NewIndex);
  }

  void Pack(void)
  {
    FList->Pack(void);
  }

  inline int Remove(T * Item);//Defined later.

  void Sort()
  {
    FList->Sort(SortingFunc);//Passes private member function.  See above.
  }
};//End of TssTempList.
//---------------------------------------------------------------------------
//This method throws an EInvalidOperation exception and aborts it's execution
//if Item = NULL and F_ALLOW_NULL is false.
template <class T> int TssTempList< T * >::Add(T * Item)//The first form.
{
  int Result = -1;
  try
  {
    if(Item == NULL && !F_ALLOW_NULL)
      throw EInvalidOperation("This instance of TssTempList was not created "
              " with the ability to store NULL objects.  The current "
              "call to Add\(\) \(the first form\) will now abort.");
    Result = FList->Add(Item);
  }//End of try.
  catch(EInvalidOperation &E)
  {
    ::Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch.
  return Result;
}//End of TssTempList< T >::Add() (the first form).
//---------------------------------------------------------------------------
//Returns the pointer to an automatically created "T" instance.  This method
//throws an EInvalidOperation exception and aborts it's execution if
//F_ALLOW_CREATE is false.
template <class T> T * TssTempList< T * >::Add()//The second form.
{
  T * pT = NULL;
  try
  {
    if(!F_ALLOW_CREATE)
      throw EInvalidOperation("This instance of TssTempList was not "
              "created with the option of creating objects for storage.  "
              "The current call to Add\(\) \(the second form\) will now "
              "terminate.");
    pT = new T;
    FList->Add(pT);
  }//End of try.
  catch(EInvalidOperation &E)
  {
    ::Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch(EInvalidOperation &E).
  catch(std::bad_alloc &e)//Exception implicitly passed by reference.
  //Traps problems with memory allocation for pT.
  {
    ::Beep(1000, 1000);
    ShowMessage(e.what());//Give the message supplied by the memory system.
    ShowMessage("The system could not allocate the memory needed for the "
            "object requested by the current call to TssTempList"
            "->Add\(\) \(the second form\).  As such, the object could "
            "not be created and could be added to the list.  The current "
            "call to Add\(\) \(the second form\) will now terminate.");
    pT = NULL;//Just in case.
  }//End of catch(std::bad_alloc).
  return pT;
}//End of TssTempList< T >::Add() (the second form).
//---------------------------------------------------------------------------
template <class T> void TssTempList< T * >::Clear()
{
  if(F_ALLOW_DELETE)
  {
    for(int i = 0; i < FList->Count; i++)
      delete Get(i);//Delete the object.
  }//End of if(F_ALLOW_DELETE).
  FList->Clear();//Delete pointers.
}//End of TssTempList< T >::Clear().
//---------------------------------------------------------------------------
template <class T> void TssTempList< T * >::Delete(int Index)
{
  if(F_ALLOW_DELETE)
    delete Get(Index);//Delete the object.
  FList->Delete(Index);//Delete pointer.
}//End of TssTempList< T >::Delete().
//---------------------------------------------------------------------------
//This method throws an EInvalidOperation exception and aborts it's execution
//if Item = NULL and F_ALLOW_NULL is false.
template <class T> void TssTempList< T * >::Insert(int Index, T * Item)
{//First form.
  try
  {
    if(Item == NULL && !F_ALLOW_NULL)
      throw EInvalidOperation("This instance of TssTempList was not "
              "created with the ability to store NULL objects.  The current "
              "call to Insert\(\) will now abort.");
    FList->Insert(Index, Item);
  }//End of try.
  catch(EInvalidOperation &E)
  {
    ::Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch.
}//End of TssTempList< T >::Insert() (the first form).
//---------------------------------------------------------------------------
//Returns the pointer to an automatically created "T" instance.  This method
//throws an EInvalidOperation exception and aborts it's execution if
//F_ALLOW_CREATE is false.
template <class T> T * TssTempList< T * >::Insert(int Index)//Second form.
{
  T * pT = NULL;
  try
  {
    if(!F_ALLOW_CREATE)
      throw EInvalidOperation("This instance of TssTempList was not "
              "created with the option of creating objects for storage.  "
              "The current call to Insert\(\) \(the second form\) will "
              "now terminate.");
    pT = new T;
    FList->Insert(Index, pT);
  }//End of try.
  catch(EInvalidOperation &E)
  {
    ::Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch(EInvalidOperation &E).
  catch(std::bad_alloc &e)//Exception implicitly passed by reference.
  //Traps problems with memory allocation for pT.
  {
    ::Beep(1000, 1000);
    ShowMessage(e.what());//Give the message supplied by the memory system.
    ShowMessage("The system could not allocate the memory needed for the "
            "object requested by the current call to TssTempList"
            "->Insert\(\) \(the second form\).  As such, the object could "
            "not be created and could be inserted into the list.  The current "
            "call to Insert\(\) \(the second form\) will now terminate.");
    pT = NULL;//Just in case.
  }//End of catch(std::bad_alloc).
  return pT;
}//End of TssTempList< T >::Insert() (the second form).
//---------------------------------------------------------------------------
//Needed by Items[] property.  Can't just call Delete().  That would delete
//the FList[Index] pointer.  We just need to re-assign the existing pointer.
//Notice that a call from Items[] automatically passes both Index and Item.
template <class T> void TssTempList< T * >::Put(int Index, T * Item)
{
  try
  {
    if(Item == NULL && !F_ALLOW_NULL)
      throw EInvalidOperation("This instance of TssTempList was not "
              "created with the ability to store NULL objects.  The current "
              "call to Put\(\) \(used by Items[]\) will now abort.");
    if(F_ALLOW_DELETE)
      delete Get(Index);//Delete the object.
    FList->Items[Index] = Item;
  }//End of try.
  catch(EInvalidOperation &E)
  {
    ::Beep(1000, 1000);
    ShowMessage(E.Message);
  }//End of catch.
}//End of TssTempList< T >::Put().
//---------------------------------------------------------------------------
//TList::Remove() calls TList->IndexOf() internally.  Therefore,
//TList::Remove() returns -1 if the pointer didn't exist in the list.
template <class T> int TssTempList< T * >::Remove(T * Item)
{
  int Result = FList->Remove(Item);
  if(F_ALLOW_DELETE && Result >= 0)
    delete Item;
  return Result;
}//End of TssTempList< T >::Remove().
//---------------------------------------------------------------------------
//This method tests both the F_ALLOW_CREATE and F_ALLOW_NULL members and
//throws exceptions accordingly.
template <class T> void TssTempList< T * >::SetListCount(int NewCount)
{
  try
  {
    if(NewCount < 0 || NewCount > MaxListSize)
      //MaxListSize is defined in Classes.pas as MaxInt / 16 (134217727).
      FList->Error("The NewCount parameter passed into the current call to "
              "TssTempList->SetListCount\(\) is out of bounds.", 0);
        //Second parameter has no effect in this instance.
    if(NewCount > FList->Count)//Try to add something to the list.
    {
      if(F_ALLOW_CREATE)
      {
        int OldCount = FList->Count;
        T * ArrayOfT = new T[NewCount - OldCount];//Allocate objects.
          //The global operator new[]() throws a std::bad_alloc exception
          //if memory allocation attempt fails.  Not true in BCB 4.0??
        FList->Count = NewCount;//Allocate space in the list.
        //OldCount now indexes the first new pointer in FList.
        for(int i = OldCount; i < FList->Count; i++)//Add objects to list.
          FList->Items[i] = &ArrayOfT[i - OldCount];//Take address.
        //ArrayOfT pointer now falls out of scope.  Objects managed by FList.
      }//End of if(!F_ALLOW_CREATE).
      else if(F_ALLOW_NULL)
        FList->Count = NewCount;
      else
        throw EInvalidOperation("");
    }//End of if(NewCount > FList->Count).
    else if(NewCount < FList->Count)
    {
      if(!F_ALLOW_CREATE && !F_ALLOW_NULL)//Consistent with the code above.
        throw EInvalidOperation("");
      while(NewCount < FList->Count)//If the list needs to be smaller.
        Delete(NewCount);//Delete decides whether to delete objects or not.
    }//End of else if(NewCount < FList->Count).
  }//End of try.
  catch(EInvalidOperation &E)
  {
    ::Beep(1000, 1000);
    ShowMessage("This instance of TssTempList does not allow the "
                "direct manipulation of it's Count property.  The current "
                "call to SetListCount\(\) will now terminate.");
  }//End of catch(EInvalidOperation &E).
  catch(std::bad_alloc &e)//Exception implicitly passed by reference.
  //The call to ::operator new [] (by ArrayOfT = new T[NewCount - OldCount])
  //failed.  The system couldn't allocate memory for all objects requested.
  {
    ::Beep(1000, 1000);
    ShowMessage(e.what());//Give the message supplied by the memory system.
    ShowMessage("The system could not allocate the memory needed for the "
            "objects requested by the current call to TssTempList"
            "->SetListCount\(\).  As such, no new objects could be added "
            "to the list.  The current call to SetListCount\(\) will now "
            "terminate.");
  }//End of catch(std::bad_alloc).
}//End of TssTempList< T >::SetListCount().
//---------------------------------------------------------------------------
#endif

