#include <vcl.h>
#pragma hdrstop

#include "WebServer.h"
//---------------------------------------------------------------------------

#pragma package(smart_init)


// This is the set of known graphic file types
// for the web server.
bool IsImageType(AnsiString sFileExt)
{
  return
    (!sFileExt.AnsiCompareIC("jpg")   ||
     !sFileExt.AnsiCompareIC("jpeg")  ||
     !sFileExt.AnsiCompareIC("gif")   ||
     !sFileExt.AnsiCompareIC("bmp")   ||
     !sFileExt.AnsiCompareIC("png"));
}

// =====================================================
// TWebServer Implementation
// =====================================================
__fastcall TWebServer::TWebServer(TComponent* AOwner)
  : TComponent(AOwner)
{
  ServerSocket = new TServerSocket(this);
  ServerSocket->Active = false;
  ServerSocket->Name = "ServerSocket";
  ServerSocket->ServerType = stNonBlocking;
  ServerSocket->OnClientRead = ProcessRequest;
  Port = 80;
}

bool TWebServer::Start()
{
  if (ServerSocket->Active) {
    return false;
  }
  try {
    // Assign TCP/IP Port to use
    ServerSocket->Port = Port;
    // Start the server
    ServerSocket->Active = true;
  }
  catch (Exception& e) {
    MessageBox(NULL, e.Message.c_str(), "Error Starting Server", MB_ICONEXCLAMATION);
    ServerSocket->Active = false;
  }
  return ServerSocket->Active;
}

void TWebServer::Stop()
{
  // Stop the server
  ServerSocket->Active = false;
}

AnsiString TWebServer::GetContentType(AnsiString sFileExt)
{
  // Default to text content type
  AnsiString sType = String("Content-type: text/") + sFileExt;

  if (IsImageType(sFileExt)) {
    // File is an image
    sType = String("Content-type: image/") + sFileExt;
  }

  if (!sFileExt.AnsiCompareIC("zip")   ||
      !sFileExt.AnsiCompareIC("doc")  ||
      !sFileExt.AnsiCompareIC("pdf"))
  {
    // File is an application document
    sType = String("Content-type: application/") + sFileExt;
  }

  return sType;
}

// This function contains the core functionality of the web server.
// It takes a request and generates output to a stream.
bool TWebServer::GenerateHTML(AnsiString sResource, TMemoryStream* Output)
{
  bool bResult = false;
  if (FileExists(sResource)) {
    // Send the specified file to the output stream
    Output->LoadFromFile(sResource);
    bResult = true;
  }
  return bResult;
}

void __fastcall TWebServer::ProcessRequest(TObject *Sender, TCustomWinSocket *Socket)
{
  // Read incoming data from the client.
  AnsiString sInput = Socket->ReceiveText();

  // Put the request into a TStringList, to
  // separate the request header lines.
  TStringList* Request = new TStringList();
  Request->CommaText = sInput;

  // Create another TStringList to hold the response
  TStringList* Response = new TStringList();

  // Make sure the request is a HTTP GET. The server
  // doesn't support other request types.
  if (Request->Strings[0] == "GET") {

    // The second item in the request is the
    // file/resource to retrieve.
    AnsiString sResource = Request->Strings[1];

    // Build the complete filename for the requested file.
    AnsiString sRequestedFile = WebDir + sResource;

    // If the request ends in a slash, append the default filename.
    char cEnd = sRequestedFile[sRequestedFile.Length()];
    if (cEnd == '\\' || cEnd == '/') {
      sRequestedFile += "index.html";
    }

    // Get file extension. Use SubString() to omit the '.'
    AnsiString sFileExt = ExtractFileExt(sRequestedFile).SubString(2,10);

    // Get content type description from file extension
    AnsiString sContentType = GetContentType(sFileExt);

    // Create an output stream for HTML
    TMemoryStream* HTMLData = new TMemoryStream();
    AnsiString sHTTPStatus;

    // Generate output.
    if (GenerateHTML(sRequestedFile, HTMLData)) {
      sHTTPStatus = "HTTP/1.0 200 OK";
    }
    else {
      sHTTPStatus = "HTTP/1.0 404 NOT FOUND";
    }

    // Create response header
    Response->Add(sHTTPStatus);
    Response->Add("MIME-version: 1.0");
    Response->Add(sContentType);
    Response->Add("");

    // Send response header
    Socket->SendText(Response->Text);

    // Send HTML
    if (HTMLData->Size > 0) {
      Socket->SendBuf(HTMLData->Memory, HTMLData->Size);
    }
    else {
      // If the requested file is not an image, return
      // a HTML page indicating the bad filename. Otherwise,
      // just return the header.
      if (!IsImageType(sFileExt)) {
        Response->Add("<HTML><HEAD><TITLE>File Not Found</TITLE></HEAD>");
        Response->Add("<BODY><H1>File Not Found</H1>");
        Response->Add("The document you requested: ") ;
        Response->Add(Request->Strings[1]);
        Response->Add("<BR>was not found.</BODY></HTML>");
        Socket->SendText(Response->Text);
      }
    }
    delete HTMLData;
  }
  else {
    // Invalid request type.
    Response->Add("HTTP/1.0 400 BAD REQUEST");
    Response->Add("MIME-version: 1.0");
    Response->Add("Content-type: text/html\n\n");
    Response->Add("<HTML><HEAD><TITLE>Invalid Request Type</TITLE></HEAD>\n");
    Response->Add("<BODY><H1>Invalid Request Type.</H1>\n");
    Response->Add("Request:<BR><PRE>");
    Response->Add(sInput);
    Response->Add("</PRE></BODY></HTML>");
    Socket->SendText(Response->Text);
  }

  delete Request;
  delete Response;

  // Done sending
  Socket->Close();
}

