
/****************************************************************************\

    SGDropTarget.cpp -- Created by Damon Chandler <dmc27@cornell.edu>

 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************

\****************************************************************************/

#include <vcl.h>
#include <grids.hpp>
#include <cassert>
#include "SGDropTarget.h"
//===========================================================================

TSGDropTarget::TSGDropTarget(
   TWinControl* ATargetWnd)
	: IDropTarget(), num_refs_(1),
   TargetWnd_(ATargetWnd),
   format_ok_(false)
{
   assert(TargetWnd_ != NULL);
}
//---------------------------------------------------------------------------

TSGDropTarget::~TSGDropTarget()
{
}
//---------------------------------------------------------------------------

//
// IUnknown Interface has three member
// functions:
// ------------------------------------
// QueryInterface, AddRef, and Release
// ------------------------------------
// Implement these as usual for any COM
// interface descendant class
//

STDMETHODIMP TSGDropTarget::
  QueryInterface(
  REFIID iid, void **ppv)
{
  // tell other objects
  // about our capabilities
  if (iid == IID_IUnknown ||
      iid == IID_IDropTarget)
  {
    *ppv = this;
    AddRef();
    return NOERROR;
  }
  *ppv = NULL;
  return E_NOINTERFACE;
}
//---------------------------------------------------------------------------

STDMETHODIMP_(ULONG) TSGDropTarget::
  AddRef()
{
  return ++num_refs_;
}
//---------------------------------------------------------------------------

STDMETHODIMP_(ULONG) TSGDropTarget::
  Release()
{
  if(--num_refs_ == 0)
  {
    delete this;
    return 0;
  }
  return num_refs_;
}
//---------------------------------------------------------------------------

//
// IDropTarget Interface provides the
// drop implementation via the
// DragEnter, DragOver, DragLeave, and
// Drop member functions.  Implement
// these members to handle specific
// formats and user feedback.
//

STDMETHODIMP TSGDropTarget::DragEnter(
  LPDATAOBJECT pDataObj,
  DWORD grfKeyState, POINTL pt,
  LPDWORD pdwEffect)
{
  FORMATETC fmtetc;
  fmtetc.cfFormat = CF_TEXT;
  fmtetc.ptd = NULL;
  fmtetc.dwAspect = DVASPECT_CONTENT;
  fmtetc.lindex = -1;
  fmtetc.tymed = TYMED_HGLOBAL;

  // does the source provide CF_TEXT?
  HRESULT hRes =
    pDataObj->QueryGetData(&fmtetc);
  if (SUCCEEDED(hRes))
  {
    format_ok_ = true;
    *pdwEffect = DROPEFFECT_COPY;
  }
  else
  {
    format_ok_ = false;
    *pdwEffect = DROPEFFECT_NONE;
  }
	return NOERROR;
}
//---------------------------------------------------------------------------

STDMETHODIMP TSGDropTarget::DragOver(
  DWORD grfKeyState, POINTL pt,
  LPDWORD pdwEffect)
{
  if (format_ok_)
  {
    // store the mouse cursor point
    PMouse_ = Point(pt.x, pt.y);
    // indicate the copy effect
    *pdwEffect = DROPEFFECT_COPY;

    //
    // notify the target window of
    // the drag over event
    //
    LPARAM lParam =
      reinterpret_cast<LPARAM>(
        &PMouse_
        );
    TargetWnd_->Perform(
      WM_OLEDRAGOVER, NULL, lParam
      );
  }
  // indicate no effect
  else *pdwEffect = DROPEFFECT_NONE;
  return NOERROR;
}
//---------------------------------------------------------------------------

STDMETHODIMP TSGDropTarget::Drop(
  LPDATAOBJECT pDataObj,
  DWORD grfKeyState, POINTL pt,
  LPDWORD pdwEffect)
{
  if (format_ok_)
  {
    // initialize a FORMATETC structure
    FORMATETC fmtetc;
    fmtetc.cfFormat = CF_TEXT;
    fmtetc.ptd = NULL;    
    fmtetc.dwAspect = DVASPECT_CONTENT;
    fmtetc.lindex = -1;
    fmtetc.tymed = TYMED_HGLOBAL;

    // user has dropped on us--get the
    // text data from drag source
    STGMEDIUM smed;
    HRESULT hResult =
      pDataObj->GetData(&fmtetc, &smed);
    if (SUCCEEDED(hResult))
    {
      //
      // notify the target window
      // of the drop event
      //
      LPARAM lParam =
        reinterpret_cast<WPARAM>(
          smed.hGlobal
          );
      TargetWnd_->Perform(
        WM_OLEDRAGDROP, NULL, lParam
        );

      // free the associated memory
      ReleaseStgMedium(&smed);
    }
    else return hResult;
  }
  return NOERROR;
}
//===========================================================================





