
/****************************************************************************\

    StringGridOLEDemo.cpp -- Created by Damon Chandler <dmc27@cornell.edu>

 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************

\****************************************************************************/

#include <vcl.h>
#pragma hdrstop

#include <clipbrd.hpp>
#include <memory>
#include "StringGridOLEDemo.h"
//===========================================================================
#pragma package(smart_init)
#pragma resource "*.dfm"
#pragma link "SGDropTarget"

TForm1 *Form1;
//===========================================================================

AnsiString __fastcall CellsToText(
  TStringGrid& AGrid,
  const TGridRect& ACells)
{
  //
  // compute the total size of the
  // buffer required to hold the text of
  // each selected cell plus the tab and
  // CR/LF delimiters
  //
  int text_len = 0;
  for (int row = ACells.Top;
       row <= ACells.Bottom;
       ++row)
  {
    for (int col = ACells.Left;
         col <= ACells.Right;
         ++col)
    {
      text_len +=
        AGrid.Cells[col][row].Length();
      if (col < ACells.Right) ++text_len;
    }
    if (row < ACells.Bottom)
      text_len += 2;
  }

  //
  // fill the AnsiString with the text
  // of each cell in a tab- and
  // CR/LF-delimited format
  //
  AnsiString text;
  text.SetLength(text_len);
  text = "";
  for (int row = ACells.Top;
       row <= ACells.Bottom;
       ++row)
  {
    for (int col = ACells.Left;
         col <= ACells.Right;
         ++col)
    {
      text += AGrid.Cells[col][row];
      if (col < ACells.Right)
        text += '\t';
    }
    if (row < ACells.Bottom)
      text += "\r\n";
  }
  return text;
}
//---------------------------------------------------------------------------

void __fastcall CopyCellsToClipboard(
  TStringGrid& AGrid,
  const TGridRect& ACells)
{
  // grab the formatted text
  AnsiString text(
    CellsToText(AGrid, ACells)
    );
  // copy the text to the clipboard
  Clipboard()->AsText = text;
}
//---------------------------------------------------------------------------

void __fastcall TextToCells(
  TStringGrid& AGrid,
  const TGridCoord& AFirstCell,
  const AnsiString AText)
{
  TGridCoord cell = AFirstCell;

  // if the text has no tab-delimiters,
  // copy it as a whole string
  if (!AText.Pos('\t'))
  {
    AGrid.Cells[cell.X][cell.Y] = AText;
    return;
  }

  //
  // parse the text by scanning its
  // contents, character by character,
  // looking for tab and CR delimiters
  //
  int word_start = 1;
  const int text_len = AText.Length();
  for (int index = 1;
       index < text_len;
       ++index)
  {
    char current_char = AText[index];
    // if it's a new column
    if (current_char == '\t')
    {
      AGrid.Cells[cell.X++][cell.Y] =
         AText.SubString(
            word_start,
            index - word_start
            );
      // skip the tab character
      word_start = index + 1;
    }
    // if it's a new row
    else if (current_char == '\r')
    {
      AGrid.Cells[cell.X][cell.Y++] =
         AText.SubString(
            word_start,
            index - word_start
            );
      cell.X = AFirstCell.X;
      // skip the CR/LF characters
      word_start = index + 2;
    }
  }
}
//---------------------------------------------------------------------------

void __fastcall PasteCellsFromClipboard(
  TStringGrid& AGrid,
  const TGridCoord& AFirstCell)
{
  // extract the text
  AnsiString text(Clipboard()->AsText);
  if (!text.IsEmpty())
  {
    // prevent grid updates
    AGrid.Rows[0]->BeginUpdate();
    try
    {
      // fill the cells with the text
      TextToCells(
        AGrid, AFirstCell, text
        );
    }
    catch (...)
    {
       // restore grid updates
       AGrid.Rows[0]->EndUpdate();
       throw;
    }
    // restore grid updates         
    AGrid.Rows[0]->EndUpdate();
  }
}
//---------------------------------------------------------------------------

// OnKeyDown event handler
void __fastcall TForm1::
  StringGrid1KeyDown(TObject *Sender,
  WORD &Key, TShiftState Shift)
{
  if (Key == 'C' &&
      Shift.Contains(ssCtrl))
  {
    CopyCellsToClipboard(
      *StringGrid1,
      StringGrid1->Selection
      );
  }
  else if (Key == 'V' &&
           Shift.Contains(ssCtrl))
  {
    PasteCellsFromClipboard(
      *StringGrid1,
      StringGrid1->Selection.TopLeft
      );
  }
}
//---------------------------------------------------------------------------

__fastcall TForm1::TForm1(
  TComponent* Owner) : TForm(Owner)
{
  OleInitialize(NULL);
   
  pDropTarget_ =
     reinterpret_cast<IDropTarget*>(
        new TSGDropTarget(this)
        );
  CoLockObjectExternal(
   pDropTarget_, true, true
   );
  RegisterDragDrop(
   StringGrid1->Handle, pDropTarget_
   );
}
//---------------------------------------------------------------------------

__fastcall TForm1::~TForm1()
{
  RevokeDragDrop(StringGrid1->Handle);
  CoLockObjectExternal(
   pDropTarget_, false, true
   );
  pDropTarget_->Release();

  OleUninitialize();
}
//---------------------------------------------------------------------------

void __fastcall TForm1::WMOleDragOver(
  TMessage& AMsg)
{
  TPoint* pPMouse =
    reinterpret_cast<TPoint*>(
      AMsg.LParam
      );
  *pPMouse =
    StringGrid1->ScreenToClient(
      *pPMouse
      );

  int col, row;
  StringGrid1->MouseToCell(
    pPMouse->x, pPMouse->y, col, row
    );
  if (col >= StringGrid1->FixedCols &&
      row >= StringGrid1->FixedRows)
  {
    StringGrid1->Col = col;
    StringGrid1->Row = row;
  }
}
//---------------------------------------------------------------------------

void __fastcall TForm1::WMOleDragDrop(
  TMessage& AMsg)
{
  HGLOBAL hData =
    reinterpret_cast<HGLOBAL>(
      AMsg.LParam
      );
  char* pText =
    reinterpret_cast<char*>(
      GlobalLock(hData)
      );
  try
  {
    AnsiString text(pText);
    GlobalUnlock(hData); hData = NULL;

    const int col = StringGrid1->Col;
    const int row = StringGrid1->Row;
    StringGrid1->Cells[col][row] = text;
  }
  catch (...)
  {
    if (hData) GlobalUnlock(hData);
    throw;
  }
}
//===========================================================================


