#include "Renderer.h"

#define GIVE 5 // arbitrary inset value for fake lighting of panel

#define texture_filter (unsigned short)0xffff - (CELL_X_SIZE - 1)
#define FreeZero(var) if (var) delete var; var = NULL

#ifdef __INTEL__
#define pi 3.141592654 		// not req in mac math kit			
#endif
#define deg_to_rad(deg) pi*deg/180
#define rad_to_deg(rad) 180*rad/pi
#define diff(x1, x2)	x2 - x1

inline float fmax(float v, float m) { return v > m ? m : v; }
inline float fmin(float v, float m) { return v < m ? m : v; }
 
Renderer::Renderer(int screen_width, int screen_height, void *inst, int vList_Size)
	:
	distances(NULL),
#if defined(__INTEL__)
	primarySurface(NULL),
	directDraw(NULL),
#endif
	num_vertices(0),
	tri_list(NULL),
	panel_list(NULL),
	panel_listMoving(NULL),
	mem_alloc_error(0),
	window(NULL),
	clipping(0),
	using_DD(0),
	valid(0),		// valid is set in subclass after testing super
	vListSize(vList_Size),
	last_mesh_index(0),
	last_panel_index(0)
{
	vertex_list = new Vertex3D[vListSize];
	if (!vertex_list) {
		do {
			vListSize -= 64;
			if (vListSize > 0)
				vertex_list = new Vertex3D[vListSize];
			else {
				vListSize = 0;
				break;
			}
		} while (!vertex_list);
	}
	
#if defined(__INTEL__)
	appInstance = (HINSTANCE)inst;
#else
	appInstance = NULL; inst;
#endif

	mNormal.Set(1.0f);
	mHi.Set(10.0f);
	mLow.Set(0.1f);

#if defined(macintosh)

	
	GDHandle		colourGDevice, deepestGDevice;
	long			deviceDepth, maxDepth;
		
	maxDepth = 0;
	for (colourGDevice = GetDeviceList(), deepestGDevice = NULL; colourGDevice;
			colourGDevice = GetNextDevice (colourGDevice)) {
		deviceDepth = (*(*colourGDevice)->gdPMap)->pixelSize;
		if (deviceDepth > maxDepth) {
			maxDepth = deviceDepth;
			deepestGDevice = colourGDevice;
		}
	}
	gDevice = deepestGDevice;
	windowRect = (*deepestGDevice)->gdRect;
	Shrink(&windowRect, screen_width, screen_height);
	window = NewCWindow(NULL, &windowRect, "\p", true, plainDBox, NULL, false, 0);
	screen_width = windowRect.right - windowRect.left;
	screen_height = windowRect.bottom - windowRect.top;
	oldScreenDepth = (*(*gDevice)->gdPMap)->pixelSize;	
#else			
	windowRect.left = 0;
	windowRect.top = 0;
	windowRect.right = GetSystemMetrics(SM_CXFULLSCREEN);
	windowRect.bottom = GetSystemMetrics(SM_CYFULLSCREEN);
	Shrink(&windowRect, screen_width, screen_height);
	screen_width = windowRect.right - windowRect.left;
	screen_height = windowRect.bottom - windowRect.top;
	window = CreateWindow(szAppName, "", WS_POPUP, windowRect.left, 
		windowRect.top, screen_width, screen_height, 
		NULL, NULL, appInstance, NULL);

#endif

	viewport_width = screen_width;
	viewport_height = screen_height;
	viewport_middle = viewport_height >> 1;	
		
	Vertex3D::InitStatic(100, 100, 100, viewport_width, viewport_height);
	lightList[0].SetupAsLight(L3D_LOCAL, 0.4, -40, -40, 10);
	InitVertexList();
	
	MoveWorld(0);
	return;
} 

Renderer::~Renderer()
{
	DeleteTriList();
	DeletePanelList();

#if defined(macintosh)
	if (window) DisposeWindow(window);	

#elif defined (__INTEL__)
	if (directDraw) {
		directDraw->RestoreDisplayMode();
		directDraw->Release();
	}
	if (primarySurface) {
		primarySurface->Release();
	}		

#endif	
	
}

void Renderer::InitVertexList()
{
	for (int n = 0; n < vListSize; n++)
		vertex_list[n].alive = 0;
}

void Renderer::DeleteTriList()		
{
	Tri3D *active_tri, *next_tri;
	active_tri = tri_list;
	while (active_tri) {		
		next_tri = active_tri->next;
		delete active_tri;
		active_tri = next_tri;
	}
	tri_list = NULL;	
}

void Renderer::DeletePanelList()		
{
	panel_list->Destroy();
	/*
	Panel *active_panel, *next_panel;
	active_panel = panel_list;
	while (active_panel) {		
		next_panel = active_panel->next;
		delete active_panel;
		active_panel = next_panel;
	}
	*/
	delete panel_list;
	panel_list = NULL;	
}

void Renderer::MoveWorld(long keys)
{
	float rx, ry, rz;
	rx = ry = rz = 0.0f;
	float tx, ty, tz;
	tx = ty = tz = 0.0f;
		
	if (keys & k_shift)
		mCurrent = &mHi;
	else
		mCurrent = &mNormal;
	if (keys & k_ctrl) { // dodge etc
		if (keys & k_left)
			tx = mCurrent->mx;
		else if (keys & k_right)
			tx = -mCurrent->mx;
		if (keys & k_up)
			ty = mCurrent->my;
		else if (keys & k_down)
			ty = -mCurrent->my;
	} 
	else {
		if (keys & k_left)
			ry = -mCurrent->ry;
		else if (keys & k_right)
			ry = mCurrent->ry;
		if (keys & k_up)
			tz = -mCurrent->mz;
		else if (keys & k_down)
			tz = mCurrent->mz;
		if (keys & k_pageup)
			rx = -mCurrent->rx;
		else if (keys & k_pagedown)
			rx = mCurrent->rx;
		if (keys & k_home)
			rz = -mCurrent->rz;
		else if (keys & k_insert)	
			rz = mCurrent->rz;
	}
	TransformWorld(tx, ty, tz, rx, ry, rz);
}

void Renderer::TranslateVertex(
	int v, 
	float tx, float ty, float tz, 
	long t)
{
	vertex_list[v].Translate(tx, ty, tz, t);			
}

void Renderer::RotateVertex(
	int v, 
	float rx, float ry, float rz,
	long t)
{
	vertex_list[v].Rotate(rx, ry, rz, t);			
}

void Renderer::TransformVertex(
	int v, 
	float tx, float ty, float tz, 
	float rx, float ry, float rz)
{
	vertex_list[v].Transform(tx, ty, tz,	
		rx, ry, rz, T_POSTROTATION | T_RELATIVE);			
}

void Renderer::PositionVertex(
	int v, 
	float tx, float ty, float tz, 
	float rx, float ry, float rz)
{
	vertex_list[v].Transform(tx, ty, tz,	
		rx, ry, rz, T_POSTROTATION | T_ABSOLUTE);			
}

void Renderer::TransformWorld(
	float tx, float ty, float tz, 
	float rx, float ry, float rz)
{
	// For remembering our "camera" coords
	vertex_list->worldPos_X += tx;
	vertex_list->worldPos_Y += ty;
	vertex_list->worldPos_Z += tz;
	vertex_list->worldRot_X += rx;
	vertex_list->worldRot_Y += ry;
	vertex_list->worldRot_Z += rz;
	
	for (int n = 0; n < num_vertices; n++)
		vertex_list[n].Transform(tx, ty, tz,	
			rx, ry, rz, T_PREROTATION | T_ABSOLUTE);			
}

void Renderer::PositionWorld(
	float tx, float ty, float tz, 
	float rx, float ry, float rz)
{
	// For remembering our "camera" coords
	vertex_list->worldPos_X += tx;
	vertex_list->worldPos_Y += ty;
	vertex_list->worldPos_Z += tz;
	vertex_list->worldRot_X += rx;
	vertex_list->worldRot_Y += ry;
	vertex_list->worldRot_Z += rz;
	
	for (int n = 0; n < num_vertices; n++)
		vertex_list[n].Transform(tx, ty, tz,	
			rx, ry, rz, T_PREROTATION | T_ABSOLUTE);			
}

int Renderer::AddVertex(
	float x, float y, float z, 
	float r, float g, float b, long vData)
{
	if (num_vertices >= vListSize) return -1; // outa luck!

	Vertex3D &vert = vertex_list[num_vertices++];
	vert.Install(x, y, z, r, g, b, vData);
	return num_vertices - 1;	
}

Tri3D *Renderer::AddTri(
	int vi1, int vi2, int vi3, 
	Uv uv1, Uv uv2, Uv uv3, 
	Rgb &p_rgb,
	int tmp, int type, long data)
{
	Tri3D *active_tri, *new_tri;
	active_tri = tri_list;
	while (active_tri) {
		if (active_tri->next)
			active_tri = active_tri->next;
		else
			break;	
	}
	
	new_tri = new Tri3D(
		vi1, vi2, vi3, uv1, uv2, uv3, 
		p_rgb, tmp, type, lightList, vertex_list, data);
	
	if (!active_tri) // must be first one
		tri_list = new_tri;
	else
		active_tri->next = new_tri;
	return new_tri;		
}

Quad Renderer::AddQuad(
	int vi1, int vi2, int vi3, 
	int vi4, Rgb &p_rgb, int tmp, int type, long dat)
{
	Quad ret;
	Uv uv1(0, 1, O_LEFT, O_TOP);
	Uv uv2(0, 0, O_LEFT, O_BOTTOM);
	Uv uv3(1, 0, O_RIGHT, O_BOTTOM);
	Uv uv4(1, 1, O_RIGHT, O_TOP);
	
	ret.t1 = AddTri(vi1, vi2, vi4, uv1, uv2, uv4, p_rgb, tmp, type);
	ret.t2 = AddTri(vi2, vi3, vi4, uv2, uv3, uv4, p_rgb, tmp, type);	
	ret.r = p_rgb.r;
	ret.g = p_rgb.g;
	ret.b = p_rgb.b;
	ret.tex = tmp;
	ret.data = dat;	
	return ret;
}

int Renderer::CreateTransformPanel(	
	float width, float height, 
	int mesh_cols, int mesh_rows,
	float rcol, float gcol, float bcol,	
	float tx, float ty, float tz,
	float rx, float ry, float rz,
	int tex_index, int vData)
{
	Panel *pp;
	if (!panel_list) panel_list = pp = new Panel(++last_panel_index);
	else pp = panel_list->Append();
	if (!pp->Create(this, width, height, mesh_cols, mesh_rows, vData))		
		return -1;
	
	int meshPointsX = mesh_cols + 1;
	int meshPointsY = mesh_rows + 1;
	if (meshPointsX < 2) meshPointsX = 2;
	if (meshPointsY < 2) meshPointsY = 2;

	float posx = -(meshPointsX / 2.0f);
	float posy = -(meshPointsY / 2.0f);
	float posz = 0;
	float vx, vy, vz;
	float colsize, rowsize;

	int q1, q2, q3, q4;
	int q = 0;

	colsize = width / (meshPointsX - 1);
	rowsize = height / (meshPointsY - 1);

	int firstv = num_vertices, lastv = -1, ext;
	float slope, s = meshPointsX * meshPointsY;
	Rgb t_rgb(rcol, gcol, bcol);
		
	// make a simple mesh	
	for (int row = 0; row < meshPointsY; row ++) {
		for (int col = 0; col < meshPointsX; col++) {
			slope = s - (float)(row * meshPointsX + col) + GIVE;
			vx = posx + col * colsize;
			vy = posy + row * rowsize;
			vz = posz;
			if ((lastv = AddVertex(vx, vy, vz, 
				rcol * fmax((slope / s), 1.0f), 
				gcol * fmax((slope / s), 1.0f), 
				bcol * fmax((slope / s), 1.0f), vData)) < 0)
				goto skip;
			TransformVertex(lastv, tx, ty, tz, rx, ry, rz);
		}		
	}
skip:	
	for (row = 0; row < (meshPointsY - 1); row++) {
		int offset = firstv + row * meshPointsX;
		for (int col = 0; col < (meshPointsX - 1); col++) {
			if ((ext = (offset + col + meshPointsX + 1)) > lastv) return -1;
			q1 = offset + col;
			q2 = offset + col + meshPointsX;
			q3 =  ext;
			q4 = offset + col + 1;			
			// pp.quadList[q++] = AddQuad(q1, q2, q3, q4, t_rgb, tex_index);
			pp->AddQuad(q++, q1, q2, q3, q4, t_rgb, tex_index);
		}
	}
	return last_panel_index;
}

int Renderer::CreateTransformQuad(	
	float width, float height, 
	float rcol, float gcol, float bcol,	
	float tx, float ty, float tz,
	float rx, float ry, float rz,
	int tex_index, int vData)
{	
	int meshPointsX = 2;
	int meshPointsY = 2;
	float posx = -(meshPointsX / 2.0f);
	float posy = -(meshPointsY / 2.0f);
	float posz = 0;
	float vx, vy, vz;
	float colsize, rowsize;
	int q1, q2, q3, q4;
	int q = 0;
	colsize = width;
	rowsize = height;
	int firstv = num_vertices, lastv = -1, ext;
	float slope, s = meshPointsX * meshPointsY;
	Rgb t_rgb(rcol, gcol, bcol);
		
	// make a 4 point quad	
	for (int row = 0; row < meshPointsY; row ++) {
		for (int col = 0; col < meshPointsX; col++) {
			slope = s - (float)(row * meshPointsX + col) + GIVE;
			vx = posx + col * colsize;
			vy = posy + row * rowsize;
			vz = posz;
			if ((lastv = AddVertex(vx, vy, vz, 
				rcol * fmax((slope / s), 1.0f), 
				gcol * fmax((slope / s), 1.0f), 
				bcol * fmax((slope / s), 1.0f), vData)) < 0)
				goto skip;
			TransformVertex(lastv, tx, ty, tz, rx, ry, rz);
		}		
	}
skip:	
	for (row = 0; row < (meshPointsY - 1); row++) {
		int offset = firstv + row * meshPointsX;
		for (int col = 0; col < (meshPointsX - 1); col++) {
			if ((ext = (offset + col + meshPointsX + 1)) > lastv) return -1;
			q1 = offset + col;
			q2 = offset + col + meshPointsX;
			q3 =  ext;
			q4 = offset + col + 1;			
			AddQuad(q1, q2, q3, q4, t_rgb, tex_index, TRI_MOVEABLE, vData);
		}
	}
	return 1;
}

void Renderer::ColorPanelCell(
	int panelID, int posx, int posy,
	float red, float green, float blue, int tmapIndex, int dat)
{
	Panel *pp = panel_list->Find(panelID);
	if (pp)
		pp->ColorCell(posx, posy, red, green, blue, tmapIndex, dat);
}

int Renderer::CreateMesh(	
	float width, float height, 
	int mesh_cols, int mesh_rows,
	float rcol, float gcol, float bcol,	
	float tx, float ty, float tz,
	float rx, float ry, float rz,
	int tex_index, int vData)
{
	int meshPointsX = mesh_cols + 1;
	int meshPointsY = mesh_rows + 1;
	float posx = -(width / 2.0f);
	float posy = -(height / 2.0f);
	float posz = 0;
	float vx, vy, vz;
	float colsize, rowsize;
	int q1, q2, q3, q4;

	Tri3D *firstTri = NULL;
	
	if (meshPointsX < 2) meshPointsX = 2;
	if (meshPointsY < 2) meshPointsY = 2;
	
	colsize = width / (mesh_cols);
	rowsize = height / (mesh_rows);
	int firstv = num_vertices, lastv = -1, ext;
	float slope, s = meshPointsX * meshPointsY;
	Rgb t_rgb(rcol, gcol, bcol);
		
	// make a simple mesh	
	for (int row = 0; row < meshPointsY; row ++) {
		for (int col = 0; col < meshPointsX; col++) {
			slope = s - (float)(row * meshPointsX + col) + GIVE;
			vx = posx + col * colsize;
			vy = posy + row * rowsize;
			vz = posz;
			if ((lastv = AddVertex(vx, vy, vz, 
				rcol * fmax((slope / s), 1.0f), 
				gcol * fmax((slope / s), 1.0f), 
				bcol * fmax((slope / s), 1.0f), vData)) < 0)
				goto skip;
			TransformVertex(lastv, tx, ty, tz, rx, ry, rz);
		}		
	}
skip:	
	for (row = 0; row < (mesh_rows); row++) {
		int offset = firstv + row * meshPointsX;
		for (int col = 0; col < mesh_cols; col++) {
			if ((ext = (offset + col + meshPointsX + 1)) > lastv) return -1;
			q1 = offset + col;
			q2 = offset + col + meshPointsX;
			q3 =  ext;
			q4 = offset + col + 1;
			Uv uv1(0, 1, O_LEFT, O_TOP);
			Uv uv2(0, 0, O_LEFT, O_BOTTOM);
			Uv uv3(1, 0, O_RIGHT, O_BOTTOM);
			Uv uv4(1, 1, O_RIGHT, O_TOP);
			
			Tri3D *tri1 = AddTri(q1, q2, q4, uv1, uv2, uv4, t_rgb, tex_index, TRI_FIXED, vData);
			Tri3D *tri2 = AddTri(q2, q3, q4, uv2, uv3, uv4, t_rgb, tex_index, TRI_FIXED | TRI_LEFTHAND, vData);	
			if (!firstTri) {
				// we know the tris within the vertex list will be consecutive
				// so for this mesh we just remember the first for future access
				if ((last_mesh_index + 1) >= MAX_MESHES) return -1;
				last_mesh_index++;
				firstTri = tri1;
				meshList[last_mesh_index].start_tri = firstTri;
				meshList[last_mesh_index].cellsX = mesh_cols;
				meshList[last_mesh_index].cellsY = mesh_rows;
			}
		}
	}
	return last_mesh_index;
}

void Renderer::ColorMeshCell(
	int cellIndex, int posx, int posy,
	float red, float green, float blue, int tmapIndex)
{
	if (cellIndex > last_mesh_index) return;
	Mesh &mesh = meshList[cellIndex];
	int offset = posy * mesh.cellsX * 2 + posx * 2;
	Tri3D *tri = mesh.start_tri;
	Tri3D *workTri = tri + offset;
	workTri->rgb1.r = red;
	workTri->rgb2.r = red;
	workTri->rgb3.r = red;
	workTri->rgb1.g = green;
	workTri->rgb2.g = green;
	workTri->rgb3.g = green;
	workTri->rgb1.b = blue;
	workTri->rgb2.b = blue;
	workTri->rgb3.b = blue;
	workTri->texmap = tmapIndex;
	workTri++;
	workTri->rgb1.r = red;
	workTri->rgb2.r = red;
	workTri->rgb3.r = red;
	workTri->rgb1.g = green;
	workTri->rgb2.g = green;
	workTri->rgb3.g = green;
	workTri->rgb1.b = blue;
	workTri->rgb2.b = blue;
	workTri->rgb3.b = blue;
	workTri->texmap = tmapIndex;	
}

int Renderer::LoadDXFTris(
	char *fileName, int tex_index, float scale, 
	float r, float g, float b,
	float tx, float ty, float tz,
	float rx, float ry, float rz)
{
	// NOTE: ignores all but the triangle list within the DXF file
	FILE *fp = fopen(fileName, "r");
	float v1, v2, v3;
	int firstv = num_vertices, lastv = -1, v_loaded = 0;
	char buff[128];
	int meshPointsX = 0, meshPointsY = 0, ext;
	int q1, q2, q3, q4;
	Rgb t_rgb(r, g, b);
	
	if (fp) {
seq_begin:	
		while (fgets(buff, 128, fp)) {
			if (strstr(buff, "POLYLINE")) {
				while (fgets(buff, 128, fp)) {
					if (strstr(buff, " 70")) {
						fgets(buff, 128, fp); // eat next line
						fscanf(fp, " 71 %d 72 %d", &meshPointsY, &meshPointsX);
						while (fgets(buff, 128, fp)) {							
							if (strstr(buff, "VERTEX")) {
								if (fgets(buff, 128, fp) && fgets(buff, 128, fp)) {
									if (fscanf(fp, " 10 %f 20 %f 30 %f", &v1, &v2, &v3) == 3) {
										if ((lastv = AddVertex(v1 * scale, v2 * scale, v3 * scale, r, g, b)) < 0)			
											goto add_quads;
										TransformVertex(lastv, tx, ty, tz, rx, ry, rz);	
										v_loaded++;
									}
								}
							}
							else if (strstr(buff, "SEQEND"))
								goto seq_begin;
						}
					}
				}
			}
		}		
add_quads:
		fclose(fp);		
		if (v_loaded) {
			for (int row = 0; row < meshPointsY - 1; row++) {
				int offset = firstv + row * meshPointsX;
				for (int col = 0; col < meshPointsX - 1; col++) {
					if ((ext = (offset + col + meshPointsX + 1)) > lastv) return 0;
					q1 = offset + col;
					q2 = offset + col + meshPointsX;
					q3 =  ext;
					q4 = offset + col + 1;
					AddQuad(q1, q2, q3, q4, t_rgb, tex_index);
				}
			}
			return 1;
		}
	}
	return 0;
}

void Renderer::ClearLights()
{
	for (int i = 0; i < MAX_LIGHTS; i++)
		lightList[i].active = 0;
	num_lights = 0;
}

#if defined(__INTEL__)

void Renderer::Shrink(RECT *rect, int w, int h)

#else

void Renderer::Shrink(Rect *rect, int w, int h)

#endif

{
	if (w == 0 && h == 0) return;
	int screenW, screenH, diffW, diffH;
	screenW = rect->right - rect->left;
	screenH = rect->bottom - rect->top;
	if (screenW > w) {
		diffW = screenW - w;
		rect->left += (diffW >> 1);
		rect->right -= (diffW >> 1);
	}
	if (screenH > h) {
		diffH = screenH - h;
		rect->top += (diffH >> 1);
		rect->bottom -= (diffH >> 1);
	}
}

void Renderer::SetClipRegion(int inset_x, int inset_y)
{
	clipRegion = NULL;
	
#if defined(macintosh)
	Point topLeft, botRight;
	Rect clipRect;	
	GrafPtr savePort;	
	
	topLeft.h = window->portRect.left;	
	topLeft.v =  window->portRect.top;
	botRight.h = window->portRect.right;
	botRight.v = window->portRect.bottom;
	
	GetPort(&savePort);
	SetPort(window);
	LocalToGlobal(&topLeft);
	LocalToGlobal(&botRight);
	SetPort(savePort);
	
	if ((clipRegion = NewRgn()) == NULL)
		return;
	
	OpenRgn();
	
	clipRect.top = windowRect.top + inset_y;
	clipRect.left = windowRect.left + inset_x;
	clipRect.bottom = windowRect.bottom - inset_y;
	clipRect.right = windowRect.right - inset_x;
	
	FrameRoundRect(&clipRect, 20, 20);
	CloseRgn(clipRegion);

#else
		
	RECT wRect, clipRect;
	
	GetClientRect(window, &wRect);
	clipRect.top = wRect.top + inset_y;
	clipRect.left = wRect.left + inset_x;
	clipRect.bottom = wRect.bottom - inset_y;
	clipRect.right = wRect.right - inset_x;
	
	clipRegion = CreateRoundRectRgn(
		clipRect.left, clipRect.top, clipRect.right, clipRect.bottom, 20, 20);
#endif
}		

void Renderer::Error(const char *message)
{		

#if defined(macintosh)

	char ErrorStr[256];
	sprintf (&ErrorStr[1], "Error: %s\n", message);
	ErrorStr[0] = strlen (&ErrorStr[1]) - 1;
	ParamText((ConstStr255Param)ErrorStr, "\p", "\p", "\p");
	Alert(ALERTBOX, NULL);
	ParamText("\p", "\p", "\p", "\p");	

#else

	MessageBox(NULL, message, "Error", MB_ICONHAND | MB_OK);

#endif

}

void Renderer::ClearWindow()
{

#if defined(macintosh)

	GrafPtr		previousPort;
	
	GetPort (&previousPort);
	SetPort (window);
	EraseRect (&window->portRect);
	SetPort (previousPort);

#elif defined(__INTEL__)

	HWND		hWnd;
	HDC			hdc;
	RECT		aWinRect;
	hWnd = window;
	hdc = GetDC(hWnd);
	GetClientRect(hWnd, (LPRECT)&aWinRect);				
	FillRect(hdc, &aWinRect, (HBRUSH) (COLOR_WINDOW+1));				

#endif
}

