#include "Vector3D.h"

int Vector3D::static_inited = 0;

int Vector3D::world_width;
int Vector3D::world_height;
int Vector3D::world_depth;
int Vector3D::raster_width; 
int Vector3D::raster_height;
int Vector3D::raster_centerX;
int Vector3D::raster_centerY;
int Vector3D::raster_min_x;
int Vector3D::raster_min_y;
int Vector3D::raster_max_x;
int Vector3D::raster_max_y;

float Vector3D::viewport_width;
float Vector3D::viewport_height;
float Vector3D::convX;
float Vector3D::convY;
float Vector3D::z_res_mul;
float Vector3D::view_plane;
float Vector3D::scaleZX;
float Vector3D::worldPos_X;
float Vector3D::worldPos_Y;
float Vector3D::worldPos_Z;
float Vector3D::worldRot_X;
float Vector3D::worldRot_Y;
float Vector3D::worldRot_Z;

inline float fciel(float v, float m) { return v > m ? m : v; }
inline float ffloor(float v, float m) { return v < m ? m : v; }

Vector3D::Vector3D(Vector3D& p1, Vector3D& p2, VectorCreator c) 
{
	Init();
	switch(c) {
	case FROMDIFF:
		x = p2.x - p1.x; 
		y = p2.y - p1.y; 
		z = p2.z - p1.z;
		break;
	case FROMDOT:
		x = p1.x * p2.x; 
		y = p1.y * p2.y; 
		z = p1.z * p2.z;
		break;
	case FROMCROSS:
		x = p1.y * p2.z - p1.z * p2.y; 
		y = p1.z * p2.x - p1.x * p2.z; 
		z = p1.x * p2.y - p1.y * p2.x;
		break;
	case FROMSUM:
		x = p2.x + p1.x; y = p2.y + p1.y; z = p2.z + p1.z;
		break;
	default:
		x = y = z = 0.0f;
	}
	CalcBase();
	SetUnitVector();
}

void Vector3D::Setup(
	Vector3D& p1, Vector3D& p2, VectorCreator c) 
{
	Init();
	switch(c) {
	case FROMDIFF:
		x = p2.x - p1.x; 
		y = p2.y - p1.y; 
		z = p2.z - p1.z;
		break;
	case FROMDOT:
		x = p1.x * p2.x; 
		y = p1.y * p2.y; 
		z = p1.z * p2.z;
		break;
	case FROMCROSS:
		x = p1.y * p2.z - p1.z * p2.y; 
		y = p1.z * p2.x - p1.x * p2.z; 
		z = p1.x * p2.y - p1.y * p2.x;
		break;
	case FROMSUM:
		x = p2.x + p1.x; y = p2.y + p1.y; z = p2.z + p1.z;
		break;
	default:
		x = y = z = 0.0f;
	}
	CalcBase();
	SetUnitVector();
}

void Vector3D::SetupAsLight(
	int type, float power, 
	float org_x, float org_y, float org_z)
{
	Init();
	light_type = type;
	x = -org_x; y = -org_y; z = -org_z;
	strength = power;
	active = 1;
	CalcBase();
	SetUnitVector();	
}
	
void Vector3D::SetupAsNormal
	(Vector3D& p1, Vector3D& p2, Vector3D &p3) 
{
	Init();
	Vector3D vec1(p1, p2, FROMDIFF);
	Vector3D vec2(p2, p3, FROMDIFF);
	Vector3D norm(vec1, vec2, FROMCROSS);

	x = norm.x;
	y = norm.y;
	z = norm.z;
	
	CalcBase();
	SetUnitVector();
}

void Vector3D::InitStatic(
	int ww, int wh, int wd, int rw, int rh, 
	int mx, int my, float vp, 
	float wpx, float wpy, float wpz,
	float wrx, float wry, float wrz)
{
	world_width = ww;
	world_height = wh;
	world_depth = wd;
	raster_width = rw;		
	raster_height = rh;
	raster_centerX = raster_width >> 1;
	raster_centerY = raster_height >> 1;
	z_res_mul = 1.0f / (float)world_depth;	
	viewport_width = viewport_height = world_width / 2;
	convX = (float)raster_width / viewport_width; 
	convY = (float)raster_height / viewport_height;
	raster_min_x = mx;
	raster_min_y = my;
	raster_max_x = raster_width - raster_min_x - 1;
	raster_max_y = raster_height - raster_min_y - 1;
	view_plane = vp;
	scaleZX = convX / z_res_mul;
	worldPos_X = wpx;
	worldPos_Y = wpy;
	worldPos_Z = wpz;
	worldRot_X = wrx;
	worldRot_Y = wry;
	worldRot_Z = wrz;
}

void Vector3D::Init()
{ 
	r = g = b = 1.0f; 
	pix_point.x = 0; 
	pix_point.y = 0; 
	invZ = 1.0f;
	outaBounds = 0;	
	light_type = L3D_NONE;
	active = 0;
	captured = 0;
} 	
	
int Vector3D::Project()
{	
	if (z <= view_plane) return -1000;
	
	invZ = 1.0f - view_plane / z;	
	scaleZ = scaleZX / z;
		
	pix_point.x = (int)(raster_centerX + x * scaleZ + 0.5f);
	pix_point.y = (int)(raster_centerY + y * scaleZ + 0.5f);	
	
	return ClipBounds();
}					

/*
void Vector3D::Transform(float tx, float ty, float tz, float rx, float ry, float rz, long t)
{
	if (t & T_ABSOLUTE) {
		Reset();
		if (t & T_PREROTATION) {
			x += + tx;
			y += + ty;
			z += + tz;
			Rotate(rx, ry, rz, t);			
		}
		else if (t & T_POSTROTATION) {
			Rotate(rx, ry, rz, t);
			x += + tx;
			y += + ty;
			z += + tz;			
		}
	}
	else if (t & T_RELATIVE) { 	
		if (t & T_PREROTATION) {								
			x += tx;
			y += ty;
			z += tz;		
			Rotate(rx, ry, rz, t);
		}
		else if (t & T_POSTROTATION) {
			Rotate(rx, ry, rz, t);
			x += tx;
			y += ty;
			z += tz;					
		}	
	}	
}
*/

void Vector3D::Transform(
	float tx, float ty, float tz, 
	float rx, float ry, float rz, long t)
{
	if (t & T_POSTROTATION) {
		Rotate(rx, ry, rz, t);
		Translate(tx, ty, tz, t);			
	} else /*if (t & T_PREROTATION)*/ {
		Translate(tx, ty, tz, t);
		Rotate(rx, ry, rz, t);			
	}
}

void Vector3D::Translate(float tx, float ty, float tz, long t)
{
	t; // not needed here	

	x += tx;
	y += ty;
	z += tz;			
}

void Vector3D::Rotate(float rx, float ry, float rz, long t)
{
	float sin_a, cos_a, temp;
		
	if (t & T_RELATIVE) Capture(); // use local coords
	
	if (ry) {
		sin_a = sin(ry);
		cos_a = cos(ry);				
		temp = x * cos_a - z * sin_a; 		
		z = x * sin_a + z * cos_a; 
		x = temp;
	}			
	if (rx) {
		sin_a = sin(rx);
		cos_a = cos(rx);
		temp = y * cos_a - z * sin_a;		
		z = y * sin_a + z * cos_a;
		y = temp;
	}
	if (rz) {
		sin_a = sin(rz);
		cos_a = cos(rz);				
		temp = x * cos_a - y * sin_a;		
		y = x * sin_a + y * cos_a; 
		x = temp;
	}
	Release(); // move rotated vertex back to orig world pos			
}

int Vector3D::ClipBounds()
{	
	if ((pix_point.x >= raster_width)
		|| (pix_point.x < 0)
		|| (pix_point.y >= raster_height) 
		|| (pix_point.y < 0))
		return -1;
	else return 1;
}

void Vector3D::Install(
	float fx, float fy, float fz, 
	float fr, float fg, float fb, long vData)
{
	// create as a gouraud vertex
	x = fx;
	y = fy;
	z = fz;
	r = fr;
	g = fg;
	b = fb;
	alive = 1;
	active = 0;
	longData = vData;
	CalcBase();
	SetUnitVector(); // maintain a unitized version for specific calculations
} 

float Vector3D::GetDistance(Vector3D &vec)
{
	return sqrt(
		(x - vec.x) * (x - vec.x) + 
		(y - vec.y) * (y - vec.y) + 
		(z - vec.z) * (z - vec.z)); 
}

float Vector3D::GetDistance(float vx, float vy, float vz)
{
	return sqrt(
		(x - vx) * (x - vx) + 
		(y - vy) * (y - vy) + 
		(z - vz) * (z - vz)); 
}

void Vector3D::CalcDistance(Vector3D &vec)
{
	// distance from point in space
	length = sqrt(
		(x - vec.x) * (x - vec.x) + 
		(y - vec.y) * (y - vec.y) + 
		(z - vec.z) * (z - vec.z)); 
}

float Vector3D::GetLength()
{
	// return length as pure vector
	return sqrt(x * x + y * y + z * z); 
}

void Vector3D::CalcLength()
{
	// set length var as pure vector
	length = sqrt(x * x + y * y + z * z); 
}

void Vector3D::Capture()
{
 	capture_x = x - base_x; 
 	capture_y = y - base_y; 
 	capture_z = z - base_z;
 	Reset();
 	captured = 1;	
}

void Vector3D::Release()
{
	if (captured) {
		x += capture_x; 
		y += capture_y; 
		z += capture_z;
		captured = 0;
	}
}