#ifndef VECTOR3D_H
#define VECTOR3D_H

#include "WinDOS.h"
#include <math.h>
#include "Color16.h"
#include "Line2D.h"

#define WORLD_WIDTH 100.0f
#define WORLD_HEIGHT 100.0f
#define WORLD_DEPTH 100.0f
#define MAX_VERTICES 4096 // limitation for demo only!
#define MAX_LIGHTS 10
#define MAX_LIGHT_DIS 100.0f
#define Z_CLIP 1000
#define VP_MIN_X 0
#define VP_MIN_Y 0
#define DEFAULT_WIDTH 640
#define DEFAULT_HEIGHT 480
#define BOUNDS_FRAME_CHECK 6
#define VIEW_PLANE 1.0F

typedef enum {
	O_UNKNOWN,
	O_LEFT,
	O_RIGHT,
	O_TOP,
	O_BOTTOM
} Orientation;

struct Uv {
	float u, v;
	int projected_x, projected_y, preprojected_x, preprojected_y;
	int projected_x_offset, projected_y_offset;
	int clipped_x, clipped_y;
	Orientation ox, oy;
	Uv(float fu, float fv, Orientation oh, Orientation ov) : 
		u(fu), v(fv) { ox = oh; oy = ov; }
	Uv(float fu, float fv) : u(fu), v(fv) { ox = O_UNKNOWN; oy = O_UNKNOWN; }
	Uv(float f) : u(f), v(f) { ox = O_UNKNOWN; oy = O_UNKNOWN; }
	Uv() : u(0), v(0) { ox = O_UNKNOWN; oy = O_UNKNOWN; }
	void Set(Uv &uv) { u = uv.u; v = uv.v; }	
	void Scale(float valu, float valv) { u *= valu; v *= valv; }
};

typedef enum {
	FROMDIFF,
	FROMCROSS,
	FROMDOT,
	FROMSUM
} VectorCreator;

enum {
	T_INVALID = 0,
	T_ABSOLUTE = 1,
	T_RELATIVE = 2,
	T_PREROTATION = 4, 	// move before rotating
	T_POSTROTATION = 8	// move after rotating
};	

enum {
	INSIDE_X = 1,
	INSIDE_Y = 2,
	INSIDE_XY = 4,
	INSIDE_Z = 8
};

enum {
	L3D_NONE,
	L3D_SPOT,
	L3D_AMBIENT,
	L3D_LOCAL,
	L3D_DISTANT
};

typedef struct {
	float clip_x, clip_y;
	long flags;
} TexClipInfo;

enum {
	tc_x = 0,
	tc_y = 1,
	tc_flag = 2
};

class Vector3D
{	
	inline int		ClipBounds();
	inline	void	Add(const Vector3D& vec) 		{ x += vec.x; y += vec.y; z += vec.z; }
	inline 	void	Subtract(const Vector3D& vec) 	{ x -= vec.x; y -= vec.y; z -= vec.z; }
	inline 	void	Scale(float f) 					{ x *= f; y *= f; z *= f; }
	inline 	void	GetCrossProduct(const Vector3D& v2, Vector3D *retval) { 
						retval->x = y * v2.z - z * v2.y; 
						retval->y = z * v2.x - x * v2.z; 
						retval->z = x * v2.y - y * v2.x; 
					}
	inline 	float	GetDotProduct(const Vector3D& v1) { 
						return v1.unitx * unitx + v1.unity * unity + v1.unitz * unitz; 
					}
	inline 	void	Translate(float tx, float ty, float tz, long t);				
	inline	void	Rotate(float rx, float ry, float rz, long t);
	inline	void	Set(float tx, float ty, float tz) { x = tx; y = ty; z = tz; }
	inline	void	Capture();
	inline 	void	Release();
	inline	void	Reset() { x = base_x; y = base_y; z = base_z; }
	
protected:
	friend class	Renderer;
	friend class	Tri3D;
	friend class 	Quad3D;
		
	unsigned char 	alive, active, outaBounds; // incrementor for frame coherence
	unsigned long	raster_rel; // position authenticator
	
	float			GetStrength(float vx, float vy, float vz);
	float			GetStrength(Vector3D &vec);
	void			CalcStrength(Vector3D &vec);
	void			CalcLength();
	float			GetLength();
	float			GetDistance(float vx, float vy, float vz);
	float			GetDistance(Vector3D &vec);
	void			CalcDistance(Vector3D &vec);
	
	float			base_x, base_y, base_z; // zero world position
	float			length, strength;
	int 			light_type, captured;	
	long			longData;		

public:	
	float			x, y, z; 						// current position
	float 			r, g, b; 						// col 0.0 - 1.0f
	float			u, v; 							// tex coords
	float			capture_x, capture_y, capture_z;// for local coord moves
	float			unitx, unity, unitz; 			// unitized version
	Point2D 		pix_point;						// output pixel location (pre-clipped)
	float 			invZ, scaleZ;					// unitized z coords
	TexClipInfo 	tex_clip; 						// tex or Gouraud clip return values
	
	void			Init(); 
	void			CalcBase() { base_x = x; base_y = y; base_z = z; }
	void			Install(float fx, float fy, float fz, float fr, float fg, float fb, long vData);
		
	// empty construction:
					Vector3D() : x(0.0f), y(0.0f), z(1.0f) { Init(); CalcBase(); }
	// immediate initialization:				
					Vector3D(float a, float b, float c) : x(a), y(b), z(c) { Init(); CalcBase(); }
					Vector3D(Vector3D& p1, Vector3D& p2, VectorCreator c);
	
	// delayed initialization:
	void			Setup(Vector3D& p1, Vector3D& p2, VectorCreator c);				
	void			SetupAsNormal(Vector3D& p1, Vector3D& p2, Vector3D &p3) ;
	void			Setup(float a, float b, float c) { 
						Init(); x = a; y = b; z = c; CalcBase(); SetUnitVector(); 
					}
	void 			SetupAsLight(int type, float power, float org_x, float org_y, float org_z);					
	inline void		Zero() 							{ x = y = z = 0.0f; Init(); }
	void			SetUnitVector() { 
						float len = sqrt(x * x + y * y + z * z); 
						unitx = x / len; 
						unity = y / len; unitz = z / len; 
					}
	void			Transform(float tx, float ty, float tz, float rx, float ry, float rz, long tt);
	int				Project();
	long			GetData() { return longData; }		

// Common	
	static int		raster_min_x, raster_min_y, raster_max_x, raster_max_y,
					raster_width, raster_height, raster_centerX, raster_centerY,
					world_width, world_height, world_depth,			
					static_inited;
	static float	convX, convY, z_res_mul, viewport_width, viewport_height, 
					view_plane, scaleZX, worldPos_X, worldPos_Y, worldPos_Z,
					worldRot_X, worldRot_Y, worldRot_Z;
	
	static void		InitStatic(	
						int ww = WORLD_WIDTH, 
						int wh = WORLD_HEIGHT, 
						int wd = WORLD_DEPTH,
						int rw = DEFAULT_WIDTH,
						int rh = DEFAULT_HEIGHT, 
						int mx = VP_MIN_X,
						int my = VP_MIN_Y,
						float vp = VIEW_PLANE, 
						float wpx = 0.0f,
						float wpy = 0.0f,
						float wpz = 0.0f,
						float wrx = 0.0f,
						float wry = 0.0f,
						float wrz = 0.0f);
	static void		Normalize(Vector3D *v) { 
						float len = sqrt(v->x * v->x + v->y * v->y + v->z * v->z); 
						v->x /= len; v->y /= len; v->z /= len; 
					}																	
};					

// Inheritance would be tidier here but these are speed-critical
// So we need to minimize virtual functions and just share the class

typedef Vector3D Vertex3D;
typedef Vector3D Light3D;

#endif