// IndexerView.cpp

using namespace std;

#include "stdafx.h"
#include "Indexer.h"

#include "IndexerDoc.h"
#include "IndexerView.h"
#include "IndexDlg.h"
#include "ProgressDlg.h"
#include "MainFrm.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define PAGECHAR '.'	// character in left margin that identifies a page break

class Progress {
	CProgressDlg* m_pCProgressDlg;
public:
	Progress(const char* text, int range = 300)
	{
		m_pCProgressDlg = 0;
		if (theApp.IsInitialized())	{
			m_pCProgressDlg = new CProgressDlg(theApp.m_pMainWnd);
			m_pCProgressDlg->Create(IDD_PROGRESSDLG);
			m_pCProgressDlg->m_Progress.SetStep(1);
			m_pCProgressDlg->m_Progress.SetRange(0, range);
			((CMainFrame*)(theApp.m_pMainWnd))->m_wndStatusBar.SetPaneText(0, text);
			((CMainFrame*)(theApp.m_pMainWnd))->m_wndStatusBar.SendMessage(WM_PAINT);
		}
	}
	~Progress()
	{
		if (m_pCProgressDlg)	{
			((CMainFrame*)(theApp.m_pMainWnd))->m_wndStatusBar.SetPaneText(0, "");
			((CMainFrame*)(theApp.m_pMainWnd))->m_wndStatusBar.SendMessage(WM_PAINT);
			delete m_pCProgressDlg;
		}
	}
	void SetRange(int n1, int n2)
	{
		if (m_pCProgressDlg)
			m_pCProgressDlg->m_Progress.SetRange(n1, n2);
	}
	void StepIt()
	{
		if (m_pCProgressDlg)
			m_pCProgressDlg->m_Progress.StepIt();
	}
};

// ---- class to get a line of text from a CRichEditCtrl control
class EditLine	{
	char* m_pLine;
	int m_nLen;
public:
	EditLine(CRichEditCtrl& rEdit, int nLineNo);
	~EditLine()
		{ delete [] m_pLine; }
	const char* Buffer() const
		{ return m_pLine; }
	BOOL IsPageBreak() const
		{ return *m_pLine == '-'; } 
	int PageNo() const;
	int Length() const
		{ return m_nLen; }
	BOOL IsEmpty() const;
};
EditLine::EditLine(CRichEditCtrl& rEdit, int nLineNo)
{
	int nLineIndex = rEdit.LineIndex(nLineNo);
	m_nLen = rEdit.LineLength(nLineIndex);
	int nLen = max(m_nLen+1, sizeof(int));
	m_pLine = new char[nLen];
	memset(m_pLine, 0, nLen);
	rEdit.GetLine(nLineNo, m_pLine, m_nLen);
}
// ----- extract the page number from a pagebreak text line
int EditLine::PageNo() const
{
	for (int i = 0; i < m_nLen && *(m_pLine + i) == '-'; i++)
		;
	return atoi(m_pLine + i);
}
BOOL EditLine::IsEmpty() const
{
	for (int i = 0; i < m_nLen && *(m_pLine + i) == ' '; i++)
		;
	return i == m_nLen;
}
// CIndexerView
IMPLEMENT_DYNCREATE(CIndexerView, CRichEditView)
BEGIN_MESSAGE_MAP(CIndexerView, CRichEditView)
	//{{AFX_MSG_MAP(CIndexerView)
	ON_COMMAND(ID_PAGENO, OnPageno)
	ON_COMMAND(ID_BUILDINDEX, OnBuildindex)
	ON_COMMAND(ID_FIND_NEXT, OnFindNext)
	ON_UPDATE_COMMAND_UI(ID_FIND_NEXT, OnUpdateFindNext)
	ON_WM_KEYDOWN()
	ON_WM_KEYUP()
	ON_WM_CHAR()
	ON_COMMAND(ID_PAGINATE, OnPaginate)
	ON_WM_LBUTTONDOWN()
	ON_WM_CREATE()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

CIndexerView::CIndexerView()
{
	m_nFontHeight = 13;
	m_nFontWidth = 7;
	m_bControlDown = FALSE;
}
// ----- current text line number (zero-based)
int CIndexerView::CurrentLineNumber()
{
	CRichEditCtrl& rEdit = GetRichEditCtrl();
	int nLineIndex = rEdit.LineIndex(-1);	// character index of 1st char, current line
	return rEdit.LineFromChar(nLineIndex);	// current line number
}
BOOL CIndexerView::Create(LPCTSTR lpszClassName, LPCTSTR lpszWindowName, DWORD dwStyle, const RECT& rect, CWnd* pParentWnd, UINT nID, CCreateContext* pContext) 
{
	BOOL rtn = CWnd::Create(lpszClassName, lpszWindowName, dwStyle, rect, pParentWnd, nID, pContext);
	newfont.CreateFont(m_nFontHeight,m_nFontWidth,0,0,400,0,0,0,0,1,2,1,49,"Courier");
	SetFont(&newfont, TRUE);
	return rtn;
}
// ---- compare two index strings, which could be phrase, phrase1\phrase2, or phrase1\\phrase2
static const CStringArray* pArray;
static int StringCompare(const CString& s1, const CString& s2)
{
	int n1 = s1.Find('\\');
	if (n1 != -1)	{
		int n2 = s2.Find('\\');
		if (n2 == n1 && s1.Left(n1).CompareNoCase(s2.Left(n2)) == 0)	{
			char c1 = s1[n1+1];
			char c2 = s2[n2+1];
			if (c1 != c2 && (c1 == '\\' || c2 == '\\'))	{
				int l1 = (s1.GetLength() - n1) - (c1 == '\\' ? 2 : 1);
				int l2 = (s2.GetLength() - n2) - (c2 == '\\' ? 2 : 1);
				return s1.Right(l1).CompareNoCase(s2.Right(l2));
			}
		}
	}
	return s1.CompareNoCase(s2);
}
// ---- compare function for qsort
static int Compare(const void* e1, const void* e2)
{
	const CString& s1 = (*pArray)[*(const int*)e1];
	const CString& s2 = (*pArray)[*(const int*)e2];
	return StringCompare(s1, s2);
}
void CIndexerView::BuildIndex(CString& strIndexEntry)
{
	if (!strIndexEntry.IsEmpty())	{
		int nIndex = strIndexEntry.Find('\\');
		if (nIndex != -1)	{
			// --- header\entry format
			CString strHeader = strIndexEntry.Left(nIndex);
			if (strIndexEntry[nIndex+1] == '\\')
				strIndexEntry = CString("  ") + strIndexEntry.Right(strIndexEntry.GetLength() - nIndex - 2);
			else
				strIndexEntry = CString("  ") + strIndexEntry.Right(strIndexEntry.GetLength() - nIndex - 1);
			if (strHeader != m_strPrevHeader)	{
				m_astrFinal.Add(strHeader);
				m_strPrevHeader = strHeader;
			}
		}
		m_astrFinal.Add(strIndexEntry);
	}
}
// CIndexerView message handlers
void CIndexerView::OnPageno() 
{
	CRichEditCtrl& rEdit = GetRichEditCtrl();

	// --- get the current caret position
	long nStartChar, nEndChar;
	rEdit.GetSel(nStartChar, nEndChar );		// character index of current position

	// --- get the character index of the 1st character, current line
	int nLineIndex = rEdit.LineIndex(-1);	// character index of 1st char, current line

	int nLineNo = CurrentLineNumber();

	if (nStartChar != nLineIndex)	{
		// ---- caret is in the text or at the end of a line,
		//      insert page number after current line
		nLineIndex = rEdit.LineIndex(nLineNo + 1);
		if (nLineIndex == -1)	{
			// ---- caret is on the last line,
			//      append page number to file
			nLineIndex = rEdit.GetWindowTextLength();
			rEdit.SetSel(nLineIndex, nLineIndex);
			rEdit.ReplaceSel("\r\n", TRUE);
			nLineIndex += 2;
		}
	}
	// ----- get the next lowest page number
	int nPageNo = 1;

	for (int i = nLineNo; i >= 0; --i)	{
		EditLine line(rEdit, i);
		if (line.IsPageBreak())	{
			if (i != nLineNo || nStartChar != nLineIndex)	{
				nPageNo = line.PageNo() + 1;
				break;
			}
			// caret is in left margin of existing PAGE statement, continue
		}
	}
	InsertPageBreak(nLineIndex, nPageNo);
}
void CIndexerView::OnBuildindex() 
{
	CStringArray astrIndex;

	m_strFindText.Empty();
	m_astrFinal.RemoveAll();

	CRichEditCtrl& rEdit = GetRichEditCtrl();
	int nLines = rEdit.GetLineCount();
	int nPageNo = 0, nPrev = 0, i;

	Progress progress("Building:", nLines);

	CWaitCursor	wait;

	for (i = 0; i < nLines; i++)	{
		progress.StepIt();
		EditLine line(rEdit, i);
		if (line.IsPageBreak())    {
			nPageNo = line.PageNo();
			if (nPageNo <= nPrev)	{
				AfxMessageBox("Sequence error", MB_ICONSTOP);
				int nPos = rEdit.LineIndex(i);
				rEdit.SetSel(nPos, nPos);
				return;
			}
			nPrev = nPageNo;
			continue;
		}
		if (!line.IsEmpty())	{
			CString strEntry;
			strEntry.Format("%s %4d", line.Buffer(), nPageNo);
			astrIndex.Add(strEntry);

			char* cp = strchr(line.Buffer(), '\\');
			if (cp != 0 && *(cp + 1) == '\\')	{
				strEntry.Format("%s %4d", cp+2, nPageNo);
				astrIndex.Add(strEntry);
			}
		}
	}

	int nEntries = astrIndex.GetSize();
	int* pnIndexEntry = new int [nEntries];

	for (i = 0; i < nEntries; i++)
		*(pnIndexEntry+i) = i;
	pArray = &astrIndex;

	qsort(pnIndexEntry, nEntries, sizeof(int), Compare);

	CString strPrevPhrase;
	int nPrevPage = 0;

	CString strIndexEntry;
	m_strPrevHeader.Empty();

	for(i = 0; i < nEntries; i++)	{
		CString strEntry = astrIndex[pnIndexEntry[i]];

		CString strPhrase(strEntry.Left(strEntry.GetLength()-5));
		int nPageNo = atoi(strEntry.Right(4));

		if (StringCompare(strPhrase, strPrevPhrase) != 0)	{
			// --- new phrase, output the prior one
			BuildIndex(strIndexEntry);

			if (!strPrevPhrase.IsEmpty() && tolower(strPhrase[0]) != tolower(strPrevPhrase[0]))	{
				// ---- new letter of the alphabet, insert a blank line
				m_astrFinal.Add(CString());
				m_strPrevHeader.Empty();
			}

			// --- insert the phrase
			strIndexEntry = strPhrase;
			strPrevPhrase = strPhrase;
			nPrevPage = 0;
		}

		if (nPageNo != nPrevPage)	{
			// --- append the page number
			CString strPno;
			strPno.Format(", %d", nPageNo);
			strIndexEntry += strPno;
			nPrevPage = nPageNo;
		}
	}
	delete [] pnIndexEntry;

	// --- output the last one
	BuildIndex(strIndexEntry);

	theApp.BuildIndexDisplay(m_astrFinal);
}
// ---- called from the CIndexDlg class's OnSave function
void CIndexerView::SaveIndex()
{
	char filebuf[MAX_PATH+1] = "";
	OPENFILENAME ofn = {
		sizeof(OPENFILENAME),
		0,								// hWndOwner
		0,								// hInstance
		"Text File (*.txt)\0*.txt\0"
		"Any File (*.*)\0*.*\0",		// lpstrFilter
		0,								// lpstrCustomFilter
		0,								// nMaxCustFilter 
		0,								// nFilterIndex 
		filebuf,						// lpstrFile 
		MAX_PATH,						// nMaxFile 
		0,								// lpstrFileTitle
		0,								// nMaxFileTitle
		0,								// lpstrInitialDir 
		0,								// lpstrTitle
		OFN_HIDEREADONLY	|
		OFN_NOCHANGEDIR		|
		OFN_OVERWRITEPROMPT,			// Flags
		0,								// nFileOffset
		0,								// nFileExtension
		"txt",							// lpstrDefExt
		0,								// lCustData
		0,								// lpfnHook 
		0								// lpTemplateName
	};
	if (GetSaveFileName(&ofn))	{
		CStdioFile ndxFile (ofn.lpstrFile, CFile::modeCreate | CFile::modeWrite | CFile::typeText);
		int nEntries = m_astrFinal.GetSize();
		for (int i = 0; i < nEntries; i++)	{
			CString strText(m_astrFinal[i]);
			ndxFile.WriteString(strText + '\n');
		}

	}
}
void CIndexerView::SelectEntry(int sel)
{
	m_strFindText = m_astrFinal[sel];
	if (!m_strFindText.IsEmpty())	{
		int nIndex = m_strFindText.GetLength() - 1;
		while (nIndex > 0)	{
			char ch = m_strFindText[nIndex];
			if (!isdigit(ch) && ch != ',' && ch != ' ')
				break;
			--nIndex;
		}
		m_strFindText = m_strFindText.Left(nIndex + 1);
		while (!m_strFindText.IsEmpty() && m_strFindText[0] == ' ')
			m_strFindText = m_strFindText.Right(m_strFindText.GetLength()-1);
		GetRichEditCtrl().SetSel(0, 0);
		if (FindText(m_strFindText))
			SetFocus();
	}
}
void CIndexerView::OnFindNext() 
{
	CRichEditView::OnFindNext(m_strFindText, TRUE, TRUE, FALSE);
}
void CIndexerView::OnUpdateFindNext(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(!m_strFindText.IsEmpty());
}
BOOL CIndexerView::IsPageBreak(int nLineNo)
{
	char line;
	GetRichEditCtrl().GetLine(nLineNo, &line, 1);
	return line == '-';
}
BOOL CIndexerView::DeletePageBreak(int nLineNo)
{
	if (IsPageBreak(nLineNo))	{
		CRichEditCtrl& rEdit = GetRichEditCtrl();
		char line[101];
		memset(line, 0, sizeof line);
		rEdit.GetLine(nLineNo, line, 100);
		int nLineIndex = rEdit.LineIndex(nLineNo);
		rEdit.SetSel(nLineIndex, nLineIndex+strlen(line));
		rEdit.Clear();
		return TRUE;
	}
	return FALSE;
}
void CIndexerView::OnKeyUp(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	if (nChar == VK_CONTROL)
		m_bControlDown = FALSE;
	CRichEditView::OnKeyUp(nChar, nRepCnt, nFlags);
}
void CIndexerView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	int nLineNo = CurrentLineNumber();
	switch (nChar)	{ 
		case VK_CONTROL:
			m_bControlDown = TRUE;
			break;
		case VK_DELETE:
			if (DeletePageBreak(nLineNo))
				return;
			break;
		case VK_END:
		case VK_HOME:
			if (m_bControlDown == FALSE && IsPageBreak(nLineNo))
				return;
			break;
		case VK_RIGHT:
			if (IsPageBreak(nLineNo))	{
				SendMessage(WM_KEYDOWN, VK_DOWN, (LONG) nRepCnt | (LONG) nFlags << 16);
				return;
			}
			break;
		default:
			break;
	}
	CRichEditView::OnKeyDown(nChar, nRepCnt, nFlags);
	KeepCaretInLeftMarginOfPageBreak();
}
void CIndexerView::KeepCaretInLeftMarginOfPageBreak()
{
	int nLineNo = CurrentLineNumber();
	if (IsPageBreak(nLineNo))	{
		long nStartChar, nEndChar;
		CRichEditCtrl& rEdit = GetRichEditCtrl();
		rEdit.GetSel(nStartChar, nEndChar );
		if (nStartChar == nEndChar)	{
			int nLineIndex = rEdit.LineIndex(nLineNo);
			rEdit.SetSel(nLineIndex, nLineIndex);
		}
	}
}
void CIndexerView::OnChar(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	int nLineNo = CurrentLineNumber();
	if (nLineNo > 0 && nChar == VK_BACK && DeletePageBreak(nLineNo-1))
		return;
	CRichEditView::OnChar(nChar, nRepCnt, nFlags);
}
void CIndexerView::OnPaginate() 
{

	CRichEditCtrl& rEdit = GetRichEditCtrl();
	int nTopLine = rEdit.GetFirstVisibleLine();
	long nBeg, nEnd;
	rEdit.GetSel(nBeg, nEnd);
	int nLines = rEdit.GetLineCount();

	Progress progress("Paginating:", nLines);

	rEdit.HideCaret();
	rEdit.ShowWindow(SW_HIDE);

	int nPageNo = 0;

	CWaitCursor	wait;

	for (int i = 0; i < nLines; i++)	{
		EditLine buf(rEdit, i);
		if (buf.IsPageBreak())	{
			int nLineIndex = rEdit.LineIndex(i);
			rEdit.SetSel(nLineIndex, nLineIndex+buf.Length()+2);
			rEdit.Clear();
			InsertPageBreak(nLineIndex, ++nPageNo);
		}
		progress.StepIt();
	}

	rEdit.LineScroll(nTopLine);
	rEdit.SetModify();
	rEdit.SetSel(nBeg, nEnd);

	rEdit.ShowWindow(SW_SHOW);
	rEdit.ShowCaret();
}
void CIndexerView::InsertPageBreak(long nLineIndex, int nPageNo)
{
	CRichEditCtrl& rEdit = GetRichEditCtrl();
	CHARFORMAT cf = {sizeof(CHARFORMAT)};
	cf.dwMask = CFM_COLOR;
	cf.crTextColor = RGB(0,0,255);
	CString strLine;
	strLine.Format("-----------------------%d-----------------------\r\n", nPageNo);
	rEdit.ReplaceSel(strLine, TRUE);
	int nNewCaretPosition = nLineIndex+strLine.GetLength();
	rEdit.SetSel(nLineIndex, nNewCaretPosition-2);
	rEdit.SetSelectionCharFormat(cf);
	rEdit.SetSel(nNewCaretPosition, nNewCaretPosition);
}
void CIndexerView::Serialize(CArchive& ar) 
{
	CString strLine;
	CRichEditCtrl& rEdit = GetRichEditCtrl();
	CWaitCursor	wait;
	if (ar.IsStoring())	{
		int nLines = rEdit.GetLineCount();
		Progress progress("Saving:", nLines);
		char lncnt[] = "..0000";
		sprintf(lncnt+2, "%4d", nLines);
		ar.WriteString(lncnt);
		ar.WriteString("\r\n");

		for (int i = 0; i < nLines; i++)	{
			EditLine buf(rEdit, i);
			if (buf.IsPageBreak())	{
				char pgbrk[] = ".0000";
				sprintf(pgbrk+1, "%4d", buf.PageNo());
				ar.WriteString(pgbrk);
			}
			else
				ar.WriteString(buf.Buffer());
			ar.WriteString("\r\n");
			progress.StepIt();
		}
	}
	else	{
		rEdit.HideCaret();
		rEdit.ShowWindow(SW_HIDE);
		Progress progress("Loading:");
		while (ar.ReadString(strLine))	{
			long nLineIndex = rEdit.GetWindowTextLength();
			if (!strLine.IsEmpty() && strLine[0] == PAGECHAR)	{
				if (strLine.GetLength() > 2 && strLine[1] == PAGECHAR)
					progress.SetRange(0, atoi(strLine.GetBuffer(0)+2));
				else	{
					InsertPageBreak(nLineIndex, atoi(strLine.GetBuffer(0) + 1));
					nLineIndex = rEdit.GetWindowTextLength();
					rEdit.SetSel(nLineIndex, nLineIndex);
				}
			}
			else	{
				rEdit.SetSel(nLineIndex, nLineIndex);
				rEdit.ReplaceSel(strLine + "\r\n", TRUE);
			}
			progress.StepIt();
		}
		rEdit.SetSel(0,0);
		rEdit.ShowWindow(SW_SHOW);
		rEdit.ShowCaret();
	}
	SetFocus();
}
void CIndexerView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	CRichEditView::OnLButtonDown(nFlags, point);
	KeepCaretInLeftMarginOfPageBreak();
}

int CIndexerView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CRichEditView::OnCreate(lpCreateStruct) == -1)
		return -1;
	m_nWordWrap = WrapToTargetDevice;
	WrapChanged();
	DragAcceptFiles(FALSE);	
	return 0;
}
