#define _EXTARRAY_C

#include "extarray.h"

/* Author: John Boyer
   Date  : 27 May 1997

   Copyright (c) 1997 by John Boyer.  Every reasonable effort has been made to
   make this software bug free, but your use of this software is at your own
   risk.  You may use this software free of charge provided that this
   copyright message remains in the source code, and provided that you list
   "Resizable Array Implementation Copyright (c) 1997 by John Boyer" in
   the software or documentation credits.  If you change the code in any
   way, you must include a detailed comment between the two lines appearing
   directly below this copyright message.  Email regarding use or modification
   of this software would be gratefully accepted by jboyer@uwi.com.
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------

   This module provides a general purpose extensible array data structure
   written in C.  The array is expanded and contracted as necessary to ensure
   the following:

   	1) The array is within a constant factor larger than its contents
        2) The cost of resizing never exceeds constant amortized time per
        	add/delete operation.

        The caller can create an extensible array by calling EANew(), giving
   both the minimum size of the array and the start size (which is often larger).
   To add an element, call EAAddElement(), providing a key name and a void
   pointer to a data structure of your choosing.  To look up a datum, call
   EAFindElement(), giving the key string; the void pointer is returned.
   Call EADeleteElement() with the key string or user data to remove an item
   from the array.  Call EADestroy() to free the array.

   The array is kept in unsorted order, and sequential searching is used.  New
   information items are added to the end of the array.  When an item is
   deleted, its space is filled by copying the item at the end of the array.
*/

/* Prototypes for private functions */

int _EAStoreElement(extArrayP theArray, charP theKey, void *userData);

int _EAGrow(extArrayP theArray);

int _EAShrink(extArrayP theArray);

int _EAReplaceArray(extArrayP theArray, ulong NewMaxSize);

/*****************************************************************************
  EANew - allocates a new extensible array

 @param minSize NI - what is the smallest permissible size of the array
 @param startSize NI - tells how big do we want the array to start

 @return a pointer to the extensible array structure

 @access public
 *****************************************************************************/

extArrayP EANew(ulong minSize, ulong startSize)
{
extArrayP theArray;
ulong I;

     theArray = (extArrayP) cp_malloc(sizeof(struct _extArray));
     if (theArray == NULL)
     {
         ReportError("EANEW: Out of memory");
         return NULL;
     }

     if (startSize < minSize) startSize = minSize;

     theArray->size = 0;
     theArray->minSize = minSize;
     theArray->maxSize = startSize;
     theArray->theList = NULL;

     if (startSize > 0)
     {
         theArray->theList = (arrayNodeP) cp_malloc(startSize*sizeof(struct _arrayNode));
         if (theArray->theList == NULL)
         {
             ReportError("EANEW: Out of memory");
             cp_free((charP) theArray);
             return NULL;
         }

         for (I = 0; I < startSize; I++)
         {
              theArray->theList[I].theKey = NULL;
              theArray->theList[I].userData = NULL;
         }
     }

     return theArray;
}

/*****************************************************************************
 EADestroy

 @param pArray LIO - pointer to the pointer containing the address allocated by EANew
 			for a new extensible array.

 @return nothing

 @access public

 Call using EADestroy(&MyArray); where a previous call to EANew would've been
 extArrayP MyArray = EANew(16, 256);
 This function will set your pointer to NULL after it frees the internal list
 of keys, the memory block then the extensible array structure itself.

 *****************************************************************************/

void EADestroy(extArrayP *pArray)
{
extArrayP theArray;
ulong I;

     if (pArray == NULL || (theArray = *pArray) == NULL)
         return;

     if (theArray->theList != NULL)
     {
         for (I = 0; I < theArray->size; I++)
              cp_free(theArray->theList[I].theKey);

         cp_free((charP) theArray->theList);
     }

     cp_free((charP) theArray);
     *pArray = NULL;
}

/*****************************************************************************
 EAAddElement

 @param theArray LI - identifies the array to add to
 @param theKey UI - gives the search key value of the object being added
 @param userData UI - an application-specific pointer to associate with the key;
 			provides 'satellite' data for the key.

 @return OK on success; NOTOK on error
 @access public
 *****************************************************************************/

int EAAddElement(extArrayP theArray, charP theKey, void *userData)
{
charP DupKey = NULL;

     if (theArray==NULL || theKey==NULL || userData==NULL)
     {
         ReportError("EAADDELEMENT: Invalid Parameters");
         return NOTOK;
     }

/* We need a duplicate of the key to store in the array */

     if ((DupKey = cp_strdup(theKey)) == NULL)
     {
         ReportError("EAADDELEMENT: Out of memory");
         return NOTOK;
     }

/* We make a call to expand the array if necessary.  The function rarely
	does more than return OK (because we usu. we don't expand) */

     if (_EAGrow(theArray) != OK)
     {
         cp_free(DupKey);
         ReportError("EAADDELEMENT: Out of memory");
         return NOTOK;
     }

/* Store the key and the user data pointer at the end of the array */

     return _EAStoreElement(theArray, DupKey, userData);
}

/*****************************************************************************
 _EAStoreElement

 @access private

 Encapsulates the act of putting the key and user data into an array element.
 *****************************************************************************/

int _EAStoreElement(extArrayP theArray, charP theKey, void *userData)
{
     theArray->theList[theArray->size].theKey = theKey;
     theArray->theList[theArray->size].userData = userData;
     theArray->size++;
     return OK;
}

/*****************************************************************************
 EADeleteElement

 @param theArray LI - identifies the array to delete from
 @param theKey UI - identifies the object to delete

 @return OK on success, NOTOK if the object couldn't be found or if there
         was an error.

 @access public

 Searches for an element which you must identify by key.  The item is removed
 from the list if found. (It is still the caller's responsibility to free
 whatever the userData was pointing at).
 *****************************************************************************/

int  EADeleteElement(extArrayP theArray, charP theKey)
{
ulong I, Pos;
int Found;

     if (theArray == NULL || theKey == NULL)
     {
         ReportError("EADELETEELEMENT: Invalid Parameters");
         return NOTOK;
     }

/* Find the item using its key value */

     for (I = 0, Found = 0; I < theArray->size; I++)
          if (cp_strcmp(theKey, theArray->theList[I].theKey) == OK)
          {
              Pos = I;
              Found = 1;
              break;
          }

/* If we didn't find the item, then there is nothing to delete */

     if (!Found)
     {
         ReportError("EADELETEELEMENT: Item not found");
         return NOTOK;
     }

/* Remove the item from the array */

     cp_free(theArray->theList[Pos].theKey);
     theArray->size--;
     theArray->theList[Pos].theKey = theArray->theList[theArray->size].theKey;
     theArray->theList[Pos].userData = theArray->theList[theArray->size].userData;

     theArray->theList[theArray->size].theKey = NULL;
     theArray->theList[theArray->size].userData = NULL;

/* Shrink the array if necessary */

     return _EAShrink(theArray);
}

/*****************************************************************************
 EAFindElement

 @param theArray LI - identifies the array to search
 @param theKey UI - identifies the object to be found

 @return the userData associated with theKey, or NULL if the object wasn't found

 @access public

 Searches for theKey within theArray.  If found, returns the associated
 userData pointer.  If theKey is not found, then NULL is returned.
 *****************************************************************************/

void *EAFindElement(extArrayP theArray, charP theKey)
{
int I;

     if (theArray == NULL || theKey == NULL)
     {
         ReportError("EAFINDELEMENT: Invalid Parameters");
         return NULL;
     }

     for (I = 0; I < theArray->size; I++)
          if (cp_strcmp(theKey, theArray->theList[I].theKey) == OK)
              return theArray->theList[I].userData;

     return NULL;
}

/*****************************************************************************
 _EAGrow

 @access private

 When the array fills up, we simply create a double sized array, then copy
 the data items into the new array.  It almost seems too simple, but the fact
 that we grow the array so infrequently means that we can afford to rebuild it
 in a straightforward fashion without losing the speed of amortized constant
 time per operation.  Very infrequently, we have an O(n) rebuild, but most of
 the time we have O(1) behavior because we don't grow the array.
 Excluding delete operations, we can average the O(n) time per expansion over
 all elements, yielding 2 units of cost per element.
 *****************************************************************************/

int  _EAGrow(extArrayP theArray)
{
ulong NewMaxSize;

/* We only want to expand when Size has reached MaxSize */

     if (theArray->size < theArray->maxSize) return OK;

     NewMaxSize = theArray->maxSize ? (theArray->maxSize<<1) : 1;
     if (NewMaxSize < theArray->minSize)
         NewMaxSize = theArray->minSize;

/* Move to a new array */

     return _EAReplaceArray(theArray, NewMaxSize);
}

/*****************************************************************************
 _EAShrink

 @access private

 When the array is down to or below 25% full, we want to cut its size in half.
 Normally we'd cut it in half when it was half full, but an alternating series
 of add and delete operations at just the right moment can cause us to shrink
 and grow on each operation.  By using the one quarter mark, we ensure that
 n/4 or operations must separate each grow and shrink operation.

 Once we allow delete operations, the amortized cost per operation is 3 time
 units, which is still constant.  Also, the array can never be larger than
 four times the number of nodes currently in it.
 *****************************************************************************/

int  _EAShrink(extArrayP theArray)
{
ulong NewMaxSize;

/* Don't want to shrink if Size is greater than one quarter of MaxSize */

     if (theArray->size > (theArray->maxSize>>2) || theArray->size == 0)
         return OK;

     NewMaxSize = theArray->maxSize>>1;

/* Don't want to shrink smaller than the given minimum size */

     if (NewMaxSize < theArray->minSize) return OK;

// Move to a new array

     return _EAReplaceArray(theArray, NewMaxSize);
}

/*****************************************************************************
 _EAReplaceArray

 @access private

 Makes a copy of the array into a new memory block that is NewMaxSize units
 in size.  Then, we swap that new memory block into theArray, and we delete
 the 'old' memory block that was in theArray at the start of the function.
 *****************************************************************************/

int _EAReplaceArray(extArrayP theArray, ulong NewMaxSize)
{
arrayNodeP Swapper;
extArrayP theNewArray;
int I;

    theNewArray = EANew(theArray->minSize, NewMaxSize);
    if (theNewArray == NULL)
    {
        ReportError("_EAREPLACEARRAY: Out of memory");
        return NOTOK;
    }

/* We need to copy the keys and user data into the new array; we NULL out
   the pointers in the 'old' array so that ownership is transferred. */

    for (I = 0; I < theArray->size; I++)
    {
         _EAStoreElement(theNewArray, theArray->theList[I].theKey,
                                      theArray->theList[I].userData);
         theArray->theList[I].theKey = NULL;
         theArray->theList[I].userData = NULL;
    }

/* Now we want to swap the new array into the existing array, and vice
   versa, so we can destroy the 'old' array block */

    Swapper = theNewArray->theList;
    theNewArray->maxSize = theArray->maxSize;
    theNewArray->size = theArray->size;
    theNewArray->theList = theArray->theList;
    theArray->theList = Swapper;
    theArray->maxSize = NewMaxSize;

    EADestroy(&theNewArray);

    return OK;
}

