#define _HASHTABL_C

#include "hashtabl.h"

/* Author: John Boyer
   Date  : 27 May 1997

   Copyright (c) 1997 by John Boyer.  Every reasonable effort has been made to
   make this software bug free, but your use of this software is at your own
   risk.  You may use this software free of charge provided that this
   copyright message remains in the source code, and provided that you list
   "Resizable Hash Table Implementation Copyright (c) 1997 by John Boyer" in
   the software or documentation credits.  If you change the code in any
   way, you must include a detailed comment between the two lines appearing
   directly below this copyright message.  Email regarding use or modification
   of this software would be gratefully accepted by jboyer@uwi.com.
-------------------------------------------------------------------------------
-------------------------------------------------------------------------------

   This module provides a general purpose resizable hash table data structure
   written in C.  The table is expanded and contracted as necessary to ensure
   the following:

   	1) The table is within a constant factor larger than its contents
        2) The cost of resizing never exceeds constant amortized time per
        	add/delete operation.

        The caller can create a hash table by calling HTNew(), giving both the
   minimum size of the hash table and the start size (which is often larger).
   To add an element, call HTAddElement(), providing a key name and a void
   pointer to a data structure of your choosing.  To look up a datum, call
   HTFindElement(), giving the key string; the void pointer is returned.
   Call HTDeleteElement() with the key string to remove an item from the table.
   Call HTDestroy() to free the table.

   The table is an array of extensible arrays.  Each extensible array keeps a
   simple unsorted list of all objects that collide at a particular hash value.
*/

/* Simply redefine this macro if you want to use a different hash function */

#define HASHFUNCTION	HashJB

/* This is my own hash function. Notice that we mix the upper bits into the
   lower bits to reduce the information loss resulting from the truncation of
   the hash value (in the return line). */

typedef unsigned long ulong;
typedef unsigned char uchar;

ulong HashJB(uchar *KeyStr, ulong N)
{
ulong Hash=0, I=0;
uchar Ch;

	for (;Ch=*KeyStr++; I^=1)
		if (I)
			Hash *= Ch;
		else	Hash += Ch;

	Hash += ((Hash&0xFFFF0000)>>16);
	Hash += ((Hash&0x0000FF00)>>8);

	return Hash & (N-1);
}

/*****************************************************************************
 *****************************************************************************/

/* Prototypes for private functions */

int _HTStoreElement(hashTableP theTable, charP theKey, void *userData);
int _HTGrow(hashTableP theTable);
int _HTShrink(hashTableP theTable);
int _HTReplaceTable(hashTableP theTable, ulong NewMaxSize);

/*****************************************************************************
  HTNew - allocates a new hash table

  @param minSize NI - what is the smallest permissible size of the table
  @param startSize NI - tells how big do we want the table to start

  @return a pointer to the hash table structure

  @access public

  NOTE:  You will not get maximum usage from your table unless the two
  	  parameters you give are powers of 2.  This is because the
          HASH_FUNCTION is using the much faster bitwise-and by N-1
          to replace modulus N.
 *****************************************************************************/

hashTableP HTNew(ulong minSize, ulong startSize)
{
hashTableP theTable;
ulong I;

     theTable = (hashTableP) cp_malloc(sizeof(struct _hashTable));
     if (theTable == NULL)
     {
         ReportError("HTNEW: Out of memory");
         return NULL;
     }

     if (minSize < 2) minSize = 2;
     if (startSize < minSize) startSize = minSize;

     theTable->size = 0;
     theTable->minSize = minSize;
     theTable->maxSize = startSize;
     theTable->theList = NULL;

     if (startSize > 0)
     {
         theTable->theList = (extArrayP) cp_malloc(startSize*sizeof(struct _extArray));
         if (theTable->theList == NULL)
         {
             ReportError("HTNEW: Out of memory");
             cp_free((charP) theTable);
             return NULL;
         }

         for (I = 0; I < startSize; I++)
              theTable->theList[I] = NULL;
     }

     return theTable;
}

/*****************************************************************************
 HTDestroy

 @param pTable LIO - pointer to the pointer containing the address allocated by HTNew
 		     for a new hash table.

 @return nothing

 @access public

 Call using HTDestroy(&MyTable); where a previous call to HTNew would've been
 hashTableP MyTable = HTNew(16, 256);
 This function will set your pointer to NULL after it frees the internal list
 of keys, the memory block then the hash table structure itself.
 *****************************************************************************/

void HTDestroy(hashTableP *pTable)
{
hashTableP theTable;
ulong I;

     if (pTable == NULL || (theTable = *pTable) == NULL)
         return;

     if (theTable->theList != NULL)
     {
         for (I = 0; I < theTable->maxSize; I++)
              if (theTable->theList[I] != NULL)
                  EADestroy(&theTable->theList[I]);

         cp_free((charP) theTable->theList);
     }

     cp_free((charP) theTable);
     *pTable = NULL;
}

/*****************************************************************************
 HTAddElement

 @param theTable LI - identifies the table to add to
 @param theKey UI - gives the search key value of the object being added
 @param userData UI - an application-specific pointer to associate with the key;
 		   provides 'satellite' data for the key.

 @return OK on success; NOTOK on error
 @access public
 *****************************************************************************/

int  HTAddElement(hashTableP theTable, charP theKey, void *userData)
{
     if (theTable==NULL || theKey==NULL || userData==NULL)
     {
         ReportError("HTADDELEMENT: Invalid parameters");
         return NOTOK;
     }

/* We make a call to expand the table if necessary.  The function rarely
	does more than return OK (because we usu. we don't expand). */

     if (_HTGrow(theTable) != OK)
     {
         ReportError("HTADDELEMENT: Failed to grow table");
         return NOTOK;
     }

/* Store the key and the user data pointer in the table */

     return _HTStoreElement(theTable, theKey, userData);
}

/*****************************************************************************
 _HTStoreElement

 @access private
 
 Encapsulates the act of putting the key and user data into the table.
 *****************************************************************************/

int _HTStoreElement(hashTableP theTable, charP theKey, void *userData)
{
ulong thePos = HASHFUNCTION(theKey, theTable->maxSize);

     if (theTable->theList[thePos] == NULL)
	 if ((theTable->theList[thePos] = EANew(0, 0)) == NULL)
         {
             ReportError("_HTSTOREELEMENT: Failed to create new collision list");
             return NOTOK;
         }

     theTable->size++;

     return EAAddElement(theTable->theList[thePos], theKey, userData);
}

/*****************************************************************************
 HTDeleteElement

 @param theTable LI - identifies the table to delete from
 @param theKey UI - identifies the object to delete

 @return OK on success, NOTOK if the object couldn't be found or if there
        was an error.

 @access public

 Searches for an element which you must identify by key.  The item is removed
 from the table if found. (It is still the caller's responsibility to free
 whatever the userData was pointing at).
 *****************************************************************************/

int  HTDeleteElement(hashTableP theTable, charP theKey)
{
ulong thePos = HASHFUNCTION(theKey, theTable->maxSize);
int Found;

     if (theTable == NULL || theKey == NULL)
     {
         ReportError("HTDELETEELEMENT: Invalid parameters");
         return NOTOK;
     }

/* Find the item using its key value */

     Found = 0;
     if (theTable->theList[thePos] != NULL)
     {
         if (EAFindElement(theTable->theList[thePos], theKey) != NULL)
             Found = 1;
     }

/* If we didn't find the item, then there is nothing to delete */

     if (!Found)
     {
         ReportError("HTDELETEELEMENT: Item not found");
         return NOTOK;
     }

/* Remove the item from the table */

     if (EADeleteElement(theTable->theList[thePos], theKey) != OK)
     {
         ReportError("HTDELETEELEMENT: Delete from collision list failed");
         return NOTOK;
     }

/* Shrink the table if necessary */

     return _HTShrink(theTable);
}

/*****************************************************************************
 HTFindElement

 @param theTable LI - identifies the table to search
 @param theKey UI - identifies the object to be found

 @return the userData associated with theKey, or NULL if the object wasn't found

 @access public

 Searches for theKey within theTable.  If found, returns the associated
 userData pointer.  If theKey is not found, then NULL is returned.
 *****************************************************************************/

void *HTFindElement(hashTableP theTable, charP theKey)
{
ulong thePos;

     if (theTable == NULL || theKey == NULL)
     {
         ReportError("HTFINDELEMENT: Invalid Parameters");
         return NULL;
     }

     thePos = HASHFUNCTION(theKey, theTable->maxSize);
     if (theTable->theList[thePos] == NULL)
         return NULL;

     return EAFindElement(theTable->theList[thePos], theKey);
}

/*****************************************************************************
 _HTGrow

 @access private

 When the table fills up, we simply create a double sized table, then copy
 the data items into the new table.  It almost seems too simple, but the fact
 that we grow the table so infrequently means that we can afford to rebuild it
 in a straightforward fashion without losing the speed of amortized constant
 time per operation.  Very infrequently, we have an O(n) rebuild, but most of
 the time we have O(1) behavior because we don't grow the table.
 Excluding delete operations, we can average the O(n) time per expansion over
 all elements, yielding 2 units of cost per element.
 *****************************************************************************/

int _HTGrow(hashTableP theTable)
{
ulong NewMaxSize;

/* We only want to expand when Size has reached MaxSize */

     if (theTable->size < theTable->maxSize) return OK;
     NewMaxSize = theTable->maxSize ? (theTable->maxSize<<1) : 2;

     if (NewMaxSize < theTable->minSize)
         NewMaxSize = theTable->minSize;

/* Move to a new table */

     return _HTReplaceTable(theTable, NewMaxSize);
}

/*****************************************************************************
 _HTShrink

 @access private

 When the table is down to or below 25% full, we want to cut its size in half.
 Normally we'd cut it in half when it was half full, but an alternating series
 of add and delete operations at just the right moment can cause us to shrink
 and grow on each operation.  By using the one quarter mark, we ensure that
 n/4 or operations must separate each grow and shrink operation.

 Once we allow delete operations, the amortized cost per operation is 3 time
 units, which is still constant.  Also, the table can never be larger than
 four times the number of nodes currently in it.
 *****************************************************************************/

int _HTShrink(hashTableP theTable)
{
ulong NewMaxSize;

/* Don't want to shrink if Size is greater than one quarter of MaxSize */

     if (theTable->size > (theTable->maxSize>>2) || theTable->size == 0)
         return OK;
     NewMaxSize = theTable->maxSize>>1;

/* Don't want to shrink smaller than the given minimum size */

     if (NewMaxSize < theTable->minSize) return OK;

// Move to a new table

     return _HTReplaceTable(theTable, NewMaxSize);
}

/*****************************************************************************
 _HTReplaceTable

 @access private

 Makes a copy of the table into a new memory block that is NewMaxSize units
 in size.  Then, we swap that new memory block into theTable, and we delete
 the 'old' memory block that was in theTable at the start of the function.
 *****************************************************************************/

int _HTReplaceTable(hashTableP theTable, ulong NewMaxSize)
{
extArrayP *Swapper;
hashTableP theNewTable;
int I, J;

    theNewTable = HTNew(theTable->minSize, NewMaxSize);
    if (theNewTable == NULL)
    {
        ReportError("_HTREPLACETABLE: Out of memory");
        return NOTOK;
    }

/* We need to copy the keys and user data into the new array */

    for (I = 0; I < theTable->maxSize; I++)
         if (theTable->theList[I] != NULL)
         {
             for (J = 0; J < theTable->theList[I]->size; J++)
             {
                  _HTStoreElement(theNewTable,
                                  theTable->theList[I]->theList[J].theKey,
                                  theTable->theList[I]->theList[J].userData);
             }
         }

/* Now we want to swap the new array into the existing array, and vice
   versa, so we can destroy the 'old' array block */

    Swapper = theNewTable->theList;
    theNewTable->size = theTable->size;
    theNewTable->maxSize = theTable->maxSize;
    theNewTable->theList = theTable->theList;
    theTable->theList = Swapper;
    theTable->maxSize = NewMaxSize;

    HTDestroy(&theNewTable);

    return OK;
}

