/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: BoxSet.java,v 1.32 1997/05/29 07:06:54 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import java.io.DataInputStream;
import java.io.IOException;
import java.util.*;

/**
 * The BoxSet maintains the collection of mailboxes, and routes
 * messages to them.  It is a thread-safe class.  <STRONG><EM>Method-
 * naming is pretty poor in this class.</EM></STRONG>
 *
 * @author Luke Weisman
 * @author Wesley Tanaka
 * @author Adam Rifkin
 * @author Daniel M. Zimmerman
 * @version 1.0b2 $Date: 1997/05/29 07:06:54 $
 * @see AckMessage
 * @see DataInputStream
 * @see InMailQueue
 * @see Inbox
 * @see MailDaemonException
 * @see Message
 * @see MultiURLClassLoader
 * @see Outbox
 * @see Place
 * @see SplitMessage
 * @see UnknownMessageException
 * @see java.io.DataInputStream
 * @see java.io.IOException
 * @see java.lang.ClassNotFoundException
 * @see java.lang.IllegalAccessException
 * @see java.lang.InstantiationException
 * @see java.util.Enumeration
 * @see java.util.Hashtable
 **/

class BoxSet extends Hashtable
{

  /**
   * When a message arrives and is accepted, this value is set to point
   * to the lucky mailbox just before waking any sleeping threads.  This
   * allows the first thread to get a lock on the instance to return the
   * mailbox with the newest message, if it so desires.  <STRONG><EM>I'm
   * a little unclear on this.  Details please.</EM></STRONG>
   */
  Inbox toWho;

  // Number of toWho waiters.  Luke's semaphore.
  int numDudes = 0;

  /**
   * This counter ticks with each new message, which allows for a
   * complete ordering of all messages which reached the system across
   * all mailboxes.
   */
  private int messageCount;

  /**
   * This is the classloader for the current mailDaemon.
   */
  private MultiURLClassLoader mucl = null;

  // constructors

  /**
   * Make a new portset.
   */

  BoxSet( MultiURLClassLoader mucl )
  {
    super();
    this.mucl = mucl;

    //Keeps absolute ordering of all messages received.
    messageCount = 0;

    toWho = null;
    numDudes = 0;
  }

  /**
   * Grab the InMailQueue associated with a given mailbox name.
   * @param name The name of the mailbox desired,
   */
  synchronized InMailQueue get( String name )
  {
    return (InMailQueue)super.get( name );
  }

  /**
   * Put an InMailQueue in the set, replacing an old one by the
   * same name, if there is one, so be careful.  <STRONG><EM>When is
   * this used and why?</EM></STRONG>
   */
  synchronized void put( String name, InMailQueue the_Q )
  {
    super.put( name, the_Q );
  }

  /**
   * A mailbox ends up calling this method to get put in the table
   * so it can receive messages,
   * @param p The new mailbox to add to the queue.
   * @exception MailDaemonException This is called if there is already
   * a mailbox with the p's name in the queue.
   */
  synchronized void register( Mailbox p )
    throws MailDaemonException
  {
    if ( get( p.name() ) != null )
      {
        //raise error of name conflict.
        throw new MailDaemonException("Conflicting mailbox names (" +
                                      p.name() + ") in MailDaemon register.");
      }
    else
      {
        //put in the portset.
        InMailQueue thep = new InMailQueue( p );
        put( p.name(), thep );
      }
  }


  synchronized void kill( Mailbox p )
  {
    InMailQueue imq = (InMailQueue)get( p.name() );
    imq.killMailbox( );

    remove( p.name() );
  }

  /**
   * Get a mailbox by name.
   *
   * @param name The name of the box.
   * @return the box, or null if it is not found.
   */
  synchronized Mailbox Grab( String n )
  {
    InMailQueue imq = (InMailQueue)get( n );
    if ( imq != null )
      return imq.Mailbox();
    else
      return null;
  }

  /**
   * Deal with a new message by routing it to the proper mailbox.  If
   * the message is accepted, wake up any sleeping threads via the
   * notifyAll() method.
   * @return The ack to send back, or null if there is no mailbox by
   *         the given name.
   */
  synchronized AckMessage insertMessage( SplitMessage m, Place from )
  {
    //Get the relevant InMailQueue.
    int ackID = -1;
    boolean acc = false;
    int bad = 0;

    Debug.lpln( 5, "Boxset looking for >" +
                m.PName() + "< " + m.PName().length() );
    InMailQueue iq = get( m.PName() );

    if ( iq != null )
      {
        //this will check to see if the message is from a newer
        //maildaemon, and if it is, reset everything in the queue.
        if ( iq.CheckOut( from, m.mydID() ) )
          {
            if ( !iq.WantMsg( m.ID(), from ) )
              {
                Debug.lpln( 13, "Sending re-ack of id " +
                            iq.AckMsgID( from ) + " to " + from );

                return new AckMessage( iq.AckMsgID( from ) );
              }
            else
              {
                // process the message
                // (could throw MailDaemonException of some sort)
                try
                  {
                    Message tempMessage = readObject (m.Message());
                    acc = iq.Receive( tempMessage, messageCount++ , from );
                  }
                catch ( MailDaemonException e )
                  {
                    if ( e instanceof UnknownMessageException )
                      bad = AckMessage.CORRUPT;
                    else
                      bad = AckMessage.ACCESS;
                    Debug.lpln( 9, "Caught bad message: " + e );
                    //since message was not received, fix the ids of the box
                    //via the Advance call (which does all but actually
                    //receive the message.
                    iq.Advance( from );
                  }

                //Wake up someone who is waiting for a message, if there
                //is one, but only if the message was accepted by a receive
                //mailbox.
                if ( acc && (iq.Mailbox() instanceof Inbox) )
                  {
                    Debug.lpln( 7, "Going to notify any sleepers." );

                    if (numDudes > 0)
                      //save the mailbox chosen.
                      toWho = (Inbox)iq.Mailbox();

                    notifyAll();
                  }

                //send the ack for the message
                ackID = iq.AckMsgID( from );
                Debug.lpln( 5, "Sending ack of id = " + ackID );

              }
          } //end checkout
        else
          //do nothing.  We got a message from a known dead maildaemon.
          bad = AckMessage.NEW_RECEIVER;

      } //if iq != null
    else
      //no mailbox
      bad = AckMessage.NO_MAILBOX;

    return new AckMessage( ackID, bad );

  } //end insertMessage

  /**
   * The object reconstructor.  This takes a datainput stream with the
   * data from a writeObject call, strips off the header information
   * which stores what kind of Message is held in the rest of the
   * stream, and then rebuilds the object.<P>
   * Note: This method uses Java's Class object to make an object of the
   * correct type.
   * @return The new object built from the stream.
   * @param  dis The datainputstream with which to build the message.
   */
  public final Message readObject( DataInputStream dis )
    throws MailDaemonException
  {
    /* Right now, the protocol I'll use will allow multiple object types,
       as long as they are derived from Message.  The byte stream will be
       a series of bytes of the form classname/data.  I'll construct an
       object of type classname, then pass it the rest of the data for
       it's own filling in of info.
       */
    String classname = null, url = null;
    Message newMessage = null;
    Class c = null;

    try
      {
        classname = dis.readUTF();
        Debug.lpln( 8, "class = >" + classname + "<" );

        url = dis.readUTF();
        Debug.lpln( 8, "url = >" + url + "<" );

        if ((mucl != null) && (url != null) && (!url.equals ("")))
          mucl.addURL( url );

        // requires an empty constructor in sub message class
        // another option is to have a constructor which takes s,i+1

        //get the class (or load it if necessary.)
        Debug.lpln( 8, "Loading class " + classname );
        if (mucl != null)
          {
            Debug.lpln (8, "mucl was not null");
            c = mucl.loadClass( classname, true );
          }
        else
          {
            Debug.lpln (8, "mucl was null");
            c = Class.forName(classname);
          }

        // and make new instance.
        Debug.lpln( 8, "Making new instance." );
        newMessage = (Message)( c ).newInstance();

        newMessage.setBoxSet (this);

        Debug.lpln( 8, "Calling readData method." );
        newMessage.readData( dis );

        Debug.lpln( 8, "Returning the product: " + newMessage );
        return newMessage;
      }
    catch (IOException e)     {
      if ( Debug.checkLevel( 11 ) )
        e.printStackTrace();
      throw new UnknownMessageException( classname + "(IO:" + e + ")" );
    }
    catch (ClassNotFoundException e)  {
      if ( Debug.checkLevel( 11 ) )
        e.printStackTrace();
      throw new UnknownMessageException( classname + "(No Class)" );
    }
    catch (InstantiationException e)  {
      if ( Debug.checkLevel( 11 ) )
        e.printStackTrace();
      throw new UnknownMessageException( classname +"(No Instantiation)");
    }
    catch (IllegalAccessException e)  {
      if ( Debug.checkLevel( 11 ) )
        e.printStackTrace();
      throw new MailDaemonException( "Message "+ classname +
                                     " gave illegal access (" +
                                     e + ")" );
    }
  }


  /**
   * Waits for a new message to arrive, and then returns the receive
   * mailbox to which that message was given. <P>
   *
   * When it returns from the wait command it checks to see if the
   * toWho variable has changed.  If it has not, it returns to sleep.
   * Upon exit it returns toWho to null.  This is so if some other
   * thread coopted the message this thread will try again.<P>
   *
   * If both time_s and time_ms are 0, will wait forever.
   *
   * @param time_s The time to wait in s.
   * @param time_ms The time to wait in ms.
   * @return The mailbox with the oldest message known about, or null
   * if there is no such mailbox.
   */
  synchronized Inbox waitForNewMessage( long time_s, int time_ms )
    throws InterruptedException
  {
    Debug.lpln( 7, "waitForNewMessage: Tinkering with suspended thread." );

    numDudes++;

    //wait until we get notified or time expires.
    wait( time_s, time_ms );

    Debug.lpln( 7, "Done with suspend.  Continuing on." );

    Inbox rp = toWho;
    numDudes--;

    if (numDudes == 0)
      //reset this to null.
      toWho = null;

    if ( ( rp == null ) && ( time_s == 0 ) && ( time_ms == 0 ) )
      //continue wait
      return waitForNewMessage( 0, 0 );
    else
      return rp;
  }

  /**
   * Check for any old message, and return the owning mailbox if there is
   * one.  Otherwise wait for a new message, and return its owning mailbox.
   * <P>
   * It skips all the Outboxs and their ilk.
   * @return A reference to the mailbox with the oldest message.
   */
  synchronized Inbox waitForMessage( int time_s, int time_ms )
    throws InterruptedException
  {
    Inbox rp = null, ttrp = null;
    Mailbox trp = null;

    //first check to see if the any mailbox has a message already.
    Enumeration e = elements();
    while ( e.hasMoreElements() )
      {
        trp = ((InMailQueue)e.nextElement()).Mailbox();
        if ( (trp != null)
             && (trp instanceof Inbox)
             && !((Inbox)trp).empty() )
          {
            ttrp = (Inbox)trp;
            if ( (rp == null) || ( rp.time() > ttrp.time() ) )
              //set rp to the rec. mailbox with oldest message.
              rp = ttrp;
          }
      }

    if ( rp == null )
      //No message, so wait for new message.
      return waitForNewMessage( time_s, time_ms );
    else
      return rp;
  }

  /**
   * Some debugging stuff.  Not sure it works, or anything
   * like that.
   */
  synchronized void mailboxDump()
  {
    System.out.println( "The keys: " );
    Enumeration k = keys();

    while ( k.hasMoreElements() )
      System.out.println( "   k>" + k.nextElement() + "<" );
  }

  String boxList( )
  {
    Enumeration k = keys();
    String s = "";
    while( k.hasMoreElements() )
      s += k.nextElement() + " ";
    return s.trim();
  }

} // end of BoxSet class
