/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
  $Id: Filter.java,v 1.14 1997/05/29 07:06:58 kiniry Exp $
*/

package info.net;

import java.net.InetAddress;
import java.net.UnknownHostException;
import info.util.Debug;
import java.util.Enumeration;
import java.util.Vector;

/**
 * The Filter class is used by inboxes to check messages
 * before accepting them into the queue.  Filters can be used
 * for many things; for example, you can ensure there are only
 * NumberMessages in the queue.  Another example is that you
 * can make the queue only accept messages from a specific Place.
 *
 * <P> To install a Filter in an inbox, use its setFilter method,
 * and pass to it the Filter you want to use.
 *
 * <P> The CheckMessage method does the actual filtering. 
 * It takes a message and the place that sent it, decides whether to
 * accept or deny it, and returns true or false depending on that
 * decision. 
 *
 * <P> CheckMessage is the method to override to change
 * the default behavior of Filter.  By default, CheckMessage calls
 * SendDenial to send a DenyMessage to the sending mailbox of any
 * message does not pass the Filter.
 *
 * @author Adam Rifkin
 * @author Wesley Tanaka
 * @author Daniel M. Zimmerman
 * @version 1.0b2 $Date: 1997/05/29 07:06:58 $
 * @see Inbox#setFilter
 */

public class Filter 
{
  protected Inbox owner;   
  protected Vector allowedClasses;
  protected Vector deniedClasses;
  protected Vector allowedHosts;
  protected Vector deniedHosts;
  protected Vector allowedPlaces;
  protected Vector deniedPlaces;

  // methods

  /**
   * Constructs a new, unbound Filter.
   */
  public Filter()
  {
    allowedClasses = new Vector();
    deniedClasses = new Vector();
    allowedHosts = new Vector();
    deniedHosts = new Vector();
    allowedPlaces = new Vector();
    deniedPlaces = new Vector();
    owner = null;
  }

  /**
   * Constructs a new Filter bound to a specific Inbox.
   * @param inbox The Inbox to which the Filter will be bound.
   */
  public Filter(Inbox inbox)
  {
    this();
    owner = inbox;
  }

  /**
   * Binds the filter to an Inbox.
   * @param inbox The Inbox to which the Filter will be bound.
   */
  public final void bind(Inbox inbox)
  {
    owner = inbox;
  }

  /**
   * Sends a connection denial to the given address, which should be an
   * Outbox if there are to be no errors.  The Denial tells the
   * Outbox that the message it sent was denied for one reason
   * or another.  It does not tell the Outbox which message was denied.
   * @param from The Outbox which is to have its connection denied.
   */
  protected final void sendDenial(Place from)
  {
    try 
      {
        owner.getMailDaemon().send
          (new DenyMessage(DenyMessage.DENY), from, owner);
      }
    catch ( NullPointerException e )
      {
        throw new RuntimeException
          ("Attempted sendDenial() On Unbound Filter." );
      }
  }

  /**
   * Adds a Class (by name) to the allowed classes list.
   * @param s A String containing the name of the Class to add to the 
   * allowed classes list.
   */
  public final void addAllowedClass(String s)
  {
    allowedClasses.addElement(s);
  }

  /**
   * Adds a Class to the allowed classes list.
   * @param c The Class to add to the allowed classes list.
   */
  public final void addAllowedClass(Class c)
  {
    addAllowedClass(c.getName());
  }

  /**
   * Removes all occurrences of a Class (by name)
   * from the allowed classes list.
   * @param s A String containing the name of the Class to remove from the
   * allowed classes list.
   */
  public final void removeAllowedClass(String s)
  {
    if (s == null)
      return;

    int i = 0;

    while (i < allowedClasses.size())
      {
        if (s.equals((String) allowedClasses.elementAt(i)))
          allowedClasses.removeElementAt(i);
        else
          i++;
      }
  }
     
  /**
   * Removes all occurrences of a Class from 
   * the allowed classes list.
   * @param c The Class to remove from the allowed classes list.
   */
  public final void removeAllowedClass(Class c)
  {
    removeAllowedClass(c.getName());
  }

  /**
   * Empties the allowed classes list.
   * This operation is not reversible.
   */
  public final void clearAllowedClasses()
  {
    allowedClasses = new Vector();
  }

  /**
   * Adds a Class (by name) to the denied classes list.
   * @param s A String containing the name of the Class to add to the 
   * denied classes list.
   */
  public final void addDeniedClass(String s)
  {
    deniedClasses.addElement(s);
  }

  /**
   * Adds a Class to the denied classes list.
   * @param c The Class to add to the denied classes list.
   */
  public final void addDeniedClass(Class c)
  {
    addDeniedClass(c.getName());
  }

  /**
   * Removes all occurrences of a Class (by name)
   * from the denied classes list.
   * @param s A String containing the name of the Class to remove from the
   * denied classes list.
   */
  public final void removeDeniedClass(String s)
  {
    if (s == null)
      return;

    int i = 0;

    while (i < deniedClasses.size())
      {
        if (s.equals((String) deniedClasses.elementAt(i)))
          deniedClasses.removeElementAt(i);
        else
          i++;
      }
  }
     
  /**
   * Removes all occurrences of a Class from 
   * the denied classes list.
   * @param c The Class to remove from the denied classes list.
   */
  public final void removeDeniedClass(Class c)
  {
    removeDeniedClass(c.getName());
  }

  /**
   * Empties the denied classes list.
   * This operation is not reversible.
   */
  public final void clearDeniedClasses()
  {
    deniedClasses = new Vector();
  }

  /**
   * Performs class checking on an incoming message.
   * It first checks the class of the message against the classes in
   * the denied classes list, and returns false if it finds a match.
   * Then, it checks the class of the message against the classes in
   * the allowed classes list, and returns true if it finds a match.
   * If it does not find a match in either list, checkClass() returns
   * false if the allowed classes list is nonempty and true otherwise.
   * @param s A String containing the class name to be checked.
   * @return true if the string passes the class check, false otherwise.
   */
  public final boolean checkClass(String s)
  {
    if (s == null)
      return false;

    Enumeration e = deniedClasses.elements();

    if (Debug.on)
      Debug.println("FilterStatus", "Performing Denied Classes Check...");

    while (e.hasMoreElements())
      {
        if (s.equals((String) e.nextElement()))
          return false;
      }

    e = allowedClasses.elements();

    if (Debug.on)
      Debug.println("FilterStatus", "Performing Allowed Classes Check...");

    while (e.hasMoreElements())
      {
        if (s.equals((String) e.nextElement()))
          return true;
      }

    if (Debug.on)
      Debug.println("FilterStatus", 
                    "Performing Allowed Classes List Size Check...");

    if (allowedClasses.size() > 0)
      return false;
    else
      return true;
  }

  /**
   * Performs class checking on an incoming message.
   * (uses the same criteria as the checkClass() which takes a String)
   * @param m The Message to be checked.
   * @return true if the message passes the class check, false otherwise.
   */
  public final boolean checkClass(Message m)
  {
    if (m == null)
      return false;

    return checkClass(m.getClass().getName());
  }

  /**
   * Performs class checking on an incoming message.
   * (uses the same criteria as the checkClass() which takes a String)
   * @param c The Class to be checked.
   * @return true if the class passes the class check, false otherwise.
   */
  public final boolean checkClass(Class c)
  {
    if (c == null)
      return false;

    return checkClass(c.getName());
  }

  /**
   * Adds a host (by InetAddress) to the allowed hosts list.
   * @param i The InetAddress of the host to add.
   */
  public final void addAllowedHost(InetAddress i)
  {
    allowedHosts.addElement(i);
  }

  /**
   * Adds a host (by hostname) to the allowed hosts list.
   * @param s A String containing the hostname of the host to add.
   */
  public final void addAllowedHost(String s)
  {
    InetAddress i;
      
    try
      {
        i = InetAddress.getByName(s);
        addAllowedHost(i);
      }
    catch (UnknownHostException e)
      {
      }
  }
   
  /**
   * Removes all occurrences of a host (by InetAddress) from the
   * allowed hosts list.
   * @param a The InetAddress of the host to remove.
   */
  public final void removeAllowedHost(InetAddress a)
  {
    if (a == null)
      return;

    int i = 0;

    while (i < allowedHosts.size())
      {
        if (a.equals((InetAddress) allowedHosts.elementAt(i)))
          allowedHosts.removeElementAt(i);
        else
          i++;
      }
  }

  /**
   * Removes all occurrences of a host (by hostname) from the
   * allowed hosts list.
   * @param s A String containing the hostname of the host to remove.
   */
  public final void removeAllowedHost(String s)
  {
    InetAddress i;

    try
      {
        i = InetAddress.getByName(s);
        removeAllowedHost(i);
      }
    catch (UnknownHostException e)
      {
      }
  }

  /**
   * Clears the allowed hosts list. This action is not reversible.
   */
  public final void clearAllowedHosts()
  {
    allowedHosts = new Vector();
  }

  /**
   * Adds a host (by InetAddress) to the denied hosts list.
   * @param i The InetAddress of the host to add.
   */
  public final void addDeniedHost(InetAddress i)
  {
    deniedHosts.addElement(i);
  }

  /**
   * Adds a host (by hostname) to the denied hosts list.
   * @param s A String containing the hostname of the host to add.
   */
  public final void addDeniedHost(String s)
  {
    InetAddress i;
      
    try
      {
        i = InetAddress.getByName(s);
        addDeniedHost(i);
      }
    catch (UnknownHostException e)
      {
      }
  }
   
  /**
   * Removes all occurrences of a host (by InetAddress) from the
   * denied hosts list.
   * @param a The InetAddress of the host to remove.
   */
  public final void removeDeniedHost(InetAddress a)
  {
    if (a == null)
      return;

    int i = 0;

    while (i < deniedHosts.size())
      {
        if (a.equals((InetAddress) deniedHosts.elementAt(i)))
          deniedHosts.removeElementAt(i);
        else
          i++;
      }
  }

  /**
   * Removes all occurrences of a host (by hostname) from the
   * denied hosts list.
   * @param s A String containing the hostname of the host to remove.
   */
  public final void removeDeniedHost(String s)
  {
    InetAddress i;

    try
      {
        i = InetAddress.getByName(s);
        removeDeniedHost(i);
      }
    catch (UnknownHostException e)
      {
      }
  }

  /**
   * Clears the allowed hosts list. This action is not reversible.
   */
  public final void clearDeniedHosts()
  {
    deniedHosts = new Vector();
  }

  /**
   * Performs host checking on an incoming message.
   * It first checks the host of the message against the hosts in
   * the denied hosts list, and returns false if it finds a match.
   * Then, it checks the host of the message against the hosts in
   * the allowed hosts list, and returns true if it finds a match.
   * If it does not find a match in either list, checkHost() returns
   * false if the allowed hosts list is nonempty and true otherwise.
   * @param p The InetAddress to be checked.
   * @return true if the InetAddress passes the host check, false otherwise.
   */
  public final boolean checkHost(InetAddress i)
  {
    if (i == null)
      return false;

    Enumeration e = deniedHosts.elements();

    if (Debug.on)
      Debug.println("FilterStatus", "Performing Denied Hosts Check...");

    while (e.hasMoreElements())
      {
        if (i.equals((InetAddress) e.nextElement()))
          return false;
      }

    e = allowedHosts.elements();

    if (Debug.on)
      Debug.println("FilterStatus", "Performing Allowed Hosts Check...");

    while (e.hasMoreElements())
      {
        if (i.equals((InetAddress) e.nextElement()))
          return true;
      }

    if (Debug.on)
      Debug.println("FilterStatus", 
                    "Performing Allowed Hosts List Size Check...");

    if (allowedHosts.size() > 0)
      return false;
    else
      return true;
  }

  /**
   * Performs host checking on an incoming message.
   * (it uses the same criteria as the checkHost() which takes an
   * InetAddress).
   * @param p The Place to be checked.
   * @return true if the Place passes the host check, false otherwise.
   */
  public final boolean checkHost(Place p)
  {
    if (p == null) 
      return false;

    return checkHost(p.addr());
  }

  /**
   * Adds a Place to the allowed places list.
   * @param p The Place to be added.
   */
  public final void addAllowedPlace(Place p)
  {
    allowedPlaces.addElement(p);
  }

  /**
   * Removes all occurrences of a Place from the allowed places list.
   * @param p The Place to be removed.
   */
  public final void removeAllowedPlace(Place p)
  {
    if (p == null)
      return;

    int i = 0;
      
    while (i < allowedPlaces.size())
      {
        if (p.equals((Place) allowedPlaces.elementAt(i)))
          allowedPlaces.removeElementAt(i);
        else
          i++;
      }
  }

  /**
   * Clears the allowed places list. This action is not reversible.
   */
  public final void clearAllowedPlaces()
  {
    allowedPlaces = new Vector();
  }

  /**
   * Adds a Place to the denied places list.
   * @param p The Place to be added.
   */
  public final void addDeniedPlace(Place p)
  {
    deniedPlaces.addElement(p);
  }

  /**
   * Removes all occurrences of a Place from the denied places list.
   * @param p The Place to be removed.
   */
  public final void removeDeniedPlace(Place p)
  {
    if (p == null)
      return;

    int i = 0;
      
    while (i < deniedPlaces.size())
      {
        if (p.equals((Place) deniedPlaces.elementAt(i)))
          deniedPlaces.removeElementAt(i);
        else
          i++;
      }
  }

  /**
   * Clears the denied places list. This action is not reversible.
   */
  public final void clearDeniedPlaces()
  {
    deniedPlaces = new Vector();
  }

  /**
   * Performs place checking on an incoming message.
   * It first checks the place of the message against the places in
   * the denied places list, and returns false if it finds a match.
   * Then, it checks the place of the message against the places in
   * the allowed places list, and returns true if it finds a match.
   * If it does not find a match in either list, checkPlace() returns
   * false if the allowed places list is nonempty and true otherwise.
   * @param m The message to be checked.
   * @return true if the message passes the place check, false otherwise.
   */
  public final boolean checkPlace(Place p)
  {
    if (p == null)
      return false;

    Enumeration e = deniedPlaces.elements();

    if (Debug.on)
      Debug.println("FilterStatus", "Performing Denied Places Check...");

    while (e.hasMoreElements())
      {
        if (p.equals((Place) e.nextElement()))
          return false;
      }

    e = allowedPlaces.elements();

    if (Debug.on)
      Debug.println("FilterStatus", "Performing Allowed Places Check...");

    while (e.hasMoreElements())
      {
        if (p.equals((Place) e.nextElement()))
          return true;
      }

    if (Debug.on)
      Debug.println("FilterStatus", 
                    "Performing Allowed Places Size Check...");

    if (allowedPlaces.size() > 0)
      return false;
    else
      return true;
  }

  /**
   * Determines whether a message/sender pair is acceptable. It performs
   * three checks - a class check, a host check and a place check. If a
   * message fails any one of these checks, it is deemed to be unacceptable.
   * Otherwise, it is deemed to be acceptable.
   * @param m The message which is to be examined.
   * @param from The address of the Outbox which sent the message.
   * @return true if the message is acceptable, false otherwise.
   */
  public boolean checkMessage(Message m, Place from)
  {
    boolean ok = (checkClass(m) && checkHost(from) && checkPlace(from));
    if (!ok) 
      {
        if (Debug.on)
          Debug.println("FilterResult", "Message " + m + "/Place " + from + 
                        " Failed Filter."); 
         
        sendDenial(from);
      }
    else if (Debug.on)
      Debug.println("FilterResult", "Message " + m + "/Place " + from + 
                    " Passed Filter.");

    return ok;
  }

} // end of Filter class

