/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
  $Id: Inbox.java,v 1.22 1997/05/29 07:07:03 kiniry Exp $
*/


package info.net;

import java.util.*;


// This needs some serious cleaning up.  -- Adam

/**
 *  The Inbox receives messages from the MailDaemon and
 *  files them away in a queue.  The messages can be recovered via the
 *  receive() method.  If there are no messages in the queue, the receive
 *  method blocks until there are.  There are other methods which allow for
 *  greater control over the Inbox such as a empty() method which
 *  checks whether the queue is currently empty, and a waitForMessage()
 *  method which will wait for a message to arrive in the queue, with 
 *  optional abilities to set finite times.<P>
 *
 *  In addition there are two other pieces of data associated with any given
 *  message, the local time stamp, which allows for a complete ordering of
 *  all messages received locally by time of arrival, and the return address
 *  which is the address of the mailbox which sent the message.<P>
 *  To get these two bits of information, use the time() and from() methods.
 *  These methods act on the oldest message in the queue.  Remove the message,
 *  lose the info.
 *
 * @author Adam Rifkin
 * @author Luke Weisman
 * @version 1.0b2 $Revision: 1.22 $ $Date: 1997/05/29 07:07:03 $
 * @see Receivebox
 * @see SyncInbox
 * @see ReceiveSet
 */

public class Inbox extends Mailbox
{ 

  /**
   * This is a queue of all pending messages which haven't been removed
   * via the receive() method.  They got in the queue via the
   * insertMessage method.
   */
  protected MessageQ incoming;

  /**
   * This filter is what is used to test incoming messages.  If the 
   * filter's checkMessage method returns false, the message is 
   * rejected, and not put in the incoming queue.
   */
  private Filter filter;


  /**
   * Make a incoming mailbox with a given name, and a reference to the
   * processes's Mail Daemon.<P>
   *
   * WARNING: Names of the form __x where x is some number are reserved.
   *    Do not use them.
   *
   * @param name  The name you want for your incoming mailbox.
   * @param m     The MailDaemon with which your incoming mailbox should \
   *              be registered.
   * @exception MailDaemonException when the name conflicts with prior
   * mailbox.
   */
  public Inbox( MailDaemon m, String name ) 
    throws MailDaemonException
  {
    super( m, name );
    incoming = new MessageQ();
  }


  /**
   * Make a incoming mailbox with no specified name.  It will automatically
   * generate a name for it.
   *
   * @param m     The MailDaemon where the new mailbox will register itself.
   */
  public Inbox( MailDaemon m )
  {
    super( m ); 
    incoming = new MessageQ();
  }




  /* 
   *                               METHODS
   */


  /**
   * Check to see if there are any message pending.
   *
   * @return   True if there are no message pending, false otherwise.
   * @see      empty
   */
  public boolean nonEmpty()
  {
    return ( !incoming.isEmpty() );
  }

   
  /**
   * Check to see if there are any messages pending.
   * 
   * @return    True if there are no messages pending, false otherwise.
   * @see       nonEmpty
   */
  public boolean empty()
  {
    return (incoming.isEmpty());
  }


  /**
   * Grab the oldest message in the queue and return it.  If there are
   * no messages, block until there is one.
   *
   * @return   The oldest message in the queue.
   */
  public Message receive()
  {
    MessageType m;

    //wait for a message to arrive, no matter what.
    silentWaitForMessage();

    // pull off the front of the queue
    m = incoming.GetAndKillFirst();

    return m.msg;
  }


  /**
   * Grab the time stamp on the oldest message in the queue.  The time
   * stamp allows for absolute ordering of all messages received by the
   * maildaemon associated with this mailbox.
   * <P> 
   * Note that the time stamp is equivelent to the message placement
   * given all messages received since the maildaemon's construction. 
   * This means you know that if the last message received anywhere had
   * a time stamp of some number, the time stamp of the second oldest
   * message would be one greater than that.
   *
   * @return  the aforementioned time.
   */
  public int time ()
  {
    if (nonEmpty())
      return incoming.GetFirst().timestamp;
    else
      return -1;
  }

  /**
   * Put a message in the queue with the specified timestamp and the
   * specified return address.  The message needs to be unpacked via
   * the Message static fromByteStream.
   *
   * @param time  The time stamp on the message
   * @param s     The message, as the raw byte stream.
   * @param from  The place which sent the messag.
   * @return      True if the message was accepted, false if it was
   filtered out.
   */
  synchronized protected boolean insertMessage(int time, 
                                               Message m,
                                               Place from )
    throws MailDaemonException
  {
    if ( ( filter == null ) || ( filter.checkMessage( m, from ) ) )
      {
        MessageType packet = new MessageType();
        packet.msg = m;
        packet.timestamp = time;
        packet.from = from;
         
        incoming.addElement(packet);
         
        //Wake up one of the people waiting for a message (if there is
        //any).
        notify();

        return true;
      }
    else
      return false;
  }


  // FIX THIS SO THE TIME UNITS ARE NORMALIZED!! -- Adam

  /**
   * Wait until a message arrives.  Will abort immediately if there
   * is already a message in the queue.   If no message arrives in the
   * given time, it simply returns, and you have to manually check to
   * see if the box has a message or not.
   * @param time_ms The time to wait in milliseconds.
   * @param time_ns The additional time to wait in nanoseconds, if you care.
   * @exception InterruptedException If the waiting thread gets 
   *              interrupted.
   */
  synchronized public void waitForMessage( long time_ms, 
                                           int time_ns ) 
    throws InterruptedException
  {
    if ( empty() )
      wait( time_ms, time_ns );
  }

  /**
   * Wait for the queue to be nonempty for time milliseconds.
   * @param time_ms The time to wait in milliseconds.
   * @exception InterruptedException If the waiting thread gets interrupted.
   */
  synchronized public void waitForMessage( long time_ms ) 
    throws InterruptedException
  {
    waitForMessage( time_ms, 0 );
  }

  /**
   * This will not exit unless the incoming mailbox is not empty.
   * @exception InterruptedException When the owning thread gets interrupted.
   */
  synchronized public void waitForMessage( ) 
    throws InterruptedException
  {
    while ( empty() )
      wait( );
  }

  /**
   * This will not exit unless the incoming mailbox is not empty.  It catches
   * interrupts and swallows them in order to do this.
   */
  synchronized public void silentWaitForMessage( ) 
  {
    while ( empty() )
      {
        try 
          {
            wait( );
          }
        catch ( InterruptedException e ) {}
      }
  }


  /**
   * @return  Returns the place which sent the current oldest message to
   *          incoming mailbox.  If you have removed this message via the
   *          receive() call, this data is lost.  (rather it will return
   *          the next message in the queue's sender)
   */
  public Place from () 
  {
    if (nonEmpty())
      return incoming.GetFirst().from;
    else
      return null;
  }


  /**
   * Set the filter of the incoming mailbox to the passed value.  If a 
   * multiple filter arrangement is desired, see the FilterSet filter. 
   * Passing null makes the filter let everything through.
   * A filter will have its bind() method called when this method is
   * called.
   *
   * @param f The filter to use on the messages.
   * @see Filter
   */
  public void setFilter( Filter f )
  {
    filter = f;

    //bind the filter to this mailbox.
    if ( filter != null )
      filter.bind( this );
  }

} // end of Inbox class
