/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: MailDaemon.java,v 1.55 1997/05/30 00:35:45 dmz Exp $
 */

package info.net;

import info.util.Debug;
import java.net.*;
import java.io.*;
import java.util.*;

/**
 * MailDaemon manages the Mailboxes of a Djinn.  It listens on a port
 * for Messages addressed to it incoming Mailboxes, and processes those
 * Messages by eventually putting them in the inbound queues of the
 * appropriate incoming Mailboxes.  It also delivers Messages from
 * the outbound queues of outgoing Mailboxes to the appropriate
 * incoming destination Mailboxes (which can be on a different host).
 * MailDaemon has a set of background threads running as a daemon,
 * that guarantees reliable, ordered message passing. <P>
 *
 * Mailboxes in a Djinn are registered with its MailDaemon by
 * the Mailbox constructors. <P>
 *
 * @author Adam Rifkin
 * @author Matt Richardson
 * @author Luke Weisman
 * @version 1.0b2 $Revision: 1.55 $ $Date: 1997/05/30 00:35:45 $
 * @see Broadcastbox
 * @see Inbox
 * @see MailDaemonIn
 * @see MailDaemonOut
 * @see Multicastbox
 * @see Outbox
 * @see Receivebox
 * @see Sendbox
 * @see java.net.DatagramSocket
 **/

public class MailDaemon
{

  /**
   * This is the ID put in messages when the target ID is not yet
   * known.  It always matches the MailDaemon it hits.
   */
  public static int TARGET_ID_UNKNOWN = -1;

  /**
   * The socket which is used by everything.
   */
  private DatagramSocket _socket;

  /**
   * This is the outgoing half of the MailDaemon.  It sends out messages,
   * periodically checks to see if they have been acknowledged, and re-
   * sends them if they haven't.  <STRONG><EM> "Periodically" checks?  I
   * hope this isn't any sort of busy-loop. </EM></STRONG>
   */
  private MailDaemonOut outDaemon = null;

  /**
   * Theis is the incoming half of the MailDaemon.  It listens for new
   * message arrivals, processes them, and places them in the
   * appropriate mailbox's queue.  Messages all have a header with the
   * target mailbox name in them.
   */
  private MailDaemonIn inDaemon = null;

  /**
   * Unique id of the MailDaemon so there is no confusion over
   * messages.  It allows for unique message ids across seperate
   * instantiations of various programs using the same port numbers
   * and machines. Currently it
   * time.  <STRONG><EM>I'm unsure as to whether this is necessary.
   * Is it actually used in places?  If so, it really needs to use a
   * different mechanism or be much more fine-grained.</EM></STRONG>
   */
  protected long id;

  /**
   * The name counter for name generation.  This is used to generate
   * locally unique mailbox names. <STRONG><EM>Should this be private?
   * </EM></STRONG>
   */
  int _nameCntr;

  /**
   * The md's personal mailbox for sending some messages.  This is
   * used if a dummy mailbox is called for for some return address,
   * or some such. <STRONG><EM>Explain in more detail.</EM></STRONG>
   */
  Mailbox MAILDAEMON;

  private static String INBOX_NAME = "__IN_MD";
  private Receivebox md_in;
  private MultiURLClassLoader mucl;

  // constructors

  /**
   * This does the actual creation of the mailDaemon.  It is called
   * once by either of the two constructors. (sub-constructor)
   *
   * @param portnumber
   * @param mucl
   */

  void createMailDaemon (int portnumber,
                         MultiURLClassLoader mucl)
    throws MailDaemonException
  {
    MakeTimeID();
    _nameCntr = -1;

    // Make the new socket.

    try
      {
        if ( portnumber >= 0 )
          _socket = new DatagramSocket( portnumber);
        else
          //let the MailDaemon choose a random port.
          _socket = new DatagramSocket();
      }
    catch (SocketException socketException)
      {
        throw new MailDaemonException ("SocketException: " +
                                       socketException.getMessage ());
      }

    // Generate the outgoing and incoming halves of the maildaemon.
    // These constructors will start their threads up.

    outDaemon = new MailDaemonOut( this, _socket );
    inDaemon = new MailDaemonIn( this, _socket , mucl);

    // Generate the mailDaemon box's name.  Can't make the box now,
    // since we are in the constructor of the MailDaemon, and the box
    // uses it.

    MAILDAEMON = new Outbox( this );
    try
      {
        md_in = new Receivebox( this, INBOX_NAME );
      }
    catch (MailDaemonException mailDaemonException)
      {
        Debug.bugExit ("createMailDaemon",
                       "MailDaemonException making new MailDaemon");
        mailDaemonException.printStackTrace();
        throw new RuntimeException("MailDaemonException");
      }

    // Keep a reference to our MultiURLClassLoader.

    this.mucl = mucl;

    // Start up the MailDaemon in and out threads.

    inDaemon.start();
    outDaemon.start();
  }


  /**
   * Make a mailDaemon with the specified port number.
   *
   * @param portnumber The port desired.
   */

  public MailDaemon (int portnumber)
    throws MailDaemonException
  {
    createMailDaemon (portnumber, new MultiURLClassLoader(null, null));
  }

  /**
   *  Make a mailDaemon with the specified port number.
   *
   * @param portnumber The port desired.
   * @param mucl Classloader to use to (possibly) remote load messages
   */

  public MailDaemon (int portnumber,
                     MultiURLClassLoader mucl)
    throws MailDaemonException
  {
    createMailDaemon (portnumber, mucl);
  }


  /**
   * Make a maildaemon with a randomly chosen port number.  It
   * will find any port number which is vacant.
   */

  public MailDaemon ()
    throws MailDaemonException
  {
    createMailDaemon (-1, null);
  }


  /**
   * Make a maildaemon with a randomly chosen port number.  It
   * will find any port number which is vacent.
   * @param mucl Classloader to use to (possibly) remote load messages
   */

  public MailDaemon (MultiURLClassLoader mucl)
    throws MailDaemonException
  {
    createMailDaemon(-1, mucl);
  }


  // Methods

  /**
   * Sends a message msg to place.<P>
   *
   * Warning: This should not be used directly.  Use some variation of
   *          an Outbox instead.
   *
   * @param msg The message to be sent.
   * @param targ The destination of the message.
   * @param from The mailbox which is sending the message.
   */
  public void send (Message msg,
                    Place targ,
                    Mailbox from)
  {
    Debug.push ("MailDaemon.send");

    if ( targ.bad() )
      System.err.println( "Will not send to bad address: " + targ
                          + " (" + targ.whyBad() + ")" );
    else
      outDaemon.send (msg, targ, from);
    Debug.pop ("MailDaemon.send");
  }

  /**
   * Register a mailbox with the mailDaemon.  This is required if the
   * mailbox is to work properly.  The mailbox registers under its
   * given name.  It should only register once.  The Mailbox ()
   * constructors _all_ register with the mailDaemon, so this method
   * should not ever need to be called by a developer.
   *
   * @param prt The Mailbox to register.
   * @exception MailDaemonException If it find the mailbox's name already in the
   *            register.  This happens if there are _either_ send or
   *            receive mailboxes in the register.
   */
  protected void register( Mailbox prt )
    throws MailDaemonException
  {
    inDaemon.register( prt );
  }

  /**
   * Get a reference to a mailbox by a name lookup.
   *
   * @param name The name of the box desired.
   * @return The local mailbox with the unique passed name.  Null
   *         if the box is not found.
   */
  public Mailbox get( String name )
  {
    return inDaemon.get( name );
  }

  /**
   * @return The local port number of the mailDaemon.  This is the
   *          port that the mailDaemon is listening on for messages
   *          from other Djinns.
   */
  public int getLocalPort()
  {
    return _socket.getLocalPort();
  }

  /**
   * @return The local address of the mailDaemon.  This is an IP
   *         address, without port number. <STRONG><EM>How is this method used?
   *         Won't you always need more than just an IP address?</EM></STRONG>
   */
  public InetAddress getLocalAddr()
  {
    try
      {
        return InetAddress.getLocalHost();
      }
    catch ( UnknownHostException e )
      {
        return null;
      }
  }

  /**
   * The MailDaemonIn half of the mailDaemon uses this method to
   * send received messages to the MailDaemonOut.  This is from
   * when the MailDaemonIn receives a message acknowledgement, and
   * the MailDaemonOut needs to process it.  <STRONG><EM>I don't
   * quite understand these comments.  Clairify please.</EM></STRONG>
   */
  void SendToOutDaemon( SplitMessage m, AckMessage am, Place p )
  {
    outDaemon.ReceiveAck( m, am, p );
  }

  /**
   * The MailDaemonIn uses the MailDaemonOut to send message
   * acknowledgements to the MailDaemonOut.  This function does
   * this by passing information to the MailDaemonOut.
   *
   * @param from The mailbox which received the original message.
   */
  void SendAck( AckMessage ack, Place targ, long md_id, String from )
  {
    outDaemon.SendAck( ack, targ, md_id, from );
  }

  /**
   * waitForNewMessage waits until a new message arrives.  It will
   * wait forever.
   * @return A reference to a receive mailbox which has a message pending.
   * @exception InterruptedException If the thread is interrupted via an
   *            interrupt call, it throws this exception.
   */
  public Inbox waitForNewMessage( )
    throws InterruptedException
  {
    return inDaemon.waitForNewMessage( 0, 0 );
  }

  /**
   * Wait until a new message arrives or until the specified time has
   * expired. <STRONG><EM>This routine should really expect a specified
   * time in millisecond, not seconds.  Otherwise you constrain the user
   * too much even though there is an alternate interface.</EM></STRONG>
   * @param time The amount of time, in milliseconds,
   *        to wait for a new message.
   * @return A reference to a receive mailbox which has a message pending or
   *         null if there is no such receive mailbox.
   * @exception InterruptedException If the thread is interrupted via an
   *            interrupt call, it throws this exception.
   */
  public Inbox waitForNewMessage( long time )
    throws InterruptedException
  {
    return inDaemon.waitForNewMessage( time, 0 );
  }

  /**
   * Wait until there is a message in some Inbox or derivitive.
   * All Outboxes and their ilk are ignored.
   * Equivelent to a waitForMessage( 0 ) call.
   * @return A reference to the receive mailbox which has oldest message
   *         pending.
   * @exception InterruptedException If the thread is interrupted via an
   *            interrupt call, it throws this exception.
   */
  public Inbox waitForMessage( )
    throws InterruptedException
  {
    return inDaemon.waitForMessage( 0, 0 );
  }

  /**
   * Wait until a message arrives.  If there is a message in any of
   * the registered receive mailboxes, it returns immediately.  It
   * returns a reference to the mailbox which has the oldest message
   * over all the mailboxes.  (i.e. the message with the smallest time
   * stamp.)  A time of 0 waits forever.
   *
   * @return A reference to the receive mailbox which has oldest message
   *         pending, or null if there is no such mailbox.
   * @param time The amount of time in milliseconds to wait for a new message.
   * @exception InterruptedException If the thread is interrupted via an
   *            interrupt call, it throws this exception.
   */
  public Inbox waitForMessage( int time )
    throws InterruptedException
  {
    return inDaemon.waitForMessage( time, 0 );
  }


  /**
   * Generate a hopefully unique name for either a send or a receive
   * mailbox.
   *
   * WARNING: To make this work, do not name mailboxes with such names
   *  as "__x" where x is any integer, as these are the names the
   * name generator makes.  (It makes them sequentially, starting from
   * 0, except the first name is reserved by the MailDaemon.
   */
  synchronized public String GenerateName( )
  {
    _nameCntr++;
    return new String( "__" + _nameCntr );
  }

  /**
   * This method will delete a mailbox and all of its queued messages
   * It should be used when a specific target has died or is
   * discovered to have never existed.  It should be used with extreme
   * caution. <P>
   * This is basically a memory management scheme so that long-term
   * Djinns that might talk to many other Djinns will not end up with
   * large blocks of unused memory.  It also has to do with
   * efficiency, since the thread will keep attempting to send packets
   * to the bad address until the table is purged.   For short term
   * Djinns it does not matter as much.<P>
   * Note that if a send mailbox attempts to send to the given place
   * after purging, a new table will be constructed, and the ID
   * counters will be reset to 0, which does not seem to pose a
   * problem, but little to no real investigation into the subject has
   * been done.  <STRONG><EM>We need to figure out if this is a problem
   * or not. Conditionaly in comments and code don't exactly instill a
   * great sense of faith.  i.e. this method "should" do this and "might"
   * work all the time...</EM></STRONG>
   *
   * @param p    The target place to no longer send messages to.
   * @param from The mailbox whose messages we should purge.  null means
   *             _all_ mailboxes, regardless of shape, creed, or color.
   * @return     The list of all unacknowledged messages to the given place.
   */
  public Vector purgeAddress( Place p, Mailbox from )
  {
    return outDaemon.purgeAddress( p, from );
  }

  /**
   * Wait until all the pending outgoing messages have been sent and
   * acknowledged.  If they never are, then this never exits.  <STRONG>
   * I can't say I'm happy with all of the "this method will never return
   * comments I see here an there.  Distributed systems _always_ need
   * timeout mechanisms and properly percolated exceptions thrown.  We
   * really need to work this out in all instances.<EM></EM></STRONG>
   * @exception InterruptedException If the thread calling this method
   *            gets an interrupt call.
   */
  public void waitUntilFlushed()
    throws InterruptedException
  {
    outDaemon.waitUntilFlushed();
  }

  /**
   * Returns true if the given box has successfully sent all its
   * messages.
   */
  public boolean isFlushed( Mailbox mb )
  {
    return outDaemon.isFlushed( mb );
  }

  /**
   * Wait until all the pending outgoing messages have been sent and
   * acknowledged.  If they never are, then this never exits.  This
   * version swallows interrupt calls.
   */
  public void silentWaitUntilFlushed()
  {
    while ( true )
      {
        try
          {
            outDaemon.waitUntilFlushed();
            return;
          }
        catch ( InterruptedException e )
          { }
      }
  }

  /**
   * Make the unique ID out of the current time.  <STRONG><EM>How unique
   * is it?</EM></STRONG>
   */
  private void MakeTimeID( )
  {
    id = System.currentTimeMillis() % ( 365 * 24 * 3600 );
    //modded out to year.
    Debug.lpln( 13, "MailDaemon id = " + id );
  }

  /**
   * Get the unique ID of the mailDaemon.  It is not really that
   * unique, since it is based on the number of quarter minute
   * intervals since the epoch, thus wrapping every 72 hours.
   * <STRONG><EM>Ah hah!</EM></STRONG>
   */
  public long getID()
  {
    return id;
  }

  /**
   * Get the MailDaemon's MultiURLClassLoader.
   * @return the MultiURLClassLoader.
   */
  public MultiURLClassLoader getClassLoader()
  {
    return mucl;
  }

  /**
   *
   **/

  String boxList ( )
  {
    return inDaemon.boxList();
  }


  /**
   * This does the actual destruction of a Mailbox.
   **/

  public void destroy( Mailbox mb )
  {
    inDaemon.killMailbox( mb );
    outDaemon.killMailbox( mb );
  }


  /**
   * This does the actual destruction of a MailDaemon.
   **/

  public void destroy ( )
  {
    // Stop threads.
    inDaemon.destroy ( );
    outDaemon.destroy ( );

    // Set them up for garbage collection.
    inDaemon = null;
    outDaemon = null;

    // kill our socket (we send a small packet to it first because
    // some implementations of java.net are broken)
    
    try 
     {
       DatagramPacket datagramPacket = 
         new DatagramPacket(new byte[1], 1,
                            InetAddress.getLocalHost(),
                            _socket.getLocalPort());
    
       DatagramSocket outSocket = new DatagramSocket();
       outSocket.send(datagramPacket);
       outSocket.close();
     }
    catch (Exception e)
     {
       Debug.lpln(13, "Exception raised when trying" +
                  "to send empty packet to our socket: " + e);
     }
    _socket.close();

    System.gc ();
  }

} // end of MailDaemon class



