/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 *  $Id: MailDaemonIn.java,v 1.43 1997/05/29 07:07:08 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import java.net.*;
import java.io.*;
import java.util.*;

/**
 * This aspect of the MailDaemon deals with the incoming messages.  It
 * listens on the given socket, pulls out any packets, and process
 * them as best it is able.  It does this on its own tread, and tries
 * not to interfear with the Djinn that owns it.
 *
 * @author Luke Weisman
 * @author Wesley Tanaka
 * @author Adam Rifkin
 * @version 1.0b2 $Revision: 1.43 $ $Date: 1997/05/29 07:07:08 $
 * @see BoxSet
 * @see MailDaemon
 * @see Mailbox
 * @see MultiURLClassLoader
 * @see Place
 * @see SplitMessage
 * @see java.lang.InterruptedException
 * @see java.lang.Thread
 * @see java.net.DatagramPacket
 * @see java.net.DatagramSocket
 */

class MailDaemonIn implements Runnable
{
  /**
   * This is the max size of any packet which can arrive and be
   * accepted.
    */

  public final int MAX_PACKET_SIZE = 64000;

  /**
    * The socket to listen on (passed by MailDaemon in constructor)
    */

  DatagramSocket _socket;

  /**
   * The set of all registered mailboxs which can receive messages.
   */

  private BoxSet portset;

  /**
   * The thread
   */

  private Thread life = null;

  /**
   * The ClassLoader for incoming messages.
   */

  MultiURLClassLoader mucl = null;

  /**
   * the owner to pass messages to if needed.
   */

  MailDaemon owner;


  // constructors

  /**
   * This makes a new MailDaemonIn which has the passed owner, and
   * listens on the passed socket.
   * @param the_owner MailDaemon which interacts with this object.
   * @param listenAt  The socket to pull packets off of.
   * @param mucl      The MultiURLClassLoader.  Set this to null
   *                  if you want to run an applet.
   */

  MailDaemonIn( MailDaemon the_owner,
                DatagramSocket listenAt,
                MultiURLClassLoader mucl )
  {
    if ( listenAt != null )
      _socket = listenAt;

    //make new set to hold all the mailboxs.
    portset = new BoxSet(mucl);
    this.mucl = mucl;

    //Start up thread for the mail deamon listener.
    life = new Thread (this, "MailDaemonInThread" );
    life.setPriority( Thread.MAX_PRIORITY );
    life.setDaemon( true );

    owner = the_owner;
  }

  void start( )
  {
    life.start();
  }


  // methods

  /**
   * Given a box, it registers its name, so it will send
   * messages tagged for it to it in the future. <P>
   */

  public void register( Mailbox rp )
    throws MailDaemonException
  {
    portset.register( rp );
  }

  public void killMailbox( Mailbox rp )
  {
    portset.kill( rp );
  }

  /**
   * Get a box from the table of all boxes by name.
   *
   * @param name The name of the box desired.
   * @return The local mailbox with the unique passed name.  Null
   *         if the box is not found.
   */

  public Mailbox get( String name )
  {
    return portset.Grab( name );
  }


  /**
   * Get list of all boxes
   */

  String boxList( )
  {
    return portset.boxList();
  }

  /**
   * This takes a DatagramPacket and does with it what it will.  It
   * attempts to turn it into a SplitMessage.  If this is successfully
   * it will deal with the splitmessage accordingly (either pass it to
   * MailDaemonOut if it is an ack, or pass it to a receivemailbox if
   * it is anything else and if there is one by the passed name.
   */

  private void processMessage( DatagramPacket packet )
  {
    String name = null;
    boolean acc = false, bad = false;
    int i;
    AckMessage ack;
    SplitMessage m = null;
    Place from = null;

    try
      {
        m = new SplitMessage( packet.getData() );
      }
    catch ( IOException e )
      {
        //message corrupt.
        bad = true;
        return;
      }

    Debug.lpln( 13, "Processing " + m );

    //This is the return address for the packet.
    from = new Place( packet.getAddress(),
                      packet.getPort(),
                      m.FName() );

    if ((m.dID () != owner.getID ()) &&
        (m.dID () != MailDaemon.TARGET_ID_UNKNOWN))
      {
        //message for different maildaemon.
        Debug.lpln( 13, "Error: maildaemon id = " + m.dID() + " not known." );

        if ( !m.isAck() )
          owner.SendAck( new AckMessage( -1, AckMessage.NEW_RECEIVER ),
                         from, m.mydID(), m.PName() );
        else
          //drop an ack to a dead maildaemon.  no point
          ;
      }

    //acks are not for us, send it on.
    if ( m.isAck() )
      {
        try
          {
            AckMessage am = (AckMessage)portset.readObject( m.Message() );
            owner.SendToOutDaemon( m, am, from );
          }
        catch ( Exception e )
          {
            e.printStackTrace();
          }
      }
    else      //deal with the normal message to some mailbox
      {
        //This will attempt to foist the message off on some
        //receive mailbox.  A null ack means there was no
        //receive mailbox by the name in the splitmessage.
        ack = portset.insertMessage( m, from );

        if ( ack != null )
          owner.SendAck( ack, from, m.mydID(), m.PName() );
        /*else
          {
          Debug.lpln( 13, "Not sending any response." );
          ;
          }
          */
      }
  } //end processMessage


  /**
   * This loops forever, listening at the assigned port for messages,
   * and when receiving messages, routing them to their proper place.
   */

  public void run()
  {
    byte [] sockbuf;
    DatagramPacket packet;

    //the listen loop.  Do this forever (or until killed)
    while ( true )
      {
        try
          {
            sockbuf = new byte[ MAX_PACKET_SIZE ];
            packet = new DatagramPacket(sockbuf,MAX_PACKET_SIZE);

            _socket.receive( packet );
            processMessage( packet );
          }
        catch (IOException ioException)
          {
            Debug.bugExit ("MailDaemonIn run",
                           "IOException in listen loop");
            ioException.printStackTrace();
            throw new RuntimeException ("IOException");
          }

        //give other threads a chance.
        //life.yield( );
      }
  }

  /**
   * Waits for a new message to arrive, and then returns the receive
   * mailbox to which that message was given.
   * @exception InterruptedException When the waiting thread gets an
   *            interrupt call.
   */

  Inbox waitForNewMessage( long time_s, int time_ms )
    throws InterruptedException
  {
    return portset.waitForNewMessage( time_s, time_ms );
  }

  /**
   * Check for any old message, and return the owning mailbox if there is
   * one.  Otherwise wait for a new message, and return its owning mailbox.
   * <P>
   * It skips all the mailboxs derived from Outbox.
   * @return    A reference to the mailbox with the oldest message.
   * @exception InterruptedException when the waiting thread gets an
   *            interrupt call.
   */

  Inbox waitForMessage( int time_s, int time_ms )
    throws InterruptedException
  {
    return portset.waitForMessage( time_s, time_ms );
  }

  /**
   * This does the actual destruction of a MailDaemonIn.
   */

  void destroy ( )
  {
    life.stop();
    life = null;
    _socket = null;
  }

} // end of MailDaemonIn class


