/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 *  $Id: MailDaemonOut.java,v 1.53 1997/05/29 07:07:11 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import java.net.*;
import java.io.*;
import java.util.*;

/**
 * This class is given a socket upon construction, and then it
 * proceeds to send given messages out that socket to the given
 * address when asked by the MailDaemon's send() method.
 * 
 * @author Luke Weisman
 * @author Wesley Tanaka
 * @author Adam Rifkin
 * @version 1.0b2 $Revision: 1.53 $ $Date: 1997/05/29 07:07:11 $
 * @see AckMessage
 * @see MailDaemon
 * @see Message
 * @see MessagePacket
 * @see OutMailQueue
 * @see OutTargetQueue
 * @see Place
 * @see SplitMessage
 * @see java.lang.InterruptedException
 * @see java.lang.NullPointerException
 * @see java.lang.Runnable
 * @see java.lang.System
 * @see java.lang.Thread
 * @see java.net.DatagramPacket
 * @see java.net.DatagramSocket
 * @see java.util.Hashtable
 */

class MailDaemonOut implements Runnable 
{

  /**
   * the time to sleep before message resending.
   */
  private static final int SLEEP_TIME = 5000;

  /**
   * set of all the outgoing destinations ever used.  These are stored
   * by machine/port/mailbox.  However, local sending mailbox names are
   * not part of the hash.
   */
  private Hashtable outPlaces;

  /**
   * the owner to pass messages to if needed.
   */
  private MailDaemon owner;

  /**
   * the socket used by the mailer
   */
  private DatagramSocket _socket = null;

  /** 
   * thread of life for the MailDaemonOut
   */
  private Thread life;


  // constructors

  /**
   * Make a new outdaemon.
   * @param own         The maildaemon boss which will give commands, etcx.
   * @param sendThrough The socket to dump packets out on.
   */

  MailDaemonOut( MailDaemon own, DatagramSocket sendThrough ) 
  {
    //assign the local socket to use.
    _socket = sendThrough;
    if ( _socket == null )
      throw new NullPointerException( "Null _socket in MailDaemonOut" );

    //make a new list of the outgoing queues.
    outPlaces = new Hashtable();
      
    //Start up thread for the mail deamon listener.
    Debug.lpln( 7, "dbg: MailDaemonIn starting thread" );
    life = new Thread (this, "MailDaemonOutThread" );
    life.setPriority( Thread.MAX_PRIORITY );
    life.setDaemon( true );

    owner = own;
  }

  void start( )
  {
    life.start();
  }


  /**
   * This function is called when adding a message to the queue.
   *
   * @return  the id of the next message to send out, or -1 if no
   *           message should be sent out.
   */

  private int QueueUpMessage( Message msg, Place targ, Mailbox from )
  {
    int msgID;
    int rval = -1;

    Debug.push( "MailDaemonOut.QueueUpMessage" );

    OutMailQueue q = (OutMailQueue)outPlaces.get( targ );
    if ( q == null )
      q = new OutMailQueue( );
      
    //adding message to queue.
    msgID = q.AddMessage( msg, from );
      
    //put modified (or new) q back in the outPlaces hashtable.
    outPlaces.put( targ, q );
      
    //only send message if all others are delt with so far.
    if ( q.PendingFrom( from ) == 1 )
      rval = msgID;
    else
      {
        Debug.lpln( 5, "Delaying send on message " + msgID );
      }

    Debug.pop( "MailDaemonOut.QueueUpMessage" );
    return rval;
  }


  /*
   * This sends an actual message without any bookkeeping or 
   * whatnot.<P>
   *
   * @param msg     The message to send.
   * @param outID   The ID of the message
   * @param targ    The destination of the message
   * @param fromP   The name of the mailbox which is sending message.
   * @param isAck   Whether the message is an ack or not.
   */

  private void doSend( Message msg, int outID, Place targ, 
                       long targ_mdID,
                       String fromP, boolean isAck )
  {
    Debug.push ("MailDaemonOut.doSend");
    try
      { 

        //SplitMessage is a vehicle which allows for the change
        //of all the listed data to a byte array, which the
        //DatagramSocket will accept and ship off.
        SplitMessage smsg = new SplitMessage( targ_mdID,
                                              owner.getID(),
                                              targ.mailbox(),
                                              fromP,
                                              isAck,
                                              outID,
                                              msg );

        byte [] omsg = smsg.toByteStream ();
        DatagramPacket pack = new DatagramPacket( omsg, omsg.length, 
                                                  targ.addr(), 
                                                  targ.port() );
          
        Debug.lpln (13, "Sending " + smsg);

        if (pack == null)
          throw new NullPointerException ("Null packet after build.");

        else if (omsg.length > 1024 * 60)
          {
            // FIX THIS so that I can send packets of size
            // greater than 64K by splitting them up. -- Adam

            Debug.bugExit ("MailDaemonOut doSend",
                           "Sending a Message that is too big.");
            throw new RuntimeException ("IOException");
          }

        else
          {
            Debug.lpln (7, "Sending the packet...");
            _socket.send (pack);
          }

        Debug.lpln( 7, "dbg: finished sending packet to " + targ );

      }

    catch (IOException ioException)
      {
        Debug.bugExit ("MailDaemonOut doSend",
                       "IOException in sending packet");
        ioException.printStackTrace();
        throw new RuntimeException("IOException");
      }

    Debug.pop ("MailDaemonOut.doSend");
  }
  

  /**
   * Deal with an expired message.  It calls the method of the
   * outbox which sent the message in the first place.
   *
   * @return true if the message should be sent anyway, false otherwise.
   */

  private boolean doExpire( MessagePacket mp, Mailbox from, Place to )
  {
    return from.timeOut( mp.msg, to );
  }


  /**
   * This process controls this thread.  It spends its time sleeping,
   * and occassionally wakes up, ships off messages if they have not
   * been acknowledged, and then goes back to sleep.
   */

  public void run( )
  {
    while ( true )
      {
        //Sleep
        try 
          {
            life.sleep( SLEEP_TIME );
          }
        catch ( InterruptedException ex )  
          { }

        runBatch();
      }
  } //end run


  /**
   * Go through all messages, sending oldest ones, calling timeouts,
   * etc.
   */

  synchronized void runBatch( )
  {
    Place to;
    MessagePacket mp;
    OutTargetQueue mq;
    OutMailQueue mq_col;
    Enumeration e_big, e;
    boolean emptyQueues;
    String from;
    long time;
     
    emptyQueues = true;
    time = (new Date()).getTime();
     
    //go through all the destinations which are being sent to by
    //anyone.
    for ( e_big = outPlaces.keys(); e_big.hasMoreElements();  )
      {
        to = (Place)e_big.nextElement();
        mq_col = (OutMailQueue)outPlaces.get( to );

        //send out one message from each sendbox sending to the 
        //specific place.
        for ( e = mq_col.keys(); e.hasMoreElements(); )
          {
            from = (String)e.nextElement();
            mq = (OutTargetQueue)mq_col.get( from );
             
            mp = mq.GetOldestMessage();
            if ( mp != null )
              {
                if ( !mp.expired( time ) 
                     || doExpire( mp, owner.get( from ), to ) )
                  //if not expired, or if send even if expired.
                  {
                    Debug.lpln( 1, "Sending stale to " +
                                to + " from " + from );

                    doSend( mp.msg, mq.GetOldestMessageID(), to,
                            mq_col.mdID(),
                            from, false );

                    emptyQueues = false;
                  }
              }
          }
      } 
     
    //if nothing was sent, then the queues are empty, so get
    //out of here as fast as possible.
    if ( emptyQueues )
      wakeTheMan();
     
  } //end runBatch


  /**
   * This will wake up the waitForPurge function(s), if necessary.
   */
  synchronized private void wakeTheMan( )
  {
    notifyAll();
  }


  /**
   * This will remove the table targeting a specific address and all the
   * associated messages.  It should be used when a specific target has
   * died or is discovered to have never existed.  It should be used with
   * extreme caution. <P>
   * Note that if a send mailbox attempts to send to the given place after
   * purging, a new table will be constructed, and the ID counters will be
   * reset to 0, which does not seem to pose a problem, but little to no
   * real investigation into the subject has been done.
   *
   * @param p The target place to no longer send messages to.
   */

  public synchronized Vector purgeAddress( Place p, Mailbox from )
  {
    Debug.lpln( 8, "Purging " + p );
    Vector v = null;

    Debug.lpln( 8, "The keys (all outgoing mailboxs): " );
    Enumeration k = outPlaces.keys();
    while ( k.hasMoreElements() )
      Debug.lpln( 8, "   k>" + k.nextElement() + "<" );
    
    OutMailQueue q = (OutMailQueue)outPlaces.get( p );
    
    //there is no queue for given place, so return now.
    if ( q == null )
      return new Vector();
    
    if ( from != null )
      {
        //just get the one mailbox's messages.
        v = q.purgeAddress( from );
        if ( q.isEmpty() )
          outPlaces.remove( p );
      }
    else
      {
        //get all the mailboxs' messages.
        v = new Vector();
        String name;
        OutTargetQueue qq;
        Enumeration ee, e = q.keys();
        while ( e.hasMoreElements() )
          {
            name = (String)e.nextElement();
            qq = (OutTargetQueue)q.get( name );
            ee = qq.elements();
            while( ee.hasMoreElements() )
              v.addElement( ee.nextElement() );
          }
        
        //kill everything
        outPlaces.remove( p );      
      }
    return v;
  }


  public synchronized void killMailbox( Mailbox from )
  {
    Enumeration e = outPlaces.keys();
    OutMailQueue q;
    Object k;
    while( e.hasMoreElements() )
      {
        k = e.nextElement();
        q = (OutMailQueue)outPlaces.get( k );
    
        //just get the one mailbox's messages.
        q.purgeAddress( from );
        if ( q.isEmpty() )
          outPlaces.remove( k );
      }
  }


  /**
   * Return whether there are any messages pending which were sent
   * by the from mailbox.
   *
   * Currently non-operational.
   * @return True if all messages have been acknowledged.  False otherwise.
   */

  public synchronized boolean isFlushed( Mailbox from )
  {
    boolean retval = true;

    for (Enumeration e = outPlaces.keys(); e.hasMoreElements() && retval;)
      {
        Hashtable target = (Hashtable) outPlaces.get( e.nextElement() );
        OutTargetQueue tq = (OutTargetQueue) target.get(from.name());
        if (tq != null)
          {
            if ( !tq.isEmpty() )
              retval = false;
          }
      }

    return retval;
  }


  /**
   * This send a message to the place 'targ'.  The from field will be
   * attached to the message as part of the return address.<P>
   * Once giving up the message through the send command, do not change
   * it any further.
   * @param msg The message to send
   * @param targ The destination for the message.
   * @from The name of the sending mailbox.
   */

  public synchronized void send( Message msg, Place targ, Mailbox from )
  {
    Debug.push ("MailDaemonOut.send");
    Debug.lpln( 1, "Sending message to " + targ + " from " + from );

    //Store the message in the outgoing mail queue so as to be
    //able to verify it later.  Also get the ID for the outgoing
    //message and check whether the message should be sent 
    //immediately.
    int outID = QueueUpMessage( msg, targ, from );

    if ( outID != -1 )
      {
        OutMailQueue q = (OutMailQueue)outPlaces.get( targ );
        doSend( msg, outID, targ, q.mdID(), from.name(), false );
      }
    Debug.pop ("MailDaemonOut.send");
  }

  
  /**
   * This is called by the MailDaemonIn segment.  It sends an ack 
   * message to the passed address.
   */

  public void SendAck( AckMessage am, Place targ, long md_id, String from )
  {
    doSend( am, am.data(), targ, md_id, from, true );
  }


  /**
   * When a message acknowledgement arrives, this routine gets called.
   * @param from Is the place which sent the ack (machine, port, and mailbox.)
   */

  synchronized void ReceiveAck( SplitMessage m, AckMessage am, Place from )
  {
    Debug.lpln( 13, "*** Ack: " + m + " ***" );

    OutTargetQueue q = null;

    //get general destination block.
    OutMailQueue q_hold = (OutMailQueue)outPlaces.get( from );

    if ( q_hold != null )
      {
        if ( q_hold.mdID() == MailDaemon.TARGET_ID_UNKNOWN )
          q_hold.setID( m.mydID() );
          
        if ( q_hold.mdID() != m.mydID() )
          Debug.lpln( 13, "Sending id does not match expected.");

        if ( am.status() == AckMessage.NEW_RECEIVER )
          {
            if ( q_hold.mdID() < m.mydID() )
              {
                Debug.lpln( 13, "Changing target receiver mdID." );
                q_hold.setID( m.mydID() );
                  
                //set all the expected acks to 0
                q_hold.resetAckNumbers( );

                // (and alert sendboxes
                //if necessary?)
              }
          }
        else  //a normal ack message - process it.
          {
            //get the specific sending portlet's queue.
            Debug.lpln( 5, "Got the general box: " + q_hold );
            q = (OutTargetQueue)q_hold.get( m.PName() );
            
            if ( q != null )
              if ( am.bad() )
                {
                  q.BadResponse( am, from );                }
              else //am is okay.
                {
                  q.AcknowledgeMessageID( m.ID() );
                  if ( q.GetOldestMessage() != null )
                    {
                      //send the oldest message in an attempt to get an ack 
                      //for it.
                      Debug.lpln( 13, "Sending unacknowledged message\n" );
                      MessagePacket mp = q.GetOldestMessage();
                      doSend( mp.msg, q.GetOldestMessageID(), from,
                              q_hold.mdID(), m.PName(), false );
                    }
                  Debug.lpln( 1, "All messages acknowledged\n" );
                }
          }
      }
    else
      {
        //received acknowledgement from unknown source.  Ignore it.
        Debug.lpln( 13, "Acknowledgement from unknown source: " +
                    from + " to " + m.PName() );
      }
  } //end ReceiveAck

  
  /**
   * Wait until all messages have been sent out. <P>
   * Currently this does not work. It just goes to sleep.
   */

  synchronized void waitUntilFlushed ()
    throws InterruptedException
  {
    wait();
  }


  /**
   * Some debugging stuff, which doesn't really work that well.
   */

  synchronized private void mailboxDump ()
  {
    System.out.println( "The keys: " );
    Enumeration k = outPlaces.keys();
        
    while ( k.hasMoreElements() )
      System.out.println( "   k>" + k.nextElement() + "<" );
  }


  /**
   * This does the actual destruction of a MailDaemonIn.
   */

  void destroy ( )
  {
    life.stop();
    life = null;
    _socket = null;
  }

} // end of MailDaemonOut class

