/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
   $Id: Mailbox.java,v 1.33 1997/05/29 07:07:13 kiniry Exp $
*/

package info.net;

import info.util.Debug;
import java.net.*;

/**
 * The Mailbox class implements address maintenence and name maintenence.  
 * All varieties of mailboxes are derived from this base class.<P>
 * The insertMessage method needs to be defined to make a mailbox
 * since it is the method that deals with new messages when they arrive.
 * Even sending mailboxes (sendBoxes) can get incoming mail telling
 * them things like their messages have been refused.
 *
 * @author Luke Weisman
 * @author Wesley Tanaka
 * @author Adam Rifkin
 * @version 1.0b2 $Date: 1997/05/29 07:07:13 $
 * @see Inbox
 * @see InetAddress
 * @see MailDaemon
 * @see Outbox
 * @see Place
 */

public abstract class Mailbox 
{

  /**
   * The name of the mailbox.
   */
  String _name;

  /**
   * The full address of the mailbox, including host and port and name.
   */
  Place _addr;

  /**
   * The mailDaemon the mailbox is associated with.
   */
  MailDaemon mailDaemon;

  /**
   * This means never time out.  Set the timeout to this.  Use the
   * setMinTimeout() method.
   * <!--* We need a better way to do this because you should
   * never force a developer to bind arbitrary values to semantic meaning
   * (like -1 to "no timeout"). *-->
   */
  public static final long NO_TIMEOUT = -1; 

  /**
   * A timeout will be generated if a message delivery takes longer than
   * timeout_time milliseconds.
   */
  private long timeout_time = NO_TIMEOUT;  //default no timeout.


  // constructors

  /**
   * Make a mailbox with the given name.
   * @param md The mailDaemon associated with the mailbox so the mailbox
   *           can generate the correct address.
   * @param    my_name The name desired for the mailbox.
   * @exception MailDaemonException if the passed name conflicts with the given
   *            name.
   */
  protected Mailbox( MailDaemon md, String my_name ) 
    throws MailDaemonException
  {
    _name = my_name;
    mailDaemon = md;
    InetAddress ina = md.getLocalAddr();
    _addr = new Place( ina, md.getLocalPort(), _name );

    try
      {
        mailDaemon.register( this );
      }
    catch ( MailDaemonException e ) { }
  }


  /**
   * Make a mailbox with a randomly generated name.  The name will be
   * made by calling the GenerateName method of the mailbox's mailDaemon.
   * @param md The mailDaemon associated with the mailbox so the mailbox
   *           can generate the correct address.
   * @see MailDaemon#GenerateName
   */

  protected Mailbox( MailDaemon md ) 
  {
    if ( md == null )
      throw new NullPointerException( "Mailbox constructor passed null "
                                      + " MailDaemon." );
    _name = md.GenerateName();
    _addr = new Place( md.getLocalAddr(), md.getLocalPort(), _name );
    mailDaemon = md;
    try {
      mailDaemon.register( this );
    } 
    catch (MailDaemonException mailDaemonException)
      {
        Debug.bugExit ("Mailbox constructor",
                       "Reserved system names have been clobbered");
        mailDaemonException.printStackTrace();
        throw new RuntimeException("MailDaemonException");
      }
  }


  // methods

  /**
   * @return The address of the mailbox.  This address can be used to 
   *         send the mailbox messages.  It is unique throughout the 
   *         internet while the program is running.
   */

  public Place addr( )
  {
    return _addr;
  }

  /**
   * @return The locally unique name of the mailbox.  (In point of
   *         fact, the name could be duplicated if there is an Outbox
   *         and a Inbox of the same name.)  <STRONG><EM> Is this
   *         still the case and, if so, why?  We shouldn't claim that
   *         a mailbox name is unique unless it is and, if it isn't,
   *         we should explain why it isn't.  Perhaps we should
   *         enforce some naming scheme on mailboxes in the future (to
   *         details and conform with types, interfaces, objects,
   *         etc. bound to a mailbox).</EM></STRONG>
   */

  public String name( )
  {
    return _name;
  }

  /**
   * Prints the name of the mailbox.
   * @return The name of the mailbox.
   */

  public String toString( )
  {
    return _name;
  }

  /**
   * Return the managing mailDaemon this mailbox.  This is the same
   * mailDaemon which was used during the mailbox's instantiation.
   */

  public MailDaemon getMailDaemon()
  {
    return mailDaemon;
  }

  /**
   * MessageDenial is called when a DenyMessage arrives, or a bad ack
   * arrives.
   * It should be overridden.  This method does nothing except print 
   * a message to the screen.
   *
   * @param m The denial message holding data about why message was denied.
   * @param from The failed address.
   */

  protected void MessageDenial( DenyMessage m, Place from )
  {
    System.err.println( name() + " not getting to " + from 
                        + " because of " + m ); 
  }

  
  /**
   * This can be used to accept DenyMessages or to do normal
   * Inbox stuff. <STRONG><EM> What does this mean?  Details please. 
   * </EM></STRONG>
   * @return true if message was accepted.  False if it was filtered
   *         out.
   */

  abstract protected boolean insertMessage(int time, Message m, Place from )
    throws MailDaemonException;


  /**
   * This is called when a sent message times out.  By default, the
   * message is returned and the outbox can deal with it.  Override
   * this method if you want to do anything interesting.  This method
   * just removes all messages to the given place.
   *
   * @return true if the message should be sent anyway.  False if not.
   */

  protected boolean timeOut( Message msg, Place to )
  {
    Debug.lpln( 3, "Timeout for " + msg + " to " + to );
    mailDaemon.purgeAddress( to, this );
    return false;
  }

  /**
   * Get the minimum time for a timeout on any message sent from the
   * box.  Timeout units are milliseconds.
   */

  public long getMinTimeout( )
  {
    return timeout_time;
  }

  /**
   * Set the timeout time.
   * @param to_time To this time in milliseconds.
   */

  public void setMinTimeout( long to_time )
  {
    timeout_time = to_time;
  }


  public void kill( )
  {
    mailDaemon.destroy( this );
    mailDaemon = null;
  }

  
  public void ressurect( MailDaemon md )
    throws MailDaemonException
  {
    mailDaemon = md;
    mailDaemon.register( this );
  }

} // end of Mailbox class
