/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: Message.java,v 1.36 1997/05/29 07:07:16 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import java.util.*;
import java.net.*;
import java.io.*;

/**
 * This is the abstract class that is the parent of all things sent
 * through the mailboxes.  To make your own message, inherit from this
 * class, or one of its children and implement the following:
 *
 *   writeData (DataOutputStream)
 *   readData (DataInputStream)
 *
 * In addition, the class needs a public default constructor that
 * takes no arguments.
 *
 * WARNING: Make sure these methods _and_ the constructor are public,
 * or you will end up with some horrid problems.
 *
 * Also make sure the class itself is public, or it will not be
 * instantiated on the receiving end.
 *
 * Some examples of derived messages are the StringMessage and
 * NumberMessage classes.
 *
 * Any message class can have a URL associated with it.  These URLs
 * are sent along with the messages to serve as a global form of
 * unique ID.  This ensures, if people are playing properly, that if a
 * message arrives at a location it is either recognized correctly, or
 * is known to be unknown.  In addition, if a message which is unknown
 * arrives, a remote class loader is invoked by the mail daemon, and the
 * class associated with the passed message and URL is loaded remotely, if
 * possible.  Before such drastic actions are taken, however, the url
 * is checked with the Message class's static URLStringSecurity object.  If
 * you don't set such an object, then one does not exist, so check out
 * the getURLChecker and setURLChecker methods.  Otherwise, any URL
 * will be trusted, and if necessary bytecode will be downloaded from
 * that site and run blindly by the java interpreter as a Message
 * object.
 *
 * @author Adam Rifkin
 * @author Luke Weisman
 * @version 1.0b2 $Revision: 1.36 $ $Date: 1997/05/29 07:07:16 $
 * @see NumberMessage
 * @see StringMessage
 * @see URLStringSecurity
 **/

public abstract class Message
{

  // instance variables

  private BoxSet bs;


  // no constructors


  // Abstract Methods

  /**
   * This takes the message and returns an array of bytes which store
   * all the relevant details of the message.  This array will then be
   * sent to some Inbox somewhere, and that will call the
   * readData method of this object to reconstruct the object.  There
   * is run-time typing so the correct type of object will be built
   * automatically, which means you don't have to worry about figuring out
   * which object is arriving, etc. <P>
   * WARNING: Currently the maximum size an object can occupy is around 61k
   *
   * @param dos is the stream to which we write our state.
   * @exception IOException is thrown if something goes wrong with the
   * write.
   **/

  abstract public void writeData( DataOutputStream dos )
    throws IOException;

  /**
   * This takes a byte array and builds an object from it.  If the method
   * is called, you know the byte array was made by the sister method
   * writeData from the same class.
   *
   * @param dis is the stream from which we read our state.
   * @exception IOException is thrown if something goes wrong with the
   * read.
   * @exception InvalidMessageException is thrown if the data that we
   * are reading doesn't agree with what we expect should be there.
   **/

  abstract public void readData( DataInputStream dis )
    throws IOException, InvalidMessageException;


  // methods

  /**
   * This turns a Message or child and packs it into the dos with its
   * class name etc. such that it can rebuilt from the readObject
   * method. <P>
   *
   * WARNING: Currently the maximum size message allowed is around 61k
   *
   * @param dos The stream to pack all the info and data in.
   */

  final public void writeObject( DataOutputStream dos )
    throws IOException
  {
    String name;
    String urlData;

    name = this.getClass().getName();

    dos.writeUTF(name);
    urlData = getExternalURL();
    dos.writeUTF(urlData);

    try
      {
        writeData(dos);
      }
    catch (Exception exception)
      {
        Debug.bugExit ("Message writeObject",
                       "Exception in writeData(dos) in " + this);
        exception.printStackTrace();
        throw new RuntimeException ("writeData Exception");
      }
  }

  /**
   * This takes the DataOutputStream from the writeObject method and turns
   * it into a byte array.
   *
   * If somewhere an IOException is thrown, the system will exit on
   * this method, when it catches it.  It is a very bad thing when
   * conversion from message to byte stream causes an error.
   */

  final protected byte [] toByteStream()
  {
    Debug.push ("Message.toByteStream()");
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    DataOutputStream dos = new DataOutputStream( bos );

    try
      {
        writeObject( dos );
        dos.flush();
        bos.flush();
      }
    catch (IOException ioException)
      {
        Debug.bugExit ("Message toByteStream", "IOException");
        ioException.printStackTrace();
        throw new RuntimeException ("IOException");
      }

    byte [] block = bos.toByteArray ();

    Debug.pop ("Message.toByteStream()");
    return block;
  }


  /**
   * Call this if you are using read and writeObject.  It sets
   * the message's parent, so all is happy.
   */

  public final void setParent( Message from )
  {
    bs = from.bs;
  }


  public final Message readObject( DataInputStream dis )
    throws MailDaemonException
  {
    //    if ( bs != null )
    // return bs.readObject(dis);
    // else
    // return BoxSet.halfReadObject( dis );

    return bs.readObject(dis);
  }


  /**
   * Set the BoxSet object which will build the message.
   */

  void setBoxSet ( BoxSet bs)
  {
    this.bs = bs;
  }


  /**
   * Return true if this object is the same object as the URL of the
   * passed string.
   */
  private boolean weAreA( String url )
  {
    try {
      URL t = getURL();

      if ( t == null )
        return true;
      else
        {
          URL t2 = new URL( url );

          if ( t.sameFile( t2 ) )
            return true;
          else
            return false;
        }
    }
    catch ( MalformedURLException e )
      {
        return false;
      }
  }


  /**
   * Get the URL associated with the given message type.  This one
   * returns the URL for the base Message class.  Derived classes should
   * return their own URLs. <P>
   * If the URL is not found, it returns null and does not raise any
   * errors.<P>
   * This function should be overloaded by children of the Message class.
   **/

  public URL getURL( )
  {
    return null;
  }


  /**
   * Return a String which has the URL information in it.  This is what
   * is shipped across the net with each message.
   **/

  public String getExternalURL( )
  {
    URL u = getURL();
    if ( u != null ) return u.toExternalForm();
    else return new String( "" );
  }

} // end of Message class
