/*
   class:       MessageQ

   des:         Used by Inbox to maintain queue of
                messages and times, etc.
*/

package info.net;

import java.util.Vector;

/**
 * This first in-first out queue is used by the Inbox to
 * store incoming messages which have not yet been pulled and delt with
 * by the user.<P>
 * It is synchronized so that checks to see if it is empty, or insertions
 * do not conflict with removals.  This class is n-thread safe, so enjoy.
 */

class MessageQ 
{

  /**
   * This is the list of all the messages in the queue, waiting
   * to be processed.
   */
  Vector incoming;


  MessageQ () {
    incoming = new Vector();
  }


  /**
   * @return true if the queue is empty, false otherwise.
   */
  public synchronized boolean isEmpty ( ) {
    return incoming.isEmpty();
  }

  /**
   * This will remove the first message from the queue.
   *
   * @return Returns the first message in the queue.
   */
  public synchronized MessageType GetAndKillFirst( )
  {
    MessageType mt = (MessageType)incoming.firstElement();
    incoming.removeElementAt(0);     // remove element at front of queue

    return mt;
  }

  /**
   * This returns a reference to the first message in the queue, but does
   * not remove it.  Be forwarned that in the process of getting it and
   * looking at it, it might get pulled from the queue by another thread,
   * if a lot of threads are using it.<P>
   * In this implementation of the package, the GetFirst is only used
   * to get the from() and time() info, so if only 1 thread is using the
   * mailbox on the receiving end, then there is no chance of this happening.
   *
   * @return The packet which was at the start of the queue.
   */
  public synchronized MessageType GetFirst( )
  {
    return (MessageType)incoming.firstElement(); 
  }

  /**
   * This adds a packet to the end of the queue.
   */
  synchronized void addElement( MessageType packet )
  {
    incoming.addElement( packet );
  }

} // end of MessageQ class
