/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: MultiURLClassLoader.java,v 1.15 1997/05/29 07:07:20 kiniry Exp $
 *
 * disclaimer: This class loader is based on the NetworkClassLoader
 * supplied as an example by the Sun Class documentation under
 * ClassLoader.
 */

package info.net;

import info.util.Debug;
import java.io.*;
import java.util.*;
import java.net.*;

/**
 * The MultiURLClassLoader allows classes to be loaded over the network.
 *
 * @author Adam Rifkin
 * @author Luke Weisman
 * @author Wesley Tanaka
 * @author Daniel M. Zimmerman
 * @version 1.0b2 $Date: 1997/05/29 07:07:20 $
 **/

public class MultiURLClassLoader extends ClassLoader
{

  /**
   * hash of all the urls
   */

  private Hashtable urlcache  =  new Hashtable();

  /**
   * hash of the classes
   */

  private Hashtable classcache = new Hashtable();

  /**
   * This is the security interface to use for checking out if urls are
   * okay or not.
   */

  private URLSecurity usi;

  /**
   * true if the security manager has been set.
   */

  private boolean usi_set = false;

  private String mostRecentURL = null;

  // constructors

  /**
   * Construct a new MultiURLClassLoader, using the provided URLs and
   * the provided URLSecurity object.
   * @param security The URLSecurity object to use, or null for no
   * security checking.
   * @param urls A space delimited set of URLs, or null for no initial
   * URLs.
   */

  public MultiURLClassLoader( String urls, URLSecurity security )
  {
    super();
    if ( security == null )
      usi = new URLOpenSecurity();
    else
      {
        usi = security;
        usi_set = true;
      }

    if ( urls != null )
      {
        StringTokenizer st = new StringTokenizer( urls, " " );
        String u;
        while( st.hasMoreTokens() )
          {
            u = st.nextToken();
            addURL( u );
          }
      }
  }


  // methods

  /**
   * Add a URL to the MultiURLClassLoader's classpath, if necessary.
   * This URL will also be the first to be checked on the next classload.
   * @param name The URL to add, in string form.
   */

  public void addURL( String name )
  {
    // Only add the URL to the cache if necessary.
    if ( urlcache.get( name ) == null )
      {
        Debug.lpln (9, "Adding URL " + name + " to urlcache.");
        urlcache.put( name, new URLClassFileLoader( name ) );
      }

    mostRecentURL = name;
  }


  /**
   * Return the current url security checker.
   * @return The URLSecurity object currently being used.
   */

  public final URLSecurity getURLChecker()
  {
    if ( usi_set )
      return usi;
    else
      return null;
  }


  /**
   * Set the current url security checker.  This will silently fail
   * if there is already a URLSecurity object being used.
   * @param new_usi The new URLSecurity object.
   */

  public final void setURLChecker( URLSecurity new_usi )
  {
    if ( !usi_set )
      {
        usi_set = true;
        usi = new_usi;
      }
  }


  /**
   * Load a class, locally or remotely.
   * @param name The name of the class to load.
   * @param resolve true if this class should be completely resolved
   * (ie: all parent classes loaded, etc.), false otherwise.
   * @exception ClassNotFoundException if no class of the given name
   * can be found by the MultiURLClassLoader.
   */

  public synchronized Class loadClass( String name, boolean resolve )
    throws ClassNotFoundException
  {
    Debug.lpln(12, "MultiURLClassLoader ( " + name + ", " + resolve + " ).");
    Class c = null;

    //first try the cache.
    c = (Class) classcache.get(name);

    if (c != null)
      {
        Debug.lpln( 9, "Detected " + name + " in class cache.");
        if (resolve)
          resolveClass(c);
        return c;
      }
    else
      {
        Debug.lpln(9, "Searching for " + name + " locally (as a system class)...");
        try
          {
            c = findSystemClass(name);
          }
        catch (ClassNotFoundException e)
          {
            Debug.lpln(9, "ClassNotFoundException raised on local search.");
            c = null;
          }
        catch (NoClassDefFoundError e)
          {
            Debug.lpln(9, "NoClassDefFoundError raised on local search.");
            c = null;
          }
      }

    if ((c == null) && (mostRecentURL != null))
      {
        Debug.lpln( 9, "Searching for " + name + " using most recent URL...");
        URLClassFileLoader ucl;

        //find class.
        try
          {
            ucl = (URLClassFileLoader)urlcache.get (mostRecentURL);
            byte data [] = ucl.loadClassData (name, usi);
            Debug.lpln(9, "Defining class...");
            c = defineClass( data, 0 , data.length );
          }
        catch ( ClassNotFoundException ex )
          {
            c = null;
          }
        catch ( ClassFormatError ex )
          {
            Debug.lpln( 9, "Error(MultiURLClassLoader.loadClass) "+ex );
            c = null;
          }
        catch ( SecurityException ex )
          {
            Debug.lpln( 9, "Error(MultiURLClassLoader.loadClass) "+ex );
            c = null;
          }
      }

    if (c == null)
      {
        Debug.lpln( 9, "Searching for " + name + " using cached URLs...");
        Enumeration e = urlcache.keys();
        URLClassFileLoader ucl;

        //find class.
        c = null;
        while( (c == null) && e.hasMoreElements() )
          {
            try
              {
                ucl = (URLClassFileLoader)urlcache.get (e.nextElement());
                byte data [] = ucl.loadClassData (name, usi);
                Debug.lpln(9, "Defining class...");
                c = defineClass( data, 0 , data.length );
              }
            catch ( ClassNotFoundException ex )
              {
                c = null;
              }
            catch ( ClassFormatError ex )
              {
                Debug.lpln( 9, "Error(MultiURLClassLoader.loadClass) "+ex );
                c = null;
              }
            catch ( SecurityException ex )
              {
                Debug.lpln( 9, "Error(MultiURLClassLoader.loadClass) "+ex );
                c = null;
              }
          }
      }

    if ( c != null )
      {
        Debug.lpln(9, "Class defined, adding to class cache...");
        classcache.put( name, c );
        if (resolve)
          resolveClass(c);
        return c;
      }
    else
      throw new ClassNotFoundException( "No class " + name
                                        + " found in MultiURL_CL" );
  }

} // end of MultiURLClassLoader class


