/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: Multicastbox.java,v 1.4 1997/04/08 22:16:40 kiniry Exp $
 */

package info.net;

import info.util.Set;
import java.util.Enumeration;

/**
 * Multicastbox is a single Mailbox that maintains a set of
 * destination Places.  The user can add or remove Places, and
 * send messages to all of the Places at once. <P>
 *
 * Multicastbox ensures that no Place is duplicated in the set.
 * Repeated adds will silently fail.  On a send command, each place
 * in the set receives exactly one copy of the message. <P>
 *
 * The difference between a Broadcastbox and a Multicastbox is
 * the protocol used for Message transfer: unicast for the former,
 * multicast for the latter. <P>
 *
 * testsend7 illustrates how to use the Multicastbox.
 *
 * @author Adam Rifkin
 * @author Eve Schooler
 * @version 1.0b2 $Date: 1997/04/08 22:16:40 $
 * @see Broadcastbox
 * @see Inbox
 * @see MailDaemon
 * @see MailDaemonException
 * @see Message
 * @see Outbox
 * @see Place
 * @see info.util.Set
 * @see java.util.Enumeration
 **/

public class Multicastbox extends Outbox
{
  /*
   * instance variable
   *
   * multicastTargetSet  - the set of all target Places to send a copy
   *                       of any Message we ship from this Mailbox.
   */

  private Set multicastTargetSet;


  // constructors
  
  /**
   * Constructs a new Multicastbox with the given name.
   *
   * @param mailDaemon is the MailDaemon this Mailbox will use
   *                   to send out messages.
   * @param mailboxName is the name of this Mailbox.
   * @exception MailDaemonException is thrown if mailboxName is
   *                   already used by the MailDaemon.
   **/

  public Multicastbox (MailDaemon mailDaemon, String mailboxName)
    throws MailDaemonException 
  {
    super (mailDaemon, mailboxName);
    multicastTargetSet = new Set ();
  }


  /**
   * Constructs a new Multicastbox with a computer-generated name.
   *
   * @param mailDaemon is the MailDaemon this Mailbox will use
   *                   to send out messages.
   **/

  public Multicastbox (MailDaemon mailDaemon) 
  {
    super (mailDaemon);
    multicastTargetSet = new Set ();
  }


  // methods
  
  /**
   * Adds this Place to the multicast set of receiving Mailboxes.
   * The passed Place should be a valid address to some Inbox.
   *
   * @param newTargetPlace is the new target Place to add to the set. 
   * @return true if the Place is new to the set, false otherwise.
   **/

  public boolean add (Place newTargetPlace)
  {
    return multicastTargetSet.add (newTargetPlace);
  }


  /**
   * Removes the named target from the multicast set of receive Mailboxes.
   *
   * @param targetPlace is the Place to remove from the set.
   * @return true if the Place was in the set, false otherwise.
   **/

  public boolean remove (Place targetPlace)
  {
    return multicastTargetSet.remove (targetPlace);
  }


  /**
   * Removes all target Places from the set of receive Mailboxes.
   **/

  public void removeAll ()
  {
    multicastTargetSet = new Set ();
  }


  /**
   * Checks if the named Place is in the multicast set.
   *
   * @param targetPlace is the target Place to check in the current set.
   * @return true if the target was in the set, false otherwise.
   **/

  public boolean has (Place targetPlace)
  {
    return multicastTargetSet.has (targetPlace);
  }


  /**
   * @return the Enumeration of all Places to which this Mailbox is bound.
   **/

  public Enumeration multicastSet ()
  {
    return multicastTargetSet.set ();
  }


  /**
   * Sends the given Message to all Mailboxes in the current multicast set.
   *
   * @param message is the message to multicast.
   **/  

  public void send (Message message)
  {
    Enumeration enumeration = multicastTargetSet.set ();

    while (enumeration.hasMoreElements ())
      doSend (message, (Place)enumeration.nextElement ());
  }

} // end of Multicastbox class

