/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 *   $Id: OutTargetQueue.java,v 1.12 1997/04/27 23:39:46 dmz Exp $
 */


package info.net;

import info.util.Debug;
import java.net.*;
import java.io.*;
import java.util.*;

/**
 * The OutTargetQueue is a vehicle for maintaining unacknowledged
 * messages to a given target Mailbox.  The queue has methods for
 * adding messages, and erasing messages when they have been properly
 * acknowledged by the remote host.
 */

class OutTargetQueue extends Vector 
{

  /**
   * This is the ID of the newest message which was ever in the queue.
   * Using this value, it is possible to compute the ids of all the
   * messages in the qeueue since they are all stored sequentially.
   */
  int outID;            

  /**
   * The owning Mailbox
   */
  Mailbox owner = null;


  // constructors

  /**
   * Make a new queue.  The outID is set to -1 (the newest message
   * is non-existant.)
   */
  OutTargetQueue( Mailbox from )
  {
    super( 1 );
    outID = -1;
    owner = from;
  }


  // methods

  /**
   * Reset the ack number so it is as if no message has ever been ack'ed
   */
  void resetAckNumber( )
  {
    outID = size() - 1;
  }

  /**
   *  Add a message to the queue and get its ID value.<P>
   *
   * NOTE:
   *    Sometime we should deal with wrapping around and range
   *    checking and all of that.  So be forwarned that this 
   *    stuff doesn't happen yet.
   */
  public synchronized int AddMessage( Message msg )
  {
    if ( msg != null )
      {
        MessagePacket blah = new MessagePacket( msg, owner );

        //New messages go at the beginning of the vector.
        if ( !isEmpty() )
          insertElementAt( blah, 0 );
        else
          addElement( blah );            

        outID++;
        return outID;
      }
    else
      throw new NullPointerException( "AddMessage passed null message." );
  }


  /**
   * This function takes an ID and deletes all messages with older or
   * equal IDs from the queue.  (Older being a smaller ID value)
   */

  public synchronized void AcknowledgeMessageID( int ID )
  {
    int mcnt, i;

    //attempt to salvage the strangeness due to processes dying and
    //being reborn.
    if ( outID < ID )
      System.err.println( "Last ID sent is less than last ID" 
                          + " acknowledged."
                          + " (try not reusing Places over "
                          + "application instantiations.)" );

    else if (!isEmpty())
      //delete all messages from the acknowledged message to the oldest
      //message.
      for ( mcnt = size() - outID + ID; mcnt > 0; mcnt-- )
        removeElementAt( indexOf( lastElement() ) );

    if ( isEmpty() )
      Debug.lpln( 5, "Wiped out all messages." );
    else
      Debug.lpln( 5, "There are " + size() + " messages left." );
  }


  /**
   * @return the oldest message in the queue.
   */

  public synchronized MessagePacket GetOldestMessage( )
  {
    if (!isEmpty())
      return (MessagePacket)lastElement();
    else
      return null;
  }


  /**
   * @return the id of the oldest message in the queue.
   */

  public synchronized int GetOldestMessageID( )
  {
    if ( size() != 0 )
      return outID - size() + 1;
    else
      return -1;
  }


  public synchronized void KillOldestMessage( )
  {
    AcknowledgeMessageID( GetOldestMessageID() );
  }


  /**
   * When an ack comes in saying the sent message was not getting
   * through, this is called.
   */

  public synchronized void BadResponse( AckMessage am, Place from )
  {
    DenyMessage dm = new DenyMessage( am.status() );
    owner.MessageDenial( dm, from );
  }

} // end of OutTargetQueue class
