/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 *  $Id: Place.java,v 1.37 1997/05/29 07:07:24 kiniry Exp $
 */

package info.net;

import java.net.*;
import java.util.*;
import java.io.*;

/**
 * The Place object serves as a unique name for mailboxes of all
 * sorts.  The main use of it is to be able to send messages to
 * specific Inboxes.  A given mailbox has a unique Place.<P>
 *
 * A Place has three components: a machine address (i.e. an
 * InetAddress object like "frankie.cs.caltech.edu"), a port number,
 * and a mailbox name (a string).  There are a whole host of
 * constructors for a Place given the variety of contexts in which it
 * will be used.
 *
 * @author Adam Rifkin
 * @author Wesley Tanaka
 * @author Luke Weisman
 * @author Daniel M. Zimmerman
 * @version 1.0b2 $Date: 1997/05/29 07:07:24 $
 * @see InetAddress
 * @see Message
 * @see InvalidMessageException
 * @see java.io.IOException
 * @see java.io.DataInputStream
 * @see java.io.DataOutputStream
 * @see java.net.UnknownHostException
 */

public class Place extends Message implements Cloneable
{

  /**
   * These are the three fields of the Place.
   */
  private InetAddress _addr;
  private int _port;
  private String _mailbox;

  /**
   * If this flag is true, the Place is an invalid address.  The reason
   * for this can be obtained via the whyBad() method.  Whether this
   * flag is true can be obtained via the bad() method.
   */
  private boolean bad = false;
  private String whyBad = null;


  // constructors

  /**
   * @param host The host machine name which holds the MailDaemon owning the
   *             mailbox which is associated with this address.
   * @param nport The port number of the MailDaemon of the mailbox.
   * @param mailbox The actual name of the mailbox.
   */

  public Place(String host, int nport, String mailbox)
  {
    if ( mailbox == null )
      registerError( "Null mailbox in Place( " + host + ", " + nport
                     + ", " + mailbox + ")" );

    try {
      _addr = InetAddress.getByName( host );
    }
    catch (java.net.UnknownHostException e)
      {
        registerError( e.toString() + " in Place( " + host + ", " + nport
                       + ", " + mailbox + " )" );
      }

    _port = nport;
    _mailbox = new String( mailbox );
  }


  /**
   *
   **/

  public Place (Place place, String mailbox)
  {
    _addr = place._addr;
    _port = place._port;
    _mailbox = mailbox;
  }


  /**
   * @param host The host machine address holding the MailDaemon owning the
   *             mailbox which is associated with this address.
   * @param nport The port number of the MailDaemon of the mailbox.
   * @param mailbox The actual name of the mailbox.
   */

  public Place( InetAddress host, int nport, String mailbox)
  {
    if (( mailbox == null ) || (host == null))
      registerError( "Null mailbox or host in Place( " + host + ", " + nport
                     + ", " + mailbox + " )" );

    _addr = host;
    _port = nport;
    _mailbox = new String( mailbox );
  }


  /**
   * This makes a place out of a string of the form: <P>
   * <pre>[host name] [port number] [mailbox name]</pre> <P>
   * it also accepts
   * <pre>[host name]/[ip string] [port number] [mailbox name]</pre> <p>
   * It is usefull for passing command line arguments with Djinn.
   *
   * @param s The string with the address information in it.
   */

  public Place( String s )
  {
    StringTokenizer st = new StringTokenizer( s, " " );

    String t = null;
    int ind = -1;
    try {
      t = st.nextToken();
      ind = t.indexOf( "/" );
      if ( ind != -1 )
        _addr = InetAddress.getByName(t.substring( ind + 1 ) );
      else
        _addr = InetAddress.getByName( t );
    }
    catch (java.net.UnknownHostException e)
      {
        //attempt to use the first part.
        if ( ind != -1 )
          {
            try
              {
                _addr = InetAddress.getByName( t.substring( 0, ind ) );
              }
            catch( java.net.UnknownHostException e2 )
              {
                registerError( e2.toString() + " in Place( " + s + " )" );
              }
          }
        else
          registerError( e.toString() + " in Place( " + s + " )" );
      }

    _port = Integer.parseInt( st.nextToken() );
    _mailbox = st.nextToken();

    if ( _mailbox == null )
      registerError( "Null mailbox passed to Place in Place( " + s + ")" );
  }


  /**
   * This makes a place out of three Strings, the first being the host,
   * the second being the port, and the third being the mailbox.
   * @param hstr The host string
   * @param pstr The port number as a string
   * @param plstr The mailbox name.
   */

  public Place( String hstr, String pstr, String plstr )
  {
    try {
      _addr = InetAddress.getByName( hstr);
    }
    catch (java.net.UnknownHostException e)
      {
        registerError( e.toString(), hstr, pstr, plstr );
      }

    _port = Integer.parseInt( pstr );
    _mailbox = new String( plstr );

    if ( _mailbox == null )
      registerError( "Mailbox passed as null.", hstr, pstr, plstr );
  }


  /**
   * This is a default constructor so the Place class can be a child
   * of the Message class.  Other than that, it should not be used.  <P>
   */

  public Place( )
  {
    _addr = null;
    _port = 0;
    _mailbox = null;
  }


  /**
   * Return the Place which is associated with the passed Mailbox.
   */

  public Place( Mailbox mb )
  {
    _addr = mb.addr().addr();
    _port = mb.addr().port();
    _mailbox = mb.addr().mailbox();
  }


  // methods

  public String host() { return _addr.getHostName(); }
  public int port() { return _port;}
  public String mailbox() { return _mailbox; }
  public InetAddress addr() { return _addr; }


  /**
   * Turn the place into a string of the form:
   * <pre>
   *     "[host name]/[host ip number] [port] [mailbox name]"
   * </pre>
   * @return String form of the place.
   */

  public String toString( )
  {
    return _addr + " " + port() + " " + _mailbox;
  }


  /**
   * Generates a hash code based on the mailbox name only.
   */

  public int hashCode( )
  {
    return mailbox().hashCode();
  }


  /**
   * Compares the machine address, the port number, and the mailbox name
   * to determine equality.
   * @return true if the two objects are equal, false otherwise.
   */

  public boolean equals( Object obj )
  {
    if ( obj instanceof Place )
      {
        Place p = (Place)obj;
        if ( ((p.mailbox()).equals(mailbox()))
             && (p.port() == port())
             && (p.addr().equals(addr())) )
          return true;
      }
    return false;
  }

  /**
   * Compares the machine address and port number, ignoring the mailbox name.
   * @return true if the machine address and port number are the same in the
   * two objects, false otherwise.
   */
  public boolean weakEquals(Object obj)
  {
    if (obj instanceof Place)
      {
        Place p = (Place) obj;
        if ((p.port() == port()) &&
            (p.addr().equals(addr())))
          return true;
      }
    return false;
  }

  public boolean bad()
  {
    return bad;
  }


  public String whyBad( )
  {
    return whyBad;
  }


  private void registerError( String err_msg )
  {
    bad = true;
    whyBad = err_msg;
    throw new RuntimeException( whyBad );
  }


  public void registerError( String err_msg, String host,
                             String port, String mailbox )
  {
    registerError( err_msg + " in Place( " + host + ", " + port
                   + ", " + mailbox + " )" );
  }


  public Object clone ()
  {
    return new Place (_addr, _port, _mailbox);
  }


  /**
   * Turn the place into a stream for sending purposes.
   * @exception IOException for the normal reasons.
   */

  public void writeData( DataOutputStream dos  )
    throws IOException
  {
    if (_addr != null)
      {
        dos.writeBoolean(true);
        dos.writeUTF(_addr.getHostName());
      }
    else dos.writeBoolean(false);

    dos.writeInt( port() );

    if (_mailbox != null)
      {
        dos.writeBoolean(true);
        dos.writeUTF(_mailbox);
      }
    else dos.writeBoolean(false);
  }


  /**
   * Turn a stream into a place.
   * @exception InvalidMessageException If the data is corrupt or the
   *            host string can not be turned into an InetAddress
   * @exception IOException for the normal reasons.
   */

  public void readData( DataInputStream dis )
    throws InvalidMessageException, IOException
  {
    try
      {
        if (dis.readBoolean())
          _addr = InetAddress.getByName(dis.readUTF());
        else
          _addr = null;

        _port = dis.readInt();

        if (dis.readBoolean())
          _mailbox = dis.readUTF();
        else
          _mailbox = null;
      }
    catch (java.net.UnknownHostException e)
      {
        throw new InvalidMessageException( "MailDaemon.Place" );
      }
  }

} // end of Place class





