/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: ReceiveSet.java,v 1.23 1997/05/29 07:07:26 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import info.util.Set;
import java.util.Enumeration;

/**
 * ReceiveSet allows the collection of multiple Inboxes in a set.
 * This set can wait for a new message or get the oldest message. <P>
 *
 * Mailboxes can be added and removed from this set easily, allowing 
 * for a dynamic collection of receive Mailboxes. <P>
 *
 * A Receivebox is not meant to be synchronized.
 * This would be nice to have. <P>
 *
 * testsend8.java and testreceive8.java illustrate how to use ReceiveSet.
 *
 * @author Adam Rifkin
 * @author Luke Weisman
 * @version 1.0b2 $Date: 1997/05/29 07:07:26 $
 * @see Inbox
 * @see MailDaemon
 * @see MailDaemonException
 * @see Message
 * @see Outbox
 * @see info.util.Set
 * @see java.util.Enumeration
 **/

public class ReceiveSet
{
  // instance variables

  Set setOfInboxes;
  MailDaemon myMailDaemon;


  // constructor

  /**
   * Constructs a new ReceiveSet.
   *
   * @param mailDaemon is the MailDaemon for getting new messages.
   **/

  public ReceiveSet (MailDaemon mailDaemon)
  {
    setOfInboxes = new Set ();
    myMailDaemon = mailDaemon;
  }


  // methods

  /**
   * Adds an Inbox to the set.
   *
   * @param newInbox is the Inbox to add to the set. 
   * @return true if newInbox was not already in the set, false otherwise.
   **/

  public boolean add (Inbox newInbox)
  {
    return setOfInboxes.add (newInbox);
  }


  /**
   * Removes the given Inbox from the set.
   *
   * @param oldInbox is the Inbox to remove from the set.
   * @return true if oldInbox was in the set, false otherwise.
   **/

  public boolean remove (Inbox oldInbox)
  {
    return setOfInboxes.remove (oldInbox);
  }


  /**
   * Checks for the given Inbox in the set.
   *
   * @param checkInbox is the Inbox to check for in the current set.
   * @return true if checkInbox was in the set, false otherwise.
   **/

  public boolean has (Inbox checkInbox)
  {
    return setOfInboxes.has (checkInbox);
  }


  /**
   * @return the Enumeration of all the receive mailboxs in set.
   **/

  public Enumeration set ()
  {
    return setOfInboxes.set ();
  }


  /**
   * Wait for the set to have a message in one of the members, and then
   * return a reference to that member.
   * First checks to see if any of the mailboxes in the set have a message.
   * If there is one, it returns the Inbox that contains the 
   * oldest message.  Otherwise it waits until one of the mailboxes has
   * a message, and then returns a reference to that mailbox. <P>
   *
   * This function will block until there is a message in one of the
   * receive mailboxes in the set.  If there is no mailbox in the set, then
   * this function blocks forever.
   *
   * @return    The receivebox in the set with the oldest message.
   * @exception InterruptedException If the calling thread gets an
   *            interrupt message, this exception is thrown.
   **/

  public Inbox waitForMessage ()
    throws InterruptedException
  {
    return waitForMessage (0);
  }


  /**
   * Wait for new message on a box, but for only a specific time.
   *
   * @param timeInMilliseconds time in milliseconds.
   * @exception InterruptedException If the calling thread gets an
   *            interrupt message, this exception is thrown.
   */

  public Inbox waitForMessage (long timeInMilliseconds)
    throws InterruptedException
  {
    Enumeration receiveSet;
    Inbox thisReceivebox = null;
    Inbox oldestReceivebox = null;
    long startTime = System.currentTimeMillis();
    long remainingTime;

    Debug.lpln (7, "Entering the waitForMessage loop of the ReceiveSet.");

    // Check if any Mailbox has a message pending.

    receiveSet = setOfInboxes.set ();

    // This is dangerous because the enumeration can change in
    // another thread.  We should change this.  -- Adam

    while (receiveSet.hasMoreElements ())
      {
        thisReceivebox = (Inbox)receiveSet.nextElement ();

        if (!thisReceivebox.empty ())
          {
            if ((oldestReceivebox == null) ||
                (oldestReceivebox.time () > thisReceivebox.time ()))
              {
                // Set oldestReceivebox to the Mailbox with oldest message.
                oldestReceivebox = thisReceivebox;
              }
          }
      }     

    remainingTime = timeInMilliseconds;

    Debug.lpln (7, "oldestReceivebox = " + oldestReceivebox);

    while ((oldestReceivebox == null) &&
           ((timeInMilliseconds == 0) || (remainingTime > 0)))
      {
        Debug.lpln (7, "ReceiveSet wait remainingTime = " + remainingTime);

        // No message, so wait for new message.

        thisReceivebox = myMailDaemon.waitForNewMessage
          (timeInMilliseconds == 0 ? 0 : remainingTime);

        if ((thisReceivebox != null) && 
            (!thisReceivebox.empty ()) && 
            has (thisReceivebox))
          oldestReceivebox = thisReceivebox;
        else
          remainingTime = timeInMilliseconds -
            (System.currentTimeMillis () - startTime);
      }

    Debug.lpln (7, "before return oldestReceivebox = " + oldestReceivebox);

    return oldestReceivebox;
  }
 

  /**
   * This is the same as waitForMessage except it does not throw any
   * exceptions.
   *
   * @see ReceiveSet#waitForMessage()
   **/

  public Inbox silentWaitForMessage( )
  {
    Inbox awaitedInbox = null;

    while (awaitedInbox == null)
      {
        try
          {
            awaitedInbox = waitForMessage ();
          }
        catch (InterruptedException exception)
          {
            // Enjoy the silence.
          }
      }
    return awaitedInbox;
  }

} // end of ReceiveSet class
