/*
 * The Infospheres Infrastructure
 * Copyright (c) 1996,1997 California Institute of Technology.
 * All Rights Reserved.
 */

/*
 * $Id: Sendbox.java,v 1.26 1997/05/29 07:07:28 kiniry Exp $
 */

package info.net;

import info.util.Debug;
import java.util.*;

/**
 * A sendbox class allows for the connection to one inbox at a time.  Once
 * connected to an receivebox using the bind method, a sendbox can
 * push messages to that receivebox.  At any point a program can
 * dynamically re-bind to another receivebox using the bind method,
 * so any messages sent will be sent to the newly bound inbox.  Bindings
 * are done by handing the bind method a Place object that has
 * the address of the desired Inbox.
 *
 * @author Adam Rifkin
 * @author Wesley Tanaka
 * @author Luke Weisman
 * @version 1.0b2 $Date: 1997/05/29 07:07:28 $
 * @see Inbox
 * @see Place
 * @see Outbox
 * @see MailDaemon
 * @see MailDaemonException
 * @see Message
 */

public class Sendbox extends Outbox
{
  /**
   * This is the current target location to send messages to.
   */
  private Place dest;


  // constructors

  /**
    * Make a Sendbox which is pre-bound to a given place.
    * @param md The maildaemon which the send mailbox is to use to send 
    *           messages
    * @param pname The name of the send mailbox.
    * @param destination The place to bind to.
    * @exception MailDaemonException if the name is already used by
    * the maildaemon.  
    */

  public Sendbox (MailDaemon md, String pname, Place destination) 
    throws MailDaemonException
  {
    super( md, pname );
    dest = destination;
  }


  /**
    * Make a Sendbox which is unbound, and which has a randomly
    * generated unique name.
    * @param md The MailDaemon the Sendbox will use to send messages.
    */

  public Sendbox (MailDaemon md) 
  {
    super( md );
    dest = null;
  }


  /**
    * Make a Sendbox with a specific name.
    * @param md The maildaemon which the send mailbox is to use to send 
    *           messages
    * @param pname The name of the send mailbox.
    * @exception MailDaemonException if the name is already used by the 
    * maildaemon.
    */

  public Sendbox (MailDaemon md, String pname)
    throws MailDaemonException
  {
    super( md, pname );
    dest = null;
  }
  

  /**
    * Make a Sendbox which is pre-bound to a given place.
    * @param md The maildaemon which the send mailbox is to use to send 
    *           messages
    * @param destination The place to bind to.
    */

  public Sendbox( MailDaemon md, Place destination )
  {
    super( md );
    dest = destination;
  }


  /**
    * Bind the Sendbox to a specified destination Place.  After this
    * is called, all messages sent through the send mailbox will be sent
    * to the given address.
    * @param destination The place to bind to.
    */

  public void bind(Place destination)
  {
    dest = destination;
  }


  /**
    * Get the current destination of all messages.
    * @return The current destination of all messages sent through this
    *         mailbox.
    */

  public Place destination()
  {
    return dest;
  }


  /**
   * Send a message to the currently bound destination.  If there is no
   * currently bound destination, the method will throw a runtime 
   * exception.
   *
   * <P>Eventually, this method will be implemented to throw a more
   * explicit and specific exception.
   */

  public void send(Message msg)
  {
    Debug.push ("Sendbox: send");
    if ( dest != null )
      doSend( msg, dest );
    else
      throw new RuntimeException
        ( "Sending a message using an unbound send mailbox." );
    Debug.pop ("Sendbox: send");
  }

} // end of Sendbox class

